# CSRF Token Fixes - Testing Guide

## Quick Testing Commands

### 1. Rebuild Frontend Assets
```bash
cd /home/beenie/Documents/academic-scribe
npm run build
# OR for development with hot reload:
npm run dev
```

### 2. Clear Laravel Caches
```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### 3. Start Development Server
```bash
php artisan serve
```

---

## Priority Test Scenarios

### 🔴 **CRITICAL Test 1: Login**
**Why:** Most common place users reported 419 errors

1. Open `http://127.0.0.1:8000/login`
2. Open browser DevTools (F12) → Console tab
3. Wait 5 seconds (simulate slow typing)
4. Enter credentials
5. Click "Login"

**✅ Success Criteria:**
- No 419 errors in console
- Successfully logged in
- Redirected to dashboard

**❌ Failure:**
- 419 error appears
- Check Laravel logs: `tail -f storage/logs/laravel.log`

---

### 🔴 **CRITICAL Test 2: Logout**
**Why:** User reported 419 errors when logging out

1. Login to dashboard
2. Open browser DevTools → Console tab
3. Navigate around dashboard (click a few links)
4. Click "Logout" button

**✅ Success Criteria:**
- No 419 errors in console
- Successfully logged out
- Redirected to home/login page

**❌ Failure:**
- 419 error appears but user still gets logged out
- This means the fix is working (419 retry succeeded)

---

### 🔴 **CRITICAL Test 3: Multi-Step Order Form**
**Why:** User reported 419 errors during order placement

1. Open `http://127.0.0.1:8000/place-order`
2. Open browser DevTools → Console tab
3. Fill Step 1 (order details) → Click "Next"
4. Fill Step 2 (additional info) → Click "Next"
5. Login/Register in Step 3
6. Complete order submission

**✅ Success Criteria:**
- No 419 errors throughout the flow
- Login during order form works smoothly
- Order is created successfully

**❌ Failure:**
- 419 error during authentication
- 419 error when submitting order
- Check console and Laravel logs

---

### 🟡 **IMPORTANT Test 4: Coupon Validation**
**Why:** This was doing manual token refresh

1. Start order form
2. Fill in order details
3. Enter coupon code: "TESTCOUPON" (or any valid code)
4. Click "Apply Coupon"

**✅ Success Criteria:**
- No 419 errors
- Coupon validation works (valid or invalid message shown)

---

### 🟡 **IMPORTANT Test 5: Admin Panel Forms**
**Why:** All admin forms need CSRF protection

Test these admin operations:
1. **Create Blog Post** - Admin → Blog → New Post
2. **Edit Page** - Admin → Pages → Edit
3. **Update Settings** - Admin → Settings
4. **Upload Image** - Any image upload form

**✅ Success Criteria:**
- All forms submit successfully
- No 419 errors

---

### 🟢 **NICE TO HAVE Test 6: Session Timeout**
**Why:** Verify error handling when session truly expires

1. Set short session lifetime in `.env`:
   ```bash
   SESSION_LIFETIME=1  # 1 minute
   ```
2. Clear config cache: `php artisan config:clear`
3. Login
4. Wait 2 minutes (do nothing)
5. Try to submit any form

**✅ Success Criteria:**
- 419 error occurs (expected)
- User sees "Session Expired" toast message
- Toast suggests refreshing the page
- After refresh, user is logged out

**Note:** Remember to restore `SESSION_LIFETIME=120` after this test!

---

## Monitoring Tools

### Browser Console Commands

```javascript
// Check current CSRF token info
window.CSRFDebug.getTokenInfo();

// Output:
// {
//   token: "YKUrwXTQ6Q...",  // First 10 chars
//   length: 40,
//   timestamp: "2025-11-01T..."
// }

// View full CSRF state
window.CSRFDebug.logState();

// Force token refresh (debugging only)
await window.CSRFDebug.forceRefresh();
```

### Check Session Cookie

1. Open DevTools (F12)
2. Go to "Application" tab
3. Expand "Cookies" → `http://127.0.0.1:8000`
4. Look for:
   - `XSRF-TOKEN` - Should be present and encrypted
   - `laravel_session` - Should be present
   - `academic_scribe_session` - May be present

---

## What to Look For

### ✅ **GOOD Signs:**

1. **Console is clean** - No CSRF-related errors
2. **Forms submit smoothly** - No delays or retries
3. **Login/logout work** - No 419 errors
4. **Cookies are set** - `XSRF-TOKEN` cookie exists

### ❌ **BAD Signs:**

1. **Repeated 419 errors** on normal operations
2. **"CSRF token refresh failed"** messages
3. **Missing XSRF-TOKEN cookie**
4. **Session cookie not persisting**

---

## Troubleshooting

### Issue 1: Still Getting 419 Errors on Login

**Check:**
```bash
# 1. Verify session table exists (if using database sessions)
php artisan tinker
>>> \DB::table('sessions')->count();

# 2. Check session configuration
php artisan config:show session

# 3. Check if cookies are enabled in browser
# DevTools → Application → Cookies → Should show cookies
```

**Solutions:**
- Clear all browser cookies for the site
- Run: `php artisan migrate` (ensure sessions table exists)
- Check `.env` has: `SESSION_DRIVER=database`

---

### Issue 2: 419 Error on First Request After Page Load

**This is normal IF:**
- You just restarted the dev server
- You cleared browser cookies
- Session has truly expired

**Solution:** Just refresh the page once

---

### Issue 3: XSRF-TOKEN Cookie Not Being Set

**Check:**
```bash
# 1. Verify web middleware is applied
php artisan route:list | grep login

# 2. Check bootstrap/app.php includes:
#    HandleInertiaRequests middleware

# 3. Verify app.blade.php has:
#    <meta name="csrf-token" content="{{ csrf_token() }}">
```

---

### Issue 4: Logout Still Giving 419

**Debug:**
```javascript
// Before clicking logout, run in console:
window.CSRFDebug.logState();

// Then click logout and watch the console
// You should see the POST request to /logout
```

**Check:**
- Is `XSRF-TOKEN` cookie present before logout?
- Are you seeing TWO 419 errors (meaning retry also failed)?
- Check Laravel logs for additional context

---

## Testing Checklist

Print this and check off as you test:

```
[ ] Login form submits without 419
[ ] Logout works without 419  
[ ] Multi-step order form works end-to-end
[ ] Order form with authentication mid-flow works
[ ] Coupon validation works
[ ] Admin blog post creation works
[ ] Admin page editing works
[ ] Admin settings update works
[ ] Session timeout shows proper error message
[ ] Multiple tabs work (login in one affects both)
```

---

## Performance Testing

### Before vs After

**Before Fixes:**
- Token refresh on every page navigation: ~50-100ms delay
- Token refresh on login: ~50-100ms delay
- Token refresh on logout: ~50-100ms delay
- Total unnecessary requests per session: 10-20+

**After Fixes:**
- No unnecessary token refreshes
- Only defensive retry on actual 419 errors
- Cleaner network waterfall
- Better user experience

**To Measure:**
1. Open DevTools → Network tab
2. Filter: XHR
3. Navigate around the app
4. You should NOT see repeated `/csrf-token` requests

---

## Deployment Checklist

Before deploying to production:

```bash
# 1. Run tests locally ✅
npm run build
php artisan serve
# ... run all tests above

# 2. Commit changes
git add .
git commit -m "fix: resolve CSRF token mismatch issues app-wide"

# 3. Deploy to staging
git push origin staging
# ... deploy staging

# 4. Test on staging
# ... run all tests again on staging environment

# 5. Deploy to production (only after staging passes)
git push origin main
# ... deploy production

# 6. Monitor production
tail -f storage/logs/laravel.log | grep "419\|CSRF"
# Watch for first hour after deployment
```

---

## Rollback Instructions

If issues arise in production:

```bash
# Quick rollback
git log --oneline -5  # Find commit before CSRF fixes
git revert <commit-hash>
git push origin main

# Then redeploy the reverted code
```

---

## Success Metrics

After deployment, monitor for 24-48 hours:

### Before Fixes:
- 419 errors: ~20-50 per day
- User complaints: Multiple reports
- Logout success rate: ~80%

### After Fixes (Expected):
- 419 errors: 0-2 per day (only genuine timeouts)
- User complaints: None
- Logout success rate: ~100%

---

## Support

If you need help:

1. **Check Laravel Logs:**
   ```bash
   tail -f storage/logs/laravel.log
   ```

2. **Enable Query Logging (if needed):**
   ```bash
   php artisan tinker
   >>> \DB::enableQueryLog();
   >>> // Do the action that causes issues
   >>> \DB::getQueryLog();
   ```

3. **Check Session Data:**
   ```bash
   php artisan tinker
   >>> \DB::table('sessions')->latest()->first();
   ```

---

**Testing Priority:** High 🔴  
**Estimated Testing Time:** 30-45 minutes  
**Required Before:** Production deployment  

**Good luck with testing!** 🎉























