# Discount Type Investigation Report

## Summary
Investigation of discount types in the database and their integration with the order form.

## Discount Types Found in Database

The system supports **two discount types**:
1. **`percentage`** - Percentage-based discounts (e.g., 10%, 25%)
2. **`fixed`** - Fixed amount discounts (e.g., $50, $100)

### Current Database State
- **Percentage coupons**: 3 active coupons
  - DEFAULT10 (10%)
  - NEWCOMER25 (25.01%)
  - FLASH30 (30%)
- **Fixed coupons**: 1 test coupon
  - FIXED50TEST ($50.00) - Created for testing

## Code Architecture

### Backend Implementation

#### Model: `app/Models/Coupon.php`
- Constants defined:
  ```php
  const DISCOUNT_FIXED = 'fixed';
  const DISCOUNT_PERCENTAGE = 'percentage';
  ```

#### Service: `app/Services/CouponService.php`
The `calculateDiscount()` method handles both types correctly:
```php
public function calculateDiscount(Coupon $coupon, float $orderTotal, int $pages = 1): float
{
    if ($coupon->discount_type === Coupon::DISCOUNT_PERCENTAGE) {
        return round(($orderTotal * $coupon->discount_amount) / 100, 2);
    }
    
    // For fixed amount discounts, ensure discount doesn't exceed order total
    return min(round((float)$coupon->discount_amount, 2), $orderTotal);
}
```

**Key Features:**
- Percentage discounts: Calculates percentage of order total
- Fixed discounts: Uses fixed amount but caps at order total (prevents negative net amounts)

#### Controller: `app/Http/Controllers/OrdersController.php`
The `validateCoupon()` endpoint returns:
```php
return response()->json([
    'valid' => true,
    'type' => $coupon->discount_type,           // 'percentage' or 'fixed'
    'amount' => $coupon->discount_amount,       // Raw amount (10 for 10%, or 50 for $50)
    'calculated_discount' => $discount,         // Actual discount amount in currency
    'coupon_type' => $coupon->coupon_type,
    'description' => $this->getCouponDescription($coupon),
    'expiry_date' => $coupon->expiry_date ? $coupon->expiry_date->format('Y-m-d') : null
]);
```

### Frontend Implementation

#### Order Form: `resources/js/Pages/Order/Create.jsx`

**Discount Application (lines 410-413):**
```javascript
const discountValue =
    response.data.type === "percentage"
        ? (parseFloat(totalAmount) * response.data.amount) / 100
        : parseFloat(response.data.amount);
```

**Discount Display (lines 1491-1514):**
- Percentage: Shows "X% OFF" with calculated dollar amount
- Fixed: Shows "$X OFF" with the fixed amount

**Frontend Calculation (lines 411-415, 657-659):**
```javascript
// When validating coupon (uses backend calculated_discount)
const discountValue = response.data.calculated_discount !== undefined
    ? parseFloat(response.data.calculated_discount)
    : response.data.type === "percentage"
        ? (parseFloat(totalAmount) * response.data.amount) / 100
        : Math.min(parseFloat(response.data.amount), parseFloat(totalAmount));

// When recalculating (caps fixed discounts)
const discount = appliedDiscount.discountType === "percentage" 
    ? (totalAmount * appliedDiscount.discountAmount) / 100 
    : Math.min(parseFloat(appliedDiscount.discountAmount) || 0, totalAmount);
```

## Test Results

### Backend Calculation Tests ✓

**Test 1: Percentage Discount**
- Coupon: DEFAULT10 (10% discount)
- Order Total: $100
- Expected Discount: $10
- Calculated Discount: $10
- **Result: ✓ CORRECT**

**Test 2: Fixed Discount**
- Coupon: FIXED50TEST ($50 fixed discount)
- Order Total: $100
- Expected Discount: $50
- Calculated Discount: $50
- **Result: ✓ CORRECT**

**Test 3: Fixed Discount Exceeding Order Total**
- Coupon: FIXED50TEST ($50 fixed discount)
- Order Total: $25
- Expected Discount: $25 (capped at order total)
- Calculated Discount: $25
- **Result: ✓ CORRECT** (Properly prevents negative net amounts)

## Integration Status

### ✅ Working Correctly

1. **Backend Validation**: Both discount types are validated correctly
2. **Backend Calculation**: Both types calculate correctly with proper capping for fixed discounts
3. **Frontend Display**: Both types display correctly with appropriate formatting
4. **Frontend Calculation**: Both types calculate correctly when applying discounts

### ✅ Issue Fixed

**Frontend Discount Calculation (lines 409-415, 657-659):**
Fixed an issue where fixed discounts weren't being capped at the order total in the frontend. The fix ensures:
1. Uses `calculated_discount` from backend response when available (properly capped)
2. Falls back to calculating on frontend with proper capping: `Math.min(discountAmount, orderTotal)`

This prevents negative net amounts when fixed discounts exceed the order total.

## Conclusion

**All discount types function correctly in tandem with the order form:**
- ✅ Percentage discounts work correctly
- ✅ Fixed discounts work correctly
- ✅ Fixed discounts are properly capped at order total
- ✅ Frontend correctly handles both types
- ✅ Backend correctly validates and calculates both types

The system is functioning as expected. Both discount types integrate properly with the order form and calculations are consistent between frontend and backend.

