# File Upload Issue - Investigation and Fix

## Issue Description
When placing an order as a client on the front-end and attaching files, the files were disappearing and not being found among the file uploads associated with the order.

## Investigation Findings

### 1. Root Cause Identified
The primary issue was in [app/Http/Controllers/OrdersController.php](app/Http/Controllers/OrdersController.php) at the `store()` method:

**Problem:** The `OrderFile::create()` call was using incorrect field values:
- `file_status` was set to `'instructions'` instead of a valid status constant
- `file_category` was missing (should be `'instructions'`)
- `access_control` field was missing entirely

**Impact:** This likely caused database insertion errors that were silently failing, resulting in files being uploaded to storage but not tracked in the database.

### 2. Secondary Issues
- Insufficient error handling and logging made it difficult to diagnose the issue
- No validation that files were successfully stored before creating database records
- No error handling for invalid files

## Changes Made

### Backend Changes

#### 1. Fixed OrderFile Creation ([OrdersController.php](app/Http/Controllers/OrdersController.php))
```php
// BEFORE (Incorrect)
OrderFile::create([
    'order_id' => $order->id,
    'file_name' => $fileName,
    'file_type' => $fileType,
    'uploader_id' => Auth::id(),
    'uploader_name' => Auth::user()->name,
    'uploader_user_type' => 'client',
    'file_url' => $filePath,
    'file_size' => $fileSize,
    'file_status' => 'instructions', // WRONG - should be a status, not category
    'date_uploaded' => now()->setTimezone($clientTimezone)
]);

// AFTER (Correct)
OrderFile::create([
    'order_id' => $order->id,
    'file_name' => $fileName,
    'file_type' => $fileType,
    'uploader_id' => Auth::id(),
    'uploader_name' => Auth::user()->name,
    'uploader_user_type' => 'client',
    'file_url' => $filePath,
    'file_size' => $fileSize,
    'file_status' => OrderFile::STATUS_PENDING,       // ✓ Correct status
    'file_category' => OrderFile::CATEGORY_INSTRUCTIONS, // ✓ Added category
    'access_control' => OrderFile::ACCESS_CLIENT_ADMIN,  // ✓ Added access control
    'date_uploaded' => now()->setTimezone($clientTimezone)
]);
```

#### 2. Added Comprehensive Logging
Added detailed logging at every stage of file upload:
- Log request details at the start of `store()` method
- Log file validation details
- Log each file as it's processed
- Log successful storage and database insertion
- Log any errors or exceptions

#### 3. Enhanced Error Handling
- Added try-catch blocks around validation
- Added file validity checks before processing
- Added checks to ensure files are stored successfully
- Added exception handling to prevent order creation failure if file upload fails
- Log detailed error messages with stack traces

#### 4. File Validation Improvements
- Check `$file->isValid()` before processing
- Log file validation errors
- Skip invalid files instead of failing the entire order

### Frontend Changes

#### 1. Added File Upload Logging ([WritingPreferences.jsx](resources/js/Components/OrderForm/WritingPreferences.jsx))
```javascript
const handleFileUpload = (e) => {
    const files = Array.from(e.target.files);
    console.log('[FILE UPLOAD] Files selected:', {
        count: files.length,
        files: files.map(f => ({
            name: f.name,
            size: f.size,
            type: f.type
        }))
    });
    setData("files", files);
    console.log('[FILE UPLOAD] Files set in form data');
};
```

#### 2. Enhanced Submission Logging ([Create.jsx](resources/js/Pages/Order/Create.jsx))
Added detailed logging before form submission:
- Log file count and details
- Log each file as it's appended to FormData
- Log when no files are present

## Testing Instructions

### 1. Test File Upload During Order Creation

1. **Clear browser cache and console**
   - Open Chrome DevTools (F12)
   - Right-click refresh button → "Empty Cache and Hard Reload"
   - Clear console

2. **Place a test order with files**
   - Navigate to the order creation page
   - Fill in all required fields
   - Attach 1-3 test files (various formats: PDF, DOCX, etc.)
   - Submit the order

3. **Monitor logs**
   
   **Browser Console:** Should see logs like:
   ```
   [FILE UPLOAD] Files selected: { count: 2, files: [...] }
   [FILE UPLOAD] Files set in form data
   [FILE UPLOAD] Checking files before submission: { hasFiles: true, filesCount: 2, ... }
   [FILE UPLOAD] Appending file 1: { name: "test.pdf", size: 12345, type: "application/pdf" }
   [FILE UPLOAD] Appending file 2: { name: "instructions.docx", size: 54321, type: "application/vnd..." }
   [FILE UPLOAD] All files appended to FormData
   Submitting order to /orders endpoint...
   ```

   **Laravel Logs:** Check `storage/logs/laravel.log` for:
   ```
   [INFO] Order creation attempt started { user_id: X, has_files: true, files_count: 2, files_data: [...] }
   [INFO] Order saved successfully, checking for file uploads { order_id: Y, has_files: true, files_count: 2 }
   [INFO] Files detected in request, starting upload process
   [INFO] Processing individual file { file_name: "test.pdf", is_valid: true, ... }
   [INFO] File stored in storage { file_path: "orders/...", file_exists: true }
   [INFO] OrderFile record created { order_file_id: Z, file_name: "test.pdf" }
   [INFO] All files processed successfully { files_processed: 2 }
   ```

4. **Verify in database**
   ```bash
   php artisan tinker
   >>> $order = \App\Models\Order::latest()->first();
   >>> $order->orderFiles;
   # Should show all uploaded files
   ```

5. **Verify in UI**
   - Navigate to the order details page
   - Check that files are listed in the "Order Files" section
   - Try downloading each file to ensure they're accessible

### 2. Test Error Cases

1. **Upload oversized file (>10MB)**
   - Should see validation error
   - Order should still be created (without the file)

2. **Upload invalid file type**
   - Should be rejected by validation
   - Check logs for validation errors

3. **Upload with network interruption**
   - Monitor logs for file validation errors
   - Order should still be created

### 3. Check Log Output

View the logs in real-time:
```bash
tail -f storage/logs/laravel.log | grep -E "FILE|Order creation|file upload"
```

## Expected Results

✅ **Files should now:**
- Upload successfully when placing an order
- Be stored in `storage/app/public/orders/`
- Have corresponding database records in `order_files` table
- Be visible in the order details page
- Be downloadable by authorized users

✅ **Logs should show:**
- Clear trace of file upload process
- Any validation or storage errors
- Successful database record creation

## Rollback Instructions

If any issues occur, revert the changes:
```bash
git checkout app/Http/Controllers/OrdersController.php
git checkout resources/js/Components/OrderForm/WritingPreferences.jsx
git checkout resources/js/Pages/Order/Create.jsx
php artisan cache:clear
php artisan config:clear
```

## Next Steps (After Testing)

Once testing confirms the fix works:

### Remove Unnecessary Logging
Some of the verbose console.log and Log::info statements can be removed or reduced to only log errors:

**Keep:**
- Error logging
- File validation failures
- Database insertion failures

**Remove:**
- Detailed file information logging (unless needed for debugging)
- "Processing individual file" logs
- Console logs in WritingPreferences.jsx

Run: `php artisan cache:clear` after making any changes.

## Files Modified

1. [app/Http/Controllers/OrdersController.php](app/Http/Controllers/OrdersController.php)
   - Fixed `OrderFile::create()` field values
   - Added comprehensive logging
   - Enhanced error handling
   - Added file validation

2. [resources/js/Components/OrderForm/WritingPreferences.jsx](resources/js/Components/OrderForm/WritingPreferences.jsx)
   - Added file upload logging

3. [resources/js/Pages/Order/Create.jsx](resources/js/Pages/Order/Create.jsx)
   - Added file submission logging

## Additional Notes

- The fix addresses a critical bug where `file_status` was set to `'instructions'` instead of using the proper `OrderFile::STATUS_PENDING` constant
- The missing `file_category` and `access_control` fields would have caused database errors
- All OrderFile model constants should now be used correctly throughout the application
- The enhanced logging will help diagnose any future file upload issues quickly
