# Manual Deployment Guide - Blog Published_At Bug Fix

## Overview
This guide walks you through manually deploying the blog post date fix to your live production server.

## Pre-Deployment Checklist

### ✅ What We Changed:
1. **Database Migration** - Fixed `published_at` column auto-update issue
2. **Backend Controller** - Updated `BlogController.php` with date preservation logic
3. **Frontend Component** - Enhanced `EditPost.jsx` with timeline display
4. **Frontend Component** - Updated `BlogPost.jsx` with published/updated dates
5. **Migration File** - Updated original migration for future installs

### ⚠️ Important Safety Notes:
- **Backup your database before running migrations**
- **Test on staging environment first if available**
- **The migration will NOT affect existing data** (only changes column definition)
- **Deployment has ZERO downtime** - users can continue using the site

---

## Deployment Steps

### STEP 1: Backup Your Production Database

Before making ANY changes, backup your database:

```bash
# SSH into your server
ssh user@your-server.com

# Navigate to your project directory
cd /path/to/academic-scribe

# Create a backup
php artisan backup:database
# OR manually with mysqldump:
mysqldump -u your_db_user -p your_db_name > backup_$(date +%Y%m%d_%H%M%S).sql
```

**Save this backup file somewhere safe!**

---

### STEP 2: Upload Changed Files to Production

You have **TWO options**: Git Pull or Manual Upload

#### Option A: Using Git (Recommended)

```bash
# SSH into your server
ssh user@your-server.com
cd /path/to/academic-scribe

# Pull the latest changes
git pull origin main
# OR if you committed to a different branch:
git pull origin your-branch-name
```

#### Option B: Manual File Upload (FTP/SFTP)

Upload these **4 files** to your server using FileZilla, WinSCP, or similar:

1. **`app/Http/Controllers/Admin/BlogController.php`**
   - Upload to: `/path/to/academic-scribe/app/Http/Controllers/Admin/`

2. **`resources/js/Pages/Admin/Blog/EditPost.jsx`**
   - Upload to: `/path/to/academic-scribe/resources/js/Pages/Admin/Blog/`

3. **`resources/js/Pages/Blog/BlogPost.jsx`**
   - Upload to: `/path/to/academic-scribe/resources/js/Pages/Blog/`

4. **`database/migrations/2025_10_27_210347_fix_blog_posts_published_at_column.php`**
   - Upload to: `/path/to/academic-scribe/database/migrations/`

5. **`database/migrations/2025_08_02_081104_create_blog_posts_table.php`**
   - Upload to: `/path/to/academic-scribe/database/migrations/`

---

### STEP 3: Install Dependencies (If Needed)

Only needed if your `package.json` or `composer.json` changed (they didn't in this case, but good practice):

```bash
# SSH into your server
ssh user@your-server.com
cd /path/to/academic-scribe

# Install/update composer dependencies
composer install --no-dev --optimize-autoloader

# Install/update npm dependencies (optional, only if package.json changed)
npm install --production
```

---

### STEP 4: Run Database Migration

This is the **critical step** that fixes the database column:

```bash
# SSH into your server
ssh user@your-server.com
cd /path/to/academic-scribe

# Run the migration
php artisan migrate

# You should see:
# INFO  Running migrations.
# 2025_10_27_210347_fix_blog_posts_published_at_column ........ DONE
```

**What this does:**
- Removes auto-update behavior from `published_at` column
- Does NOT affect any existing data
- Takes less than 1 second to run
- No downtime required

---

### STEP 5: Build Frontend Assets

Rebuild the JavaScript/CSS assets with the new changes:

```bash
# Still on your server
cd /path/to/academic-scribe

# Build for production
npm run build

# This will take 30-60 seconds and should complete successfully
```

**Expected output:** You should see `✓ built in XXs` at the end.

---

### STEP 6: Clear All Caches

Clear Laravel caches to ensure changes take effect immediately:

```bash
# Still on your server
cd /path/to/academic-scribe

# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Optional: Optimize for production
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

---

### STEP 7: Fix Existing Posts with Future Dates (Optional)

If you have posts currently showing 404 errors due to future dates, run this SQL query:

```bash
# Connect to MySQL
mysql -u your_db_user -p your_db_name

# Run this query to fix future dates:
UPDATE blog_posts 
SET published_at = NOW() 
WHERE published_at > DATE_ADD(NOW(), INTERVAL 5 MINUTE) 
  AND status = 'published';

# Exit MySQL
exit;
```

**What this does:**
- Finds any published posts with dates more than 5 minutes in the future
- Sets them to the current time
- Makes them immediately visible on your blog

---

### STEP 8: Restart Services (If Needed)

Depending on your server setup, you may need to restart services:

```bash
# If using Laravel Octane
php artisan octane:reload

# If using PHP-FPM
sudo systemctl restart php8.2-fpm  # Adjust version as needed

# If using queue workers
php artisan queue:restart

# If using Nginx
sudo systemctl reload nginx

# If using Apache
sudo systemctl reload apache2
```

---

### STEP 9: Verify Deployment

Test that everything works:

#### 9.1 Check Migration Status
```bash
php artisan migrate:status

# Should show:
# 2025_10_27_210347_fix_blog_posts_published_at_column ... Ran
```

#### 9.2 Test Admin Panel
1. Visit: `https://your-site.com/admin/blog/posts`
2. Edit an existing published post
3. Change the title
4. Click "Save"
5. **Verify:**
   - ✅ Post saves successfully
   - ✅ `published_at` date stayed the same (didn't change to today)
   - ✅ Post is immediately visible (no 404)
   - ✅ New "Publish Timeline" card shows in sidebar with all dates

#### 9.3 Test Frontend Blog
1. Visit: `https://your-site.com/blog`
2. Click on a blog post
3. **Verify:**
   - ✅ Author name is displayed
   - ✅ Published date is shown
   - ✅ Updated date is shown (if post was edited)
   - ✅ Post loads without errors

#### 9.4 Check Browser Console
1. Press F12 to open developer tools
2. Look at Console tab
3. **Verify:** No JavaScript errors

#### 9.5 Check Laravel Logs
```bash
tail -f storage/logs/laravel.log

# Look for any errors
# You should see successful log entries like:
# "Already published - preserving original published_at"
```

---

## Rollback Plan (If Something Goes Wrong)

If you encounter issues, here's how to rollback:

### Rollback Database Migration
```bash
php artisan migrate:rollback --step=1
```

This will restore the old column definition with auto-update.

### Restore Files
If you used Git:
```bash
git reset --hard HEAD~1  # Go back one commit
```

If you uploaded manually:
- Re-upload the old versions of the files from your local backup

### Restore Database Backup
```bash
mysql -u your_db_user -p your_db_name < backup_YYYYMMDD_HHMMSS.sql
```

---

## Troubleshooting

### Problem: Migration Fails
**Error:** "SQLSTATE[42000]: Syntax error..."

**Solution:**
```bash
# Check database connection
php artisan db:show

# Check migration file syntax
cat database/migrations/2025_10_27_210347_fix_blog_posts_published_at_column.php

# Try again with verbose output
php artisan migrate --verbose
```

### Problem: Frontend Build Fails
**Error:** "npm ERR! code ELIFECYCLE"

**Solution:**
```bash
# Clear npm cache
npm cache clean --force

# Remove node_modules and reinstall
rm -rf node_modules package-lock.json
npm install

# Try build again
npm run build
```

### Problem: Changes Not Visible
**Solution:**
```bash
# Clear ALL caches
php artisan optimize:clear
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

# Rebuild optimizations
php artisan optimize

# Hard refresh browser: Ctrl+Shift+R (Windows) or Cmd+Shift+R (Mac)
```

### Problem: "Class not found" Error
**Solution:**
```bash
# Regenerate autoload files
composer dump-autoload

# Clear cache again
php artisan cache:clear
```

---

## Post-Deployment Monitoring

### Day 1-3 After Deployment
Monitor these:

1. **Laravel Logs**
   ```bash
   tail -f storage/logs/laravel.log
   ```

2. **Server Error Logs**
   ```bash
   # Nginx
   tail -f /var/log/nginx/error.log
   
   # Apache
   tail -f /var/log/apache2/error.log
   ```

3. **Database Queries**
   - Check for slow queries
   - Monitor `blog_posts` table updates

4. **User Reports**
   - Watch for 404 errors on blog posts
   - Check if dates are preserving correctly

---

## Summary of What Changed

### Files Modified:
1. ✅ `app/Http/Controllers/Admin/BlogController.php` - Date preservation logic
2. ✅ `resources/js/Pages/Admin/Blog/EditPost.jsx` - Enhanced timeline UI
3. ✅ `resources/js/Pages/Blog/BlogPost.jsx` - Display dates on frontend
4. ✅ `database/migrations/2025_10_27_210347_fix_blog_posts_published_at_column.php` - New migration
5. ✅ `database/migrations/2025_08_02_081104_create_blog_posts_table.php` - Updated original

### Database Changes:
- ✅ `blog_posts.published_at` column no longer auto-updates

### What's Fixed:
- ✅ Published dates preserve original value when editing
- ✅ No more 404 errors after saving posts
- ✅ Timezone issues resolved
- ✅ SEO-friendly date preservation
- ✅ Beautiful admin timeline display
- ✅ Public blog shows published/updated dates

---

## Quick Command Reference

```bash
# Complete deployment in one go:
ssh user@your-server.com
cd /path/to/academic-scribe
git pull origin main
php artisan migrate
npm run build
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
php artisan config:cache
php artisan route:cache
```

---

## Support

If you encounter any issues during deployment:

1. **Check Laravel logs**: `storage/logs/laravel.log`
2. **Check server logs**: `/var/log/nginx/error.log` or `/var/log/apache2/error.log`
3. **Verify migration status**: `php artisan migrate:status`
4. **Test locally first**: Make sure changes work on your local environment

---

**Deployment Time Estimate:** 5-10 minutes
**Downtime Required:** None (zero downtime deployment)
**Risk Level:** Low (changes are non-destructive)

Good luck with your deployment! 🚀

