# Price Calculator Popup - Workflow Document

## Overview
Create a reusable price calculator popup/modal that can be triggered from any service page or blog post, allowing clients (authenticated or anonymous) to quickly check assignment costs without navigating to the full order form.

---

## 1. OBJECTIVES

### Primary Goals
- ✅ Provide instant pricing estimates from any page (service pages, blog posts)
- ✅ Reduce friction in the pricing discovery process
- ✅ Support both authenticated and anonymous users
- ✅ Use the same pricing calculation logic as the main order form
- ✅ Encourage users to proceed to full order form after seeing pricing

### User Experience Goals
- **Quick**: Open, fill minimal fields, get instant pricing
- **Transparent**: Show pricing breakdown clearly
- **Non-intrusive**: Modal/popup that doesn't interrupt browsing
- **Responsive**: Works on mobile and desktop
- **Accessible**: Keyboard navigation, screen reader support

---

## 2. PRICING CALCULATION LOGIC (Existing)

### Data Source
All pricing data comes from the database via Laravel models:
- `AcademicLevel::all()` - Returns all academic levels with `id`, `value`, `name`
- `PaperType::all()` - Returns all paper types with `id`, `value`, `name`
- `Subject::all()` - Returns all subjects with `id`, `value`, `name`
- `Pricing::all()` - Returns all deadlines with `id`, `value`, `urgency`, `duration`
- `AdminSetting` - Contains `price_per_slide` setting

The calculation uses the `value` field from each model as multipliers in the formula.

### Current Calculation Formula
```php
// Step 1: Base cost per page (double-spaced)
baseCostPerPage = (academic_level + subject + paper_type) + urgency

// Step 2: Apply spacing multiplier
costPerPage = baseCostPerPage * spacing
// spacing: 1 (double), 1.5 (1.5x), 2 (single)

// Step 3: Calculate base price
basePrice = costPerPage * pages

// Step 4: Add slides (if applicable)
basePrice += (slides * price_per_slide)

// Step 5: Apply discounts (for authenticated users)
finalPrice = basePrice - discount
```

### Required Fields for Calculation
**Minimum Required (Anonymous):**
- `academic_level_id` (integer, required)
- `paper_type_id` (integer, required)
- `deadline_id` (integer, required)
- `pages` (integer, min: 1, required)
- `subject_id` (integer, required)

**Optional Fields:**
- `slides` (integer, min: 0, nullable)
- `spacing` (string: 'single', '1.5', or default double)

**Additional for Authenticated Users:**
- `coupon_code` (string, nullable)

---

## 3. API ENDPOINTS (Existing)

### Endpoint 1: Anonymous Pricing (Public)
```
POST /calculate-pricing
Content-Type: application/json

Request Body:
{
    "academic_level_id": 1,
    "paper_type_id": 2,
    "deadline_id": 3,
    "pages": 5,
    "subject_id": 4,
    "slides": 0 (optional),
    "spacing": "double" (optional)
}

Response:
{
    "success": true,
    "base_price": 125.50,
    "currency": "USD",
    "estimated_savings": {
        "min": 12.55,
        "max": 25.10,
        "min_final_price": 100.40,
        "max_final_price": 112.95
    },
    "discount_info": {
        "minimum_discount": 10,
        "newcomer_discount": 20,
        "guarantee_message": "..."
    }
}
```

### Endpoint 2: Authenticated Pricing (If User is Logged In)
```
POST /calculate-order-pricing
Headers: X-CSRF-TOKEN (if authenticated)
Content-Type: application/json

Request Body:
{
    "academic_level_id": 1,
    "paper_type_id": 2,
    "deadline_id": 3,
    "pages": 5,
    "subject_id": 4,
    "slides": 0,
    "spacing": "double",
    "coupon_code": "FIRST20" (optional)
}

Response:
{
    "success": true,
    "pricing": {
        "base_price": 125.50,
        "total_before_discount": 125.50,
        "final_price": 100.40,
        "discount_amount": 25.10,
        "breakdown": {
            "pages_cost": 125.50,
            "slides_cost": 0,
            "cost_per_page": 25.10
        }
    }
}
```

---

## 4. COMPONENT STRUCTURE

### 4.1 Main Component: `PriceCalculatorPopup`
**Location:** `resources/js/Components/PriceCalculator/PriceCalculatorPopup.jsx`

**Props:**
```javascript
{
    isOpen: boolean,              // Control modal visibility
    onClose: function,            // Close handler
    triggerSource: string,        // 'service-page' | 'blog-post' | 'cta-button'
    
    // Pricing data (same structure as order form)
    academicLevels: Array<{ id: number, value: number, name: string }>,
    paperTypes: Array<{ id: number, value: number, name: string }>,
    subjects: Array<{ id: number, value: number, name: string }>,
    deadlines: Array<{ id: number, value: number, urgency: string, duration: string }>,
    singleSlidePrice: number,     // From AdminSetting->price_per_slide
    
    // User data
    isAuthenticated: boolean,
    user?: {                      // If authenticated
        id: number,
        email: string
    },
    
    // Optional pre-filled data
    initialData?: {
        academic_level_id?: number,
        paper_type_id?: number,
        subject_id?: number
    }
}
```

**State Management:**
```javascript
{
    // Form Data
    formData: {
        academic_level_id: null,
        paper_type_id: null,
        deadline_id: null,
        pages: 1,
        subject_id: null,
        slides: 0,
        spacing: 'double',
        coupon_code: ''  // Only if authenticated
    },
    
    // Pricing Data
    pricing: {
        base_price: 0,
        final_price: 0,
        discount_amount: 0,
        currency: 'USD',
        breakdown: {},
        isLoading: false,
        error: null
    },
    
    // UI State
    activeStep: 'basic' | 'advanced' | 'results',
    errors: {},
    touched: {}
}
```

### 4.2 Sub-Components

#### 4.2.1 `PriceCalculatorForm`
**Purpose:** Collect user input for pricing calculation

**Data Source:** Options come from database models (AcademicLevel, PaperType, Subject, Pricing) - same as order form

**Fields (Two-Step Approach):**

**Step 1 - Basic Fields (Always Visible):**
1. Academic Level (required)
   - Dropdown/Radio selection
   - Options: From `AcademicLevel::all()` - e.g., High School, Undergraduate, Masters, PHD
   - Value stored: `id` (integer)
   - Display: `name` field
   
2. Paper Type (required)
   - Dropdown/Radio selection
   - Options: From `PaperType::all()` - e.g., Essay, Research Paper, Dissertation, etc.
   - Value stored: `id` (integer)
   - Display: `name` field
   
3. Subject (required)
   - Searchable dropdown
   - Options: From `Subject::all()` - all subjects from database
   - Value stored: `id` (integer)
   - Display: `name` field
   
4. Pages (required)
   - Number input with +/- buttons
   - Default: 1, Min: 1, Max: 100
   - Value stored: integer
   
5. Deadline/Urgency (required)
   - Radio button grid or dropdown
   - Options: From `Pricing::all()` - all deadline options from database
   - Value stored: `id` (integer)
   - Display: `urgency + " " + duration` (e.g., "3 days", "6 hours")

**Step 2 - Advanced Fields (Collapsible/Optional):**
6. Slides (optional)
   - Number input
   - Default: 0, Min: 0
   
7. Spacing (optional)
   - Radio buttons: Double (default), 1.5, Single
   
8. Coupon Code (optional, authenticated only)
   - Text input with validation button
   - Show "Apply Coupon" button

#### 4.2.2 `PriceCalculatorResults`
**Purpose:** Display calculated pricing

**For Anonymous Users:**
```
┌─────────────────────────────────────┐
│ Estimated Price                     │
├─────────────────────────────────────┤
│ Base Price:           $125.50       │
│                                       │
│ 💰 You Could Save:                  │
│   10% - 20% discount available      │
│   Final Price: $100.40 - $112.95    │
│                                       │
│ 📊 Pricing Breakdown:               │
│   • Cost per page: $25.10           │
│   • Pages: 5                        │
│   • Total: $125.50                  │
│                                       │
│ ✅ All customers save at least 10%  │
│                                       │
│ [Calculate Again] [Place Order]     │
└─────────────────────────────────────┘
```

**For Authenticated Users:**
```
┌─────────────────────────────────────┐
│ Your Price                          │
├─────────────────────────────────────┤
│ Base Price:           $125.50       │
│ Discount:             -$25.10       │
│ ─────────────────────────────────   │
│ Final Price:          $100.40       │
│                                       │
│ 📊 Pricing Breakdown:               │
│   • Cost per page: $25.10           │
│   • Pages: 5                        │
│   • Slides: 0                       │
│   • Spacing: Double                 │
│   • Subtotal: $125.50               │
│   • Discount (FIRST20): -$25.10     │
│   • Total: $100.40                  │
│                                       │
│ [Calculate Again] [Place Order]     │
└─────────────────────────────────────┘
```

#### 4.2.3 `PriceCalculatorTrigger`
**Purpose:** Button/link component to open the calculator

**Variants:**
- Primary CTA: "Get Instant Pricing" or "Calculate Price"
- Secondary: "Check Pricing" or "Price Calculator"
- Icon button: 💰 icon
- Text link: "See pricing" or "How much?"

**Props:**
```javascript
{
    variant: 'primary' | 'secondary' | 'icon' | 'link',
    className?: string,
    children?: ReactNode,
    source: string  // Track where it was clicked
}
```

---

## 5. USER FLOW

### 5.1 Opening the Calculator

**From Service Page:**
1. User clicks "Get Instant Pricing" button
2. Modal opens with pre-filled data (if applicable):
   - Paper type pre-selected based on service page
   - Subject might be suggested based on page context
3. User sees basic form fields

**From Blog Post:**
1. User clicks "Calculate Price" CTA in blog post
2. Modal opens with minimal pre-filled data
3. User sees basic form fields

### 5.2 Filling the Form

**Step 1: Basic Information**
1. User selects Academic Level (e.g., "Master's")
2. User selects Paper Type (e.g., "Essay")
3. User searches and selects Subject (e.g., "Business")
4. User sets Pages (e.g., 5)
5. User selects Deadline (e.g., "3 days")

**Real-time Calculation (Optional Enhancement):**
- As user fills fields, show "Calculating..." state
- Auto-calculate when all required fields are filled
- Show estimated price range immediately

**Step 2: Advanced Options (Optional)**
- User can expand "Advanced Options" section
- Add slides if needed
- Change spacing if needed
- Enter coupon code (if authenticated)

### 5.3 Viewing Results

**After Form Submission:**
1. Show loading state: "Calculating your price..."
2. API call to `/calculate-pricing` or `/calculate-order-pricing`
3. Display results in `PriceCalculatorResults` component
4. Show breakdown and final price

**Results Actions:**
- **"Calculate Again"**: Reset form, keep modal open
- **"Place Order"**: Close modal, redirect to `/place-order` with form data preserved
- **"Close"**: Close modal, return to previous page

### 5.4 Proceeding to Order

**Data Preservation:**
- Store form data in session/localStorage
- Pass data to order form via URL params or session
- Pre-fill order form with calculator data

**Redirect Flow:**
```
Price Calculator → /place-order?prefill=true
Order Form → Auto-populate fields from calculator data
```

---

## 6. TECHNICAL IMPLEMENTATION DETAILS

### 6.1 Data Requirements

**Database Models Structure:**
The pricing data comes from the database and follows the same structure as the order form:

**Models:**
- `AcademicLevel`: `{ id, value, name }` - e.g., `{ id: 2, value: 1.2, name: "Undergraduate" }`
- `PaperType`: `{ id, value, name }` - e.g., `{ id: 1, value: 5, name: "Essay" }`
- `Subject`: `{ id, value, name }` - e.g., `{ id: 1, value: 2, name: "Business" }`
- `Pricing` (deadlines): `{ id, value, urgency, duration }` - e.g., `{ id: 3, value: 5, urgency: "3", duration: "days" }`
- `AdminSetting`: Contains `price_per_slide` setting

**Backend Data Retrieval (Same as Order Form):**
```php
// In Controller (same pattern as OrdersController::create())
$academicLevels = AcademicLevel::all();
$deadlines = Pricing::all();
$paperTypes = PaperType::all();
$subjects = Subject::all();
$costPerSlide = AdminSetting::where('price_per_slide', '>', 0)->value('price_per_slide');
```

**Frontend Data Structure (Passed via Inertia or API):**
```javascript
{
    academicLevels: [
        { id: 1, value: 1, name: "High School" },
        { id: 2, value: 1.2, name: "Undergraduate" },
        { id: 3, value: 1.4, name: "Masters" },
        { id: 4, value: 1.8, name: "PHD" }
    ],
    paperTypes: [
        { id: 1, value: 5, name: "Essay" },
        { id: 2, value: 8, name: "Research Paper" },
        // ... etc
    ],
    subjects: [
        { id: 1, value: 2, name: "Business" },
        { id: 2, value: 3, name: "Computer Science" },
        // ... etc
    ],
    deadlines: [
        { 
            id: 1, 
            value: 5,
            urgency: "3",
            duration: "days"
        },
        { 
            id: 2, 
            value: 10,
            urgency: "6",
            duration: "hours"
        },
        // ... etc
    ],
    singleSlidePrice: 5,  // From AdminSetting->price_per_slide
    isAuthenticated: boolean,
    user: User | null
}
```

**Important Note:** The API endpoints expect **IDs** (integers), not formatted strings. The calculator form must send:
- `academic_level_id` (integer)
- `paper_type_id` (integer)
- `deadline_id` (integer)
- `subject_id` (integer)

### 6.2 State Management

**Options:**
1. **Local Component State** (Recommended for MVP)
   - Use React useState/useReducer
   - Simpler, self-contained
   
2. **Global State** (For Future Enhancement)
   - Context API or Zustand
   - Share state across components
   - Persist data across page navigation

### 6.3 API Integration

**Implementation:**
```javascript
// Anonymous pricing
const calculateAnonymousPricing = async (formData) => {
    const response = await axios.post('/calculate-pricing', {
        academic_level_id: formData.academic_level_id,  // Integer ID from AcademicLevel model
        paper_type_id: formData.paper_type_id,          // Integer ID from PaperType model
        deadline_id: formData.deadline_id,              // Integer ID from Pricing model
        pages: formData.pages,                          // Integer
        subject_id: formData.subject_id,                // Integer ID from Subject model
        slides: formData.slides || 0                    // Integer (optional)
        // Note: spacing is not in anonymous endpoint, but can be added if needed
    });
    return response.data;
};

// Authenticated pricing
const calculateAuthenticatedPricing = async (formData) => {
    const response = await axios.post('/calculate-order-pricing', {
        academic_level_id: formData.academic_level_id,  // Integer ID
        paper_type_id: formData.paper_type_id,          // Integer ID
        deadline_id: formData.deadline_id,              // Integer ID
        pages: formData.pages,                          // Integer
        subject_id: formData.subject_id,                // Integer ID
        slides: formData.slides || 0,                   // Integer (optional)
        spacing: formData.spacing || 'double',          // String: 'single', '1.5', or 'double'
        coupon_code: formData.coupon_code || null       // String (optional)
    }, {
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    });
    return response.data;
};
```

**Important:** The form must send **integer IDs** to the API, not the formatted strings used in the order form UI. The calculator should work with IDs directly.

### 6.4 Form Validation

**Client-Side Validation:**
- Required fields must be filled
- Pages must be >= 1
- Slides must be >= 0
- Show inline error messages
- Disable submit button until valid

**Server-Side Validation:**
- Already exists in backend
- Return error messages if validation fails
- Display errors in UI

### 6.5 Error Handling

**Scenarios:**
1. **Network Error**: Show "Unable to calculate. Please try again."
2. **Validation Error**: Show field-specific errors
3. **Server Error**: Show generic error message
4. **Timeout**: Show timeout message, allow retry

**Error Display:**
```javascript
{
    errors: {
        academic_level_id: "Please select an academic level",
        general: "Unable to calculate pricing. Please try again."
    }
}
```

---

## 7. UI/UX CONSIDERATIONS

### 7.1 Modal/Popup Design

**Layout Options:**
1. **Modal Overlay** (Recommended)
   - Full-screen overlay with backdrop
   - Centered modal (max-width: 600px on desktop)
   - Close button (X) in top-right
   - Click outside to close (optional)

2. **Slide-in Panel** (Alternative)
   - Slides in from right (desktop) or bottom (mobile)
   - Less intrusive
   - Better for mobile experience

**Responsive Behavior:**
- Desktop: Centered modal, max-width 600px
- Tablet: Full-width with padding
- Mobile: Full-screen modal

### 7.2 Form UX

**Field Organization:**
- Group related fields together
- Use clear labels and placeholders
- Show required field indicators (*)
- Provide helpful hints/tooltips

**Input Types:**
- Dropdowns for selections (Academic Level, Paper Type)
- Searchable dropdown for Subjects
- Number inputs with +/- buttons for Pages/Slides
- Radio buttons for Deadline (visual grid)
- Radio buttons for Spacing

**Progressive Disclosure:**
- Show basic fields first
- "Advanced Options" collapsible section
- Don't overwhelm with all fields at once

### 7.3 Loading States

**During Calculation:**
```
┌─────────────────────────────────────┐
│ [Spinner] Calculating your price... │
└─────────────────────────────────────┘
```

**Skeleton Loading (Optional):**
- Show skeleton placeholders while fetching options
- Better perceived performance

### 7.4 Success/Results Display

**Visual Hierarchy:**
1. Final price (largest, bold)
2. Breakdown (collapsed by default, expandable)
3. Call-to-action buttons

**Color Coding:**
- Base price: Gray/neutral
- Discount: Green (savings)
- Final price: Primary color (emphasized)

### 7.5 Accessibility

**Requirements:**
- Keyboard navigation (Tab, Enter, Esc)
- Screen reader support (ARIA labels)
- Focus management (trap focus in modal)
- Close on Esc key
- Clear focus indicators

---

## 8. INTEGRATION POINTS

### 8.1 Service Pages Integration

**Location:** All service pages (e.g., `/services/essay-writing`)

**Data Requirements:**
Service pages need to pass the pricing data (same as order form) to the calculator:
- `academicLevels` - From `AcademicLevel::all()`
- `paperTypes` - From `PaperType::all()`
- `subjects` - From `Subject::all()`
- `deadlines` - From `Pricing::all()`
- `singleSlidePrice` - From `AdminSetting->price_per_slide`

**Implementation:**
```jsx
// In ServicePage component (or pass via Inertia props from controller)
import { PriceCalculatorTrigger } from '@/Components/PriceCalculator/PriceCalculatorTrigger';
import { PriceCalculatorPopup } from '@/Components/PriceCalculator/PriceCalculatorPopup';

function ServicePage({ 
    academicLevels, 
    paperTypes, 
    subjects, 
    deadlines, 
    singleSlidePrice,
    isAuthenticated,
    user 
}) {
    const [calculatorOpen, setCalculatorOpen] = useState(false);
    
    return (
        <>
            {/* CTA Button */}
            <PriceCalculatorTrigger
                variant="primary"
                onClick={() => setCalculatorOpen(true)}
                source="service-page"
            >
                Get Instant Pricing
            </PriceCalculatorTrigger>
            
            {/* Popup */}
            <PriceCalculatorPopup
                isOpen={calculatorOpen}
                onClose={() => setCalculatorOpen(false)}
                triggerSource="service-page"
                academicLevels={academicLevels}
                paperTypes={paperTypes}
                subjects={subjects}
                deadlines={deadlines}
                singleSlidePrice={singleSlidePrice}
                isAuthenticated={isAuthenticated}
                user={user}
                initialData={{
                    // Optionally pre-fill based on service page context
                    paper_type_id: paperType?.id,  // If known from page
                }}
            />
        </>
    );
}
```

**Alternative: Fetch Data in Calculator Component**
If data is not passed via props, the calculator can fetch it via API endpoint (to be created) or use a shared data provider.

### 8.2 Blog Posts Integration

**Location:** Blog post pages (e.g., `/blog/how-to-write-essay`)

**Data Requirements:**
Blog posts also need pricing data. Options:
1. Pass via Inertia props from BlogController
2. Fetch via API endpoint when calculator opens (lazy loading)
3. Use a global data provider/context

**Implementation:**
```jsx
// In BlogPost component
function BlogPost({ 
    academicLevels, 
    paperTypes, 
    subjects, 
    deadlines, 
    singleSlidePrice,
    isAuthenticated,
    user 
}) {
    const [calculatorOpen, setCalculatorOpen] = useState(false);
    
    return (
        <>
            {/* CTA in blog content */}
            <PriceCalculatorTrigger
                variant="secondary"
                onClick={() => setCalculatorOpen(true)}
                source="blog-post"
            >
                Calculate Your Essay Price
            </PriceCalculatorTrigger>
            
            {/* Popup */}
            <PriceCalculatorPopup
                isOpen={calculatorOpen}
                onClose={() => setCalculatorOpen(false)}
                triggerSource="blog-post"
                academicLevels={academicLevels}
                paperTypes={paperTypes}
                subjects={subjects}
                deadlines={deadlines}
                singleSlidePrice={singleSlidePrice}
                isAuthenticated={isAuthenticated}
                user={user}
            />
        </>
    );
}
```

### 8.3 Header/Navigation Integration (Optional)

**Location:** Main navigation header

**Implementation:**
```jsx
// In Header/Navigation component
<PriceCalculatorTrigger
    variant="link"
    source="header"
>
    Price Calculator
</PriceCalculatorTrigger>
```

---

## 9. DATA FLOW TO ORDER FORM

### 9.1 Data Preservation Methods

**Option 1: URL Parameters (Recommended for Anonymous)**
```
/place-order?
    academic_level_id=2&
    paper_type_id=1&
    subject_id=5&
    pages=5&
    deadline_id=3&
    slides=0&
    spacing=double
```

**Option 2: Session Storage (Recommended for Authenticated)**
```javascript
// Store in sessionStorage
sessionStorage.setItem('calculator_data', JSON.stringify(formData));

// Retrieve in order form
const calculatorData = JSON.parse(sessionStorage.getItem('calculator_data'));
```

**Option 3: Laravel Session (Server-side)**
```php
// Store in session before redirect
session(['calculator_data' => $request->all()]);

// Retrieve in order form controller
$calculatorData = session('calculator_data');
```

### 9.2 Order Form Pre-population

**In Order Form Component:**
```javascript
useEffect(() => {
    // Check for calculator data
    const urlParams = new URLSearchParams(window.location.search);
    const calculatorData = sessionStorage.getItem('calculator_data');
    
    if (urlParams.get('prefill') === 'true' && calculatorData) {
        const data = JSON.parse(calculatorData);
        // Pre-fill form fields
        setData({
            academic_level_id: data.academic_level_id,
            paper_type_id: data.paper_type_id,
            // ... etc
        });
        
        // Clear stored data
        sessionStorage.removeItem('calculator_data');
    }
}, []);
```

---

## 10. TESTING CONSIDERATIONS

### 10.1 Test Scenarios

**Functional Tests:**
- ✅ Open calculator from service page
- ✅ Open calculator from blog post
- ✅ Fill all required fields and calculate
- ✅ Fill with optional fields (slides, spacing)
- ✅ Test with coupon code (authenticated)
- ✅ Test error handling (network error, validation error)
- ✅ Test data preservation to order form
- ✅ Test responsive design (mobile, tablet, desktop)

**User Experience Tests:**
- ✅ Modal opens/closes smoothly
- ✅ Form validation works correctly
- ✅ Loading states display properly
- ✅ Results display correctly
- ✅ Keyboard navigation works
- ✅ Screen reader compatibility

**Edge Cases:**
- ✅ Invalid API response
- ✅ Very large page numbers
- ✅ Zero or negative values
- ✅ Rapid field changes
- ✅ Multiple rapid calculations

---

## 11. ANALYTICS & TRACKING

### 11.1 Events to Track

**Calculator Events:**
- `price_calculator_opened` (source: service-page/blog-post/header)
- `price_calculator_field_changed` (field_name, value)
- `price_calculator_calculated` (pricing_data)
- `price_calculator_order_clicked` (final_price)
- `price_calculator_closed` (with/without calculation)

**Conversion Tracking:**
- Calculator → Order Form conversion rate
- Calculator → Completed Order conversion rate
- Average time in calculator
- Most common field combinations

---

## 12. FUTURE ENHANCEMENTS

### Phase 2 Features (Post-MVP)
1. **Real-time Calculation**: Calculate as user types
2. **Save Calculations**: Allow users to save multiple calculations
3. **Compare Prices**: Compare different configurations side-by-side
4. **Email Results**: Send pricing estimate via email
5. **Currency Conversion**: Show prices in different currencies
6. **Price History**: Show price trends over time
7. **Bulk Calculator**: Calculate price for multiple assignments
8. **Recommended Packages**: Suggest package deals based on calculation

---

## 13. IMPLEMENTATION PHASES

### Phase 1: MVP (Minimum Viable Product)
**Scope:**
- ✅ Basic modal component
- ✅ Form with required fields only
- ✅ API integration (anonymous pricing)
- ✅ Results display
- ✅ Integration on 1 service page (test)
- ✅ Redirect to order form

**Timeline:** 1-2 weeks

### Phase 2: Enhanced Features
**Scope:**
- ✅ Authenticated user support
- ✅ Coupon code validation
- ✅ Advanced fields (slides, spacing)
- ✅ Better error handling
- ✅ Integration on all service pages
- ✅ Integration on blog posts

**Timeline:** 1 week

### Phase 3: Polish & Optimization
**Scope:**
- ✅ Real-time calculation
- ✅ Improved UX/animations
- ✅ Analytics integration
- ✅ A/B testing
- ✅ Performance optimization

**Timeline:** 1 week

---

## 14. QUESTIONS TO RESOLVE

### Before Implementation:
1. **Modal vs Slide-in**: Which UX pattern do we prefer?
2. **Pre-fill Strategy**: Should we pre-fill based on page context?
3. **Real-time Calculation**: Calculate on field change or button click?
4. **Advanced Fields**: Show by default or in collapsible section?
5. **Mobile Experience**: Full-screen modal or slide-in panel?
6. **Data Persistence**: URL params, sessionStorage, or Laravel session?
7. **Styling**: Match existing design system or create new styles?
8. **Error Messages**: Inline errors or toast notifications?
9. **Loading States**: Skeleton loaders or simple spinners?
10. **Accessibility**: Full WCAG compliance or basic accessibility?

---

## 15. OPEN QUESTIONS FOR FINETUNING

**Please Review and Provide Feedback On:**

1. **Field Selection**: Are the required fields appropriate, or should we simplify further?
2. **UX Flow**: Does the two-step (basic/advanced) approach work, or prefer single form?
3. **Results Display**: Should we show price range for anonymous users or specific price?
4. **CTA Buttons**: Should "Place Order" redirect immediately or show confirmation?
5. **Integration Points**: Where else should the calculator be accessible (footer, sidebar)?
6. **Pre-fill Logic**: Should we pre-fill paper type based on service page URL?
7. **Validation Timing**: Validate on blur, on change, or only on submit?
8. **Error Recovery**: How should users recover from validation errors?
9. **Mobile Optimization**: Any specific mobile UX considerations?
10. **Analytics**: Which metrics are most important to track?

---

## END OF WORKFLOW DOCUMENT

**Next Steps:**
1. Review this workflow document
2. Provide feedback on open questions
3. Finalize implementation approach
4. Create detailed technical specifications
5. Begin implementation

