# 📄 Schema Splitting Fix - Separate JSON-LD Scripts

## Issue
Homepage schemas were displaying as a single `@graph` JSON-LD script instead of separate `<script>` tags for each schema.

**Before:**
```html
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@graph": [
    { "@type": "Organization", ... },
    { "@type": "WebSite", ... },
    { "@type": "Service", ... },
    { "@type": "BreadcrumbList", ... }
  ]
}
</script>
```

**After:**
```html
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "Organization",
  ...
}
</script>

<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "WebSite",
  ...
}
</script>

<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "Service",
  ...
}
</script>

<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "BreadcrumbList",
  ...
}
</script>
```

---

## Root Cause
The database stores structured data using the **`@graph` format**, which is valid but combines multiple schemas into a single JSON-LD object.

The `StructuredData.jsx` component was rendering it as-is without splitting it into separate scripts.

---

## Solution Applied

### Updated: `StructuredData.jsx`

Added logic to detect `@graph` format and split it into separate `<script>` tags:

```javascript
{structuredData ? (
    (() => {
        // If it has @graph, split into separate scripts
        if (structuredData["@graph"] && Array.isArray(structuredData["@graph"])) {
            return structuredData["@graph"].map((schema, index) => (
                <script
                    key={`schema-${index}`}
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify({
                            "@context": structuredData["@context"] || "https://schema.org",
                            ...schema
                        }),
                    }}
                />
            ));
        }
        
        // Otherwise, render as single script (for non-graph schemas)
        return (
            <script
                type="application/ld+json"
                dangerouslySetInnerHTML={{
                    __html: JSON.stringify(structuredData),
                }}
            />
        );
    })()
) : (
    // Fallback to hardcoded schemas
    ...
)}
```

---

## How It Works

### 1. **Detect `@graph` Format**
```javascript
if (structuredData["@graph"] && Array.isArray(structuredData["@graph"]))
```
- Checks if structured data has a `@graph` property
- Checks if `@graph` is an array of schemas

### 2. **Map Each Schema to Separate Script**
```javascript
return structuredData["@graph"].map((schema, index) => (
    <script key={`schema-${index}`} type="application/ld+json">
        {
            "@context": "https://schema.org",
            ...schema  // Organization, WebSite, Service, etc.
        }
    </script>
))
```
- Iterates through each schema in the `@graph` array
- Creates a separate `<script>` tag for each
- Adds the `@context` to each individual schema

### 3. **Handles Non-Graph Format**
```javascript
// Otherwise, render as single script
return <script type="application/ld+json">...</script>
```
- If not `@graph` format, renders as a single script
- Supports both formats from database

---

## Result

### Homepage Now Shows:
✅ 4 separate `<script type="application/ld+json">` tags  
✅ Each schema has its own `@context`  
✅ Cleaner HTML structure  
✅ Better for debugging  
✅ Easier to validate individual schemas  

### The 4 Schemas:
1. **Organization** - Company info, contact, social media
2. **WebSite** - Website metadata
3. **Service** - Academic writing services with pricing
4. **BreadcrumbList** - Homepage breadcrumb

---

## Benefits

### For SEO:
- ✅ Google can parse schemas independently
- ✅ Each schema validates separately
- ✅ Clearer structure for crawlers

### For Developers:
- ✅ Easier to debug individual schemas
- ✅ Cleaner HTML source
- ✅ Better separation of concerns

### For Content Managers:
- ✅ Can still edit as single JSON in admin panel
- ✅ Database format unchanged
- ✅ Component handles the splitting

---

## Testing

### 1. **View Homepage Source**
```bash
Visit: / (homepage)
View Source: Ctrl+U
Search: "application/ld+json"
Expected: 4 separate script tags
```

### 2. **Validate Each Schema**
```bash
Visit: https://search.google.com/test/rich-results
Enter: Your homepage URL
Check: Should show 4 valid schemas
```

### 3. **Check Individual Schemas**
Each script should look like:
```html
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "Organization",  <!-- or WebSite, Service, BreadcrumbList -->
  "name": "Academic Scribe",
  ...
}
</script>
```

---

## Compatibility

### Database Format
✅ **Supports `@graph` format** (current database format)  
✅ **Supports single schema format** (for other pages)  
✅ **No database changes needed**  

### Admin Panel
✅ Still edit as single JSON in admin  
✅ Component handles splitting automatically  
✅ Can save in either format  

---

## Notes

### Why `@graph` Format Exists in Database
The `@graph` format is a valid JSON-LD pattern that:
- Groups related schemas together
- Shares a single `@context`
- Reduces JSON size in database
- Is recommended by Google for complex pages

### Why We Split It
While `@graph` is valid, separate scripts are:
- Easier to read in HTML source
- Easier to debug individual schemas
- Cleaner for validation tools
- More modular

### Best of Both Worlds
- **Database:** Stores compact `@graph` format
- **Frontend:** Renders separate scripts
- **Admin:** Edits as single JSON
- **Component:** Handles the transformation

---

## Files Modified

1. ✅ `resources/js/Components/StructuredData.jsx`
   - Added `@graph` detection
   - Added schema splitting logic
   - Maintains backward compatibility

---

## Quick Reference

### Check Current Homepage Schemas
```bash
# Visit homepage
curl http://localhost | grep "application/ld+json" -A 20

# Count schemas
curl http://localhost | grep -c "application/ld+json"
# Should return: 4
```

### Clear Cache After Changes
```bash
php artisan cache:forget page_content_homepage
```

### Rebuild Frontend
```bash
npm run build
```

---

## Summary

**Problem:** Schemas displayed as single `@graph` script  
**Cause:** Database uses `@graph` format, component rendered it as-is  
**Solution:** Detect `@graph` and split into separate scripts  
**Result:** 4 clean, separate JSON-LD scripts  
**Status:** ✅ FIXED  

**Now your homepage has beautiful, separated schema scripts!** 🎉

---

## What You'll See Now

Visit your homepage and view source. You'll see:

```html
<!-- Schema 1: Organization -->
<script type="application/ld+json">
{"@context":"https://schema.org","@type":"Organization","name":"Academic Scribe",...}
</script>

<!-- Schema 2: WebSite -->
<script type="application/ld+json">
{"@context":"https://schema.org","@type":"WebSite","name":"Academic Scribe",...}
</script>

<!-- Schema 3: Service -->
<script type="application/ld+json">
{"@context":"https://schema.org","@type":"Service","name":"Academic Writing Services",...}
</script>

<!-- Schema 4: BreadcrumbList -->
<script type="application/ld+json">
{"@context":"https://schema.org","@type":"BreadcrumbList","itemListElement":[...]}
</script>
```

**Clean, separate, and perfect!** ✨

