<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Conversation;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class AutoResolveConversations extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'messaging:auto-resolve {--dry-run : Show what would be resolved without actually resolving}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically resolve conversations that have been inactive for a specified period';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        
        // Check if auto-resolution is enabled
        if (!config('messaging.automation.auto_resolve_enabled')) {
            $this->info('Auto-resolution is disabled in configuration.');
            return 0;
        }

        $daysInactive = config('messaging.automation.auto_resolve_after_days');
        
        if (!$daysInactive) {
            $this->error('Auto-resolve days not configured.');
            return 1;
        }

        $cutoffDate = Carbon::now()->subDays($daysInactive);
        
        $this->info("Looking for active conversations inactive since: {$cutoffDate->format('Y-m-d H:i:s')}");

        // Find conversations that should be auto-resolved
        $conversations = Conversation::where('status', 'active')
            ->where(function ($query) use ($cutoffDate) {
                $query->where('last_activity_at', '<', $cutoffDate)
                      ->orWhere(function ($q) use ($cutoffDate) {
                          $q->whereNull('last_activity_at')
                            ->where('updated_at', '<', $cutoffDate);
                      });
            })
            ->with(['participants', 'lastMessage'])
            ->get();

        if ($conversations->isEmpty()) {
            $this->info('No conversations found for auto-resolution.');
            return 0;
        }

        $this->info("Found {$conversations->count()} conversation(s) for auto-resolution:");

        $resolvedCount = 0;
        $systemUser = $this->getSystemUser();

        foreach ($conversations as $conversation) {
            $lastActivity = $conversation->last_activity_at ?? $conversation->updated_at;
            $participantNames = $conversation->participants->pluck('name')->join(', ');
            
            $this->line("- ID: {$conversation->id} | Participants: {$participantNames} | Last Activity: {$lastActivity->format('Y-m-d H:i:s')}");

            if (!$isDryRun) {
                try {
                    if ($conversation->markAsResolved($systemUser, true)) {
                        $resolvedCount++;
                        
                        // Log the auto-resolution
                        Log::info('Conversation auto-resolved', [
                            'conversation_id' => $conversation->id,
                            'last_activity' => $lastActivity,
                            'participants' => $conversation->participants->pluck('id')->toArray()
                        ]);

                        // TODO: Send notification to participants about auto-resolution
                        // This will be implemented in the notification phase
                    }
                } catch (\Exception $e) {
                    $this->error("Failed to resolve conversation {$conversation->id}: {$e->getMessage()}");
                    Log::error('Auto-resolve failed', [
                        'conversation_id' => $conversation->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        }

        if ($isDryRun) {
            $this->info("DRY RUN: Would resolve {$conversations->count()} conversation(s).");
        } else {
            $this->info("Successfully auto-resolved {$resolvedCount} conversation(s).");
        }

        return 0;
    }

    /**
     * Get or create a system user for automated actions.
     */
    private function getSystemUser(): User
    {
        // Try to find an existing system/admin user
        $systemUser = User::where('user_type', 'super_admin')->first();
        
        if (!$systemUser) {
            $systemUser = User::where('user_type', 'admin')->first();
        }

        if (!$systemUser) {
            // If no admin exists, create a system user
            $systemUser = User::create([
                'name' => 'System',
                'email' => 'system@messaging.local',
                'user_type' => 'admin',
                'password' => bcrypt('system-auto-generated'),
                'email_verified_at' => now(),
            ]);
        }

        return $systemUser;
    }
}
