<?php

namespace App\Console\Commands;

use App\Models\BlogPost;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class BlogBackupCommand extends Command
{
    protected $signature = 'blog:backup {--ids=* : Specific post IDs to backup} {--all : Backup all posts}';
    
    protected $description = 'Backup blog posts content and structured data before AEO enhancements';

    public function handle()
    {
        $this->info('Starting blog post backup...');
        
        $query = BlogPost::query();
        
        if ($this->option('all')) {
            $posts = $query->get();
            $this->info('Backing up all blog posts');
        } elseif ($ids = $this->option('ids')) {
            $posts = $query->whereIn('id', $ids)->get();
            $this->info('Backing up posts: ' . implode(', ', $ids));
        } else {
            $this->error('Please specify --all or --ids=1,2,3');
            return 1;
        }

        if ($posts->isEmpty()) {
            $this->warn('No posts found to backup');
            return 1;
        }

        $timestamp = now()->format('Y-m-d_His');
        $backupData = [
            'timestamp' => $timestamp,
            'created_at' => now()->toDateTimeString(),
            'post_count' => $posts->count(),
            'posts' => []
        ];

        foreach ($posts as $post) {
            $backupData['posts'][] = [
                'id' => $post->id,
                'title' => $post->title,
                'slug' => $post->slug,
                'excerpt' => $post->excerpt,
                'content' => $post->content,
                'structured_data' => $post->structured_data,
                'meta_title' => $post->meta_title,
                'meta_description' => $post->meta_description,
                'meta_keywords' => $post->meta_keywords,
                'updated_at' => $post->updated_at
            ];
        }

        // Save to storage/backups/blog
        $filename = "blog_backup_{$timestamp}.json";
        $path = "backups/blog/{$filename}";
        
        Storage::disk('local')->put($path, json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        $fullPath = storage_path("app/{$path}");
        
        $this->info("✓ Backup completed successfully!");
        $this->info("✓ {$posts->count()} posts backed up");
        $this->info("✓ Backup saved to: {$fullPath}");
        $this->newLine();
        $this->comment("To restore this backup later, run:");
        $this->comment("php artisan blog:restore {$timestamp}");

        return 0;
    }
}
