<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class CleanOptimizedImages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'images:clean
                            {--directory=public/images : Directory to clean}
                            {--formats=webp,avif : Comma-separated list of formats to clean}
                            {--orphaned : Only remove orphaned optimized images (no original)}
                            {--dry-run : Show what would be cleaned without actually doing it}
                            {--older-than=30 : Remove optimized images older than X days}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up optimized images (WebP, AVIF) that are no longer needed';

    protected $stats = [
        'scanned' => 0,
        'cleaned' => 0,
        'space_freed' => 0,
    ];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🧹 Starting optimized images cleanup...');
        $this->newLine();

        $directory = $this->option('directory');
        $formats = explode(',', $this->option('formats'));
        $orphanedOnly = $this->option('orphaned');
        $dryRun = $this->option('dry-run');
        $olderThan = (int) $this->option('older-than');

        if ($dryRun) {
            $this->warn('🔍 DRY RUN MODE - No files will be actually deleted');
            $this->newLine();
        }

        $fullPath = public_path($directory);

        if (!File::exists($fullPath)) {
            $this->error("❌ Directory does not exist: {$fullPath}");
            return Command::FAILURE;
        }

        $this->cleanDirectory($fullPath, $formats, $orphanedOnly, $dryRun, $olderThan);
        $this->displayCleanupResults();

        return Command::SUCCESS;
    }

    /**
     * Clean directory of optimized images
     */
    protected function cleanDirectory(string $directory, array $formats, bool $orphanedOnly, bool $dryRun, int $olderThan)
    {
        $files = File::allFiles($directory);
        $cutoffDate = now()->subDays($olderThan);

        foreach ($files as $file) {
            $extension = strtolower($file->getExtension());

            if (!in_array($extension, $formats)) {
                continue;
            }

            $this->stats['scanned']++;
            $filePath = $file->getPathname();
            $shouldDelete = false;

            // Check if file is older than cutoff
            if ($olderThan > 0) {
                $fileTime = File::lastModified($filePath);
                if ($fileTime < $cutoffDate->timestamp) {
                    $shouldDelete = true;
                }
            }

            // Check if it's orphaned (no original file)
            if ($orphanedOnly) {
                $originalPath = $this->getOriginalPath($filePath);
                if (!File::exists($originalPath)) {
                    $shouldDelete = true;
                }
            }

            // If no specific criteria, delete all optimized images
            if (!$orphanedOnly && $olderThan === 0) {
                $shouldDelete = true;
            }

            if ($shouldDelete) {
                $fileSize = File::size($filePath);
                $relativePath = str_replace(public_path(), '', $filePath);

                if ($dryRun) {
                    $this->line("🔍 Would delete: {$relativePath} (" . $this->formatBytes($fileSize) . ")");
                } else {
                    File::delete($filePath);
                    $this->line("🗑️  Deleted: {$relativePath}");
                    $this->stats['cleaned']++;
                    $this->stats['space_freed'] += $fileSize;
                }
            }
        }
    }

    /**
     * Get the original file path for an optimized image
     */
    protected function getOriginalPath(string $optimizedPath): string
    {
        $pathInfo = pathinfo($optimizedPath);
        $directory = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];

        // Try common original extensions
        $originalExtensions = ['jpg', 'jpeg', 'png', 'gif'];

        foreach ($originalExtensions as $ext) {
            $originalPath = $directory . '/' . $filename . '.' . $ext;
            if (File::exists($originalPath)) {
                return $originalPath;
            }
        }

        return '';
    }

    /**
     * Display cleanup results
     */
    protected function displayCleanupResults()
    {
        $this->newLine();
        $this->info('📊 CLEANUP RESULTS');
        $this->info('==================');
        $this->line("🔍 Files scanned: {$this->stats['scanned']}");
        $this->line("🗑️  Files cleaned: {$this->stats['cleaned']}");
        $this->line("💾 Space freed: " . $this->formatBytes($this->stats['space_freed']));

        $this->newLine();

        if ($this->stats['cleaned'] > 0) {
            $this->info("🎉 Cleanup completed successfully!");
        } else {
            $this->info("ℹ️  No files needed cleaning.");
        }
    }

    /**
     * Format bytes to human readable format
     */
    protected function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= (1 << (10 * $pow));

        return round($bytes, 2) . ' ' . $units[$pow];
    }
}
