<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use App\Services\BlogFileProcessor;

class DebugBlogParser extends Command
{
    protected $signature = 'blog:debug-parser {upload_id}';
    protected $description = 'Debug blog file parser with detailed output';

    public function handle()
    {
        $uploadId = $this->argument('upload_id');
        $config = config('blog_uploads');
        $uploadDisk = Storage::disk($config['storage']['upload_disk']);

        $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;
        $files = $uploadDisk->files($uploadPath);

        if (empty($files)) {
            $this->error("No files found for upload ID: {$uploadId}");
            return 1;
        }

        foreach ($files as $file) {
            $this->info("Processing file: " . basename($file));
            $content = $uploadDisk->get($file);

            // Test the parser step by step
            $processor = new BlogFileProcessor();

            try {
                // Use reflection to access private methods for debugging
                $reflection = new \ReflectionClass($processor);

                // Test comment removal
                $removeComments = $reflection->getMethod('removeComments');
                $removeComments->setAccessible(true);
                $contentWithoutComments = $removeComments->invoke($processor, $content);

                // Test object extraction
                $extractObject = $reflection->getMethod('extractExportedObject');
                $extractObject->setAccessible(true);
                $extractedObject = $extractObject->invoke($processor, $contentWithoutComments);

                // Test tokenization
                $tokenize = $reflection->getMethod('tokenizeJavaScript');
                $tokenize->setAccessible(true);
                $tokens = $tokenize->invoke($processor, $extractedObject);

                $this->info("Total tokens: " . count($tokens));

                // Find tokens around "meta" and "content"
                $metaIndex = -1;
                $contentIndex = -1;

                for ($i = 0; $i < count($tokens); $i++) {
                    if ($tokens[$i]['value'] === 'meta') {
                        $metaIndex = $i;
                    }
                    if ($tokens[$i]['value'] === 'content') {
                        $contentIndex = $i;
                    }
                }

                $this->info("Meta field found at token index: " . ($metaIndex >= 0 ? $metaIndex : 'NOT FOUND'));
                $this->info("Content field found at token index: " . ($contentIndex >= 0 ? $contentIndex : 'NOT FOUND'));

                if ($metaIndex >= 0) {
                    $this->info("Tokens around 'meta' field:");
                    $start = max(0, $metaIndex - 5);
                    $end = min(count($tokens) - 1, $metaIndex + 50);

                    for ($i = $start; $i <= $end; $i++) {
                        $token = $tokens[$i];
                        $marker = ($i === $metaIndex) ? ' <-- META' : '';
                        $this->line("{$i}: {$token['type']} = " . json_encode($token['value']) . $marker);
                    }
                }

                if ($contentIndex >= 0) {
                    $this->info("\nTokens around 'content' field:");
                    $start = max(0, $contentIndex - 5);
                    $end = min(count($tokens) - 1, $contentIndex + 10);

                    for ($i = $start; $i <= $end; $i++) {
                        $token = $tokens[$i];
                        $marker = ($i === $contentIndex) ? ' <-- CONTENT' : '';
                        $this->line("{$i}: {$token['type']} = " . json_encode($token['value']) . $marker);
                    }
                }

                // Test parsing with detailed logging
                $this->info("\nAttempting to parse with detailed logging...");
                $parsed = $this->parseTokensWithDebug($tokens);

                $this->info("Final parsed keys: " . implode(', ', array_keys($parsed)));
            } catch (\Exception $e) {
                $this->error("Parsing failed: " . $e->getMessage());
                $this->error("Stack trace: " . $e->getTraceAsString());
            }
        }

        return 0;
    }

    private function parseTokensWithDebug(array $tokens): array
    {
        $result = [];
        $i = 0;

        // Skip opening brace
        if ($i < count($tokens) && $tokens[$i]['value'] === '{') {
            $i++;
        }

        while ($i < count($tokens)) {
            if ($i >= count($tokens)) {
                $this->info("Reached end of tokens at index: $i");
                break;
            }

            $token = $tokens[$i];
            $this->info("Processing token $i: {$token['type']} = " . json_encode($token['value']));

            // Skip closing brace
            if ($token['value'] === '}') {
                $this->info("Found closing brace, ending parsing");
                break;
            }

            // Skip commas
            if ($token['value'] === ',') {
                $this->info("Skipping comma");
                $i++;
                continue;
            }

            // Parse property
            if ($token['type'] === 'identifier' || $token['type'] === 'string') {
                $key = $token['value'];
                $this->info("Found property key: $key");
                $i++;

                // Skip colon
                if ($i < count($tokens) && $tokens[$i]['value'] === ':') {
                    $this->info("Found colon, parsing value...");
                    $i++;

                    // Parse value
                    if ($i < count($tokens)) {
                        $valueResult = $this->parseValueWithDebug($tokens, $i, $key);
                        $result[$key] = $valueResult['value'];
                        $i = $valueResult['nextIndex'];
                        $this->info("Parsed $key, next index: $i");
                    } else {
                        $this->info("No more tokens for value");
                        $i++;
                    }
                } else {
                    $this->info("No colon found, skipping");
                    $i++;
                }
            } else {
                $this->info("Not a property, skipping");
                $i++;
            }
        }

        return $result;
    }

    private function parseValueWithDebug(array $tokens, int $startIndex, string $key): array
    {
        if ($startIndex >= count($tokens)) {
            $this->info("No tokens left for value of $key");
            return [
                'value' => null,
                'nextIndex' => $startIndex + 1
            ];
        }

        $token = $tokens[$startIndex];
        $this->info("Parsing value for $key: {$token['type']} = " . json_encode($token['value']));

        switch ($token['type']) {
            case 'string':
                $value = $token['value'];
                if (strlen($value) > 100) {
                    $this->info("Long string value for $key (length: " . strlen($value) . ")");
                }
                return [
                    'value' => $value,
                    'nextIndex' => $startIndex + 1
                ];

            case 'number':
            case 'identifier':
                return [
                    'value' => $token['value'],
                    'nextIndex' => $startIndex + 1
                ];

            case 'symbol':
                if ($token['value'] === '{') {
                    $this->info("Parsing object for $key");
                    return $this->parseObjectWithDebug($tokens, $startIndex, $key);
                }
                if ($token['value'] === '[') {
                    $this->info("Parsing array for $key");
                    return $this->parseArrayWithDebug($tokens, $startIndex, $key);
                }
                break;
        }

        return [
            'value' => null,
            'nextIndex' => $startIndex + 1
        ];
    }

    private function parseObjectWithDebug(array $tokens, int $startIndex, string $key): array
    {
        $result = [];
        $i = $startIndex + 1; // Skip opening brace
        $braceLevel = 1; // Track nested braces

        $this->info("Starting object parsing for $key at index $i");

        while ($i < count($tokens) && $braceLevel > 0) {
            if ($i >= count($tokens)) {
                $this->info("Ran out of tokens while parsing object for $key");
                break;
            }

            $token = $tokens[$i];

            if ($token['value'] === '{') {
                $braceLevel++;
                $this->info("Found opening brace, level: $braceLevel");
            } elseif ($token['value'] === '}') {
                $braceLevel--;
                $this->info("Found closing brace, level: $braceLevel");
                if ($braceLevel === 0) {
                    $i++;
                    break;
                }
            }

            // Only parse a few properties to avoid infinite output
            if ($braceLevel === 1 && count($result) < 5) {
                if ($token['value'] === ',') {
                    $i++;
                    continue;
                }

                if ($token['type'] === 'identifier' || $token['type'] === 'string') {
                    $objKey = $token['value'];
                    $this->info("Found object property: $objKey");
                    $i++;

                    // Skip colon
                    if ($i < count($tokens) && $tokens[$i]['value'] === ':') {
                        $i++;

                        if ($i < count($tokens)) {
                            // For complex objects, just store placeholder to avoid infinite recursion
                            $result[$objKey] = "PLACEHOLDER";
                            $i++;
                        } else {
                            $i++;
                        }
                    } else {
                        $i++;
                    }
                } else {
                    $i++;
                }
            } else {
                $i++;
            }
        }

        $this->info("Finished parsing object for $key at index $i");

        return [
            'value' => $result,
            'nextIndex' => $i
        ];
    }

    private function parseArrayWithDebug(array $tokens, int $startIndex, string $key): array
    {
        $result = [];
        $i = $startIndex + 1; // Skip opening bracket
        $bracketLevel = 1; // Track nested brackets

        $this->info("Starting array parsing for $key at index $i");

        while ($i < count($tokens) && $bracketLevel > 0) {
            if ($i >= count($tokens)) {
                $this->info("Ran out of tokens while parsing array for $key");
                break;
            }

            $token = $tokens[$i];

            if ($token['value'] === '[') {
                $bracketLevel++;
            } elseif ($token['value'] === ']') {
                $bracketLevel--;
                if ($bracketLevel === 0) {
                    $i++;
                    break;
                }
            }

            // Only parse a few items to avoid infinite output
            if ($bracketLevel === 1 && count($result) < 5) {
                if ($token['value'] === ',') {
                    $i++;
                    continue;
                }

                $result[] = "ARRAY_ITEM";
                $i++;
            } else {
                $i++;
            }
        }

        $this->info("Finished parsing array for $key at index $i");

        return [
            'value' => $result,
            'nextIndex' => $i
        ];
    }
}
