<?php

namespace App\Console\Commands;

use App\Models\WriterProfile;
use App\Services\NicknameGeneratorService;
use Illuminate\Console\Command;

class GenerateWriterNicknames extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'writers:generate-nicknames {--dry-run : Show what would be updated without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate nicknames for existing writers who do not have them';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        // Get writers without nicknames
        $writersWithoutNicknames = WriterProfile::whereNull('nickname')
            ->orWhere('nickname', '')
            ->with('user')
            ->get();

        if ($writersWithoutNicknames->isEmpty()) {
            $this->info('All writers already have nicknames!');
            return 0;
        }

        $this->info("Found {$writersWithoutNicknames->count()} writers without nicknames.");

        if ($dryRun) {
            $this->warn('DRY RUN MODE - No changes will be made');
        }

        $bar = $this->output->createProgressBar($writersWithoutNicknames->count());
        $bar->start();

        foreach ($writersWithoutNicknames as $writerProfile) {
            if (!$writerProfile->user) {
                $this->warn("Writer profile {$writerProfile->id} has no associated user, skipping...");
                continue;
            }

            $nickname = NicknameGeneratorService::generateNickname(
                $writerProfile->user->id,
                $writerProfile->user->name
            );

            if ($dryRun) {
                $this->line("\nWould generate nickname '{$nickname}' for writer {$writerProfile->user->name} (ID: {$writerProfile->user->id})");
            } else {
                $writerProfile->nickname = $nickname;
                $writerProfile->save();
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine();

        if ($dryRun) {
            $this->info('Dry run completed. Use --dry-run=false to apply changes.');
        } else {
            $this->info('Successfully generated nicknames for all writers!');
        }

        return 0;
    }
} 