<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use App\Services\ImageOptimizationService;
use App\Helpers\ImageHelper;

class OptimizeImages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'images:optimize
                            {--path=* : Specific image paths to optimize}
                            {--directory=images : Directory to scan for images (relative to public)}
                            {--type=blog : Default conversion type (hero,blog,service,thumbnail,avatar,logo)}
                            {--formats=webp,avif : Comma-separated list of formats to generate}
                            {--force : Force re-optimization of existing images}
                            {--dry-run : Show what would be optimized without actually doing it}
                            {--collection=images : Collection name for organized storage}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize existing images by converting them to modern formats (WebP, AVIF) with responsive sizes';

    protected $imageService;
    protected $supportedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff'];
    protected $stats = [
        'processed' => 0,
        'optimized' => 0,
        'skipped' => 0,
        'errors' => 0,
        'space_saved' => 0,
    ];

    public function __construct(ImageOptimizationService $imageService)
    {
        parent::__construct();
        $this->imageService = $imageService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Starting image optimization process...');
        $this->newLine();

        $paths = $this->option('path');
        $directory = $this->option('directory');
        $type = $this->option('type');
        $formats = explode(',', $this->option('formats'));
        $force = $this->option('force');
        $dryRun = $this->option('dry-run');
        $collection = $this->option('collection');

        if ($dryRun) {
            $this->warn('🔍 DRY RUN MODE - No images will be actually optimized');
            $this->newLine();
        }

        // Process specific paths if provided
        if (!empty($paths)) {
            $this->processSpecificPaths($paths, $type, $formats, $force, $dryRun, $collection);
        } else {
            // Process entire directory
            $this->processDirectory($directory, $type, $formats, $force, $dryRun, $collection);
        }

        $this->displayResults();
        return Command::SUCCESS;
    }

    /**
     * Process specific image paths
     */
    protected function processSpecificPaths(array $paths, string $type, array $formats, bool $force, bool $dryRun, string $collection)
    {
        $this->info("📁 Processing " . count($paths) . " specific image(s)...");
        $this->newLine();

        foreach ($paths as $path) {
            $this->processImage($path, $type, $formats, $force, $dryRun, $collection);
        }
    }

    /**
     * Process entire directory
     */
    protected function processDirectory(string $directory, string $type, array $formats, bool $force, bool $dryRun, string $collection)
    {
        // Fix: Build the correct path - directory is relative to public
        $fullPath = public_path($directory);

        if (!File::exists($fullPath)) {
            $this->error("❌ Directory does not exist: {$fullPath}");
            $this->line("💡 Make sure the directory exists. Current directory should be: " . $directory);
            $this->line("💡 Try creating it with: mkdir " . $fullPath);
            return;
        }

        $this->info("📁 Scanning directory: {$fullPath}");

        $images = $this->findImages($fullPath);

        if (empty($images)) {
            $this->warn("⚠️  No images found in directory: {$fullPath}");
            $this->line("💡 Supported formats: " . implode(', ', $this->supportedExtensions));
            return;
        }

        $this->info("🖼️  Found " . count($images) . " image(s) to process");
        $this->newLine();

        $progressBar = $this->output->createProgressBar(count($images));
        $progressBar->setFormat('verbose');

        foreach ($images as $imagePath) {
            // Fix: Get relative path correctly
            $relativePath = '/' . str_replace(public_path() . DIRECTORY_SEPARATOR, '', $imagePath);
            $relativePath = str_replace('\\', '/', $relativePath); // Normalize path separators

            $this->processImage($relativePath, $type, $formats, $force, $dryRun, $collection);
            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine(2);
    }

    /**
     * Process a single image
     */
    protected function processImage(string $imagePath, string $type, array $formats, bool $force, bool $dryRun, string $collection)
    {
        $this->stats['processed']++;

        // Clean up path - ensure it starts with /
        $imagePath = '/' . ltrim($imagePath, '/');
        $fullPath = public_path($imagePath);

        if (!File::exists($fullPath)) {
            $this->error("❌ Image not found: {$fullPath}");
            $this->stats['errors']++;
            return;
        }

        // Check if already optimized (unless force is used)
        if (!$force && $this->isAlreadyOptimized($imagePath)) {
            $this->line("⏭️  Skipping already optimized: {$imagePath}");
            $this->stats['skipped']++;
            return;
        }

        if ($dryRun) {
            $this->line("🔍 Would optimize: {$imagePath} (type: {$type})");
            return;
        }

        try {
            $originalSize = File::size($fullPath);

            // Convert the image
            $optimizedUrls = $this->imageService->convertStaticImage($imagePath, $type, $collection);

            // Calculate space saved
            $optimizedSize = $this->calculateOptimizedSize($optimizedUrls);
            $spaceSaved = $originalSize - $optimizedSize;
            $this->stats['space_saved'] += max(0, $spaceSaved);

            $this->info("✅ Optimized: {$imagePath}");
            $this->line("   📊 Original: " . $this->formatBytes($originalSize));
            $this->line("   📊 Optimized: " . $this->formatBytes($optimizedSize));
            $this->line("   💾 Saved: " . $this->formatBytes($spaceSaved));

            $this->stats['optimized']++;
        } catch (\Exception $e) {
            $this->error("❌ Failed to optimize {$imagePath}: " . $e->getMessage());
            $this->stats['errors']++;
        }
    }

    /**
     * Find all images in directory recursively
     */
    protected function findImages(string $directory): array
    {
        $images = [];

        if (!File::exists($directory)) {
            return $images;
        }

        $files = File::allFiles($directory);

        foreach ($files as $file) {
            $extension = strtolower($file->getExtension());
            if (in_array($extension, $this->supportedExtensions)) {
                $images[] = $file->getPathname();
            }
        }

        return $images;
    }

    /**
     * Check if image is already optimized
     */
    protected function isAlreadyOptimized(string $imagePath): bool
    {
        $pathInfo = pathinfo($imagePath);
        $baseName = $pathInfo['filename'];
        $directory = $pathInfo['dirname'];

        // Check if WebP version exists
        $webpPath = public_path($directory . '/' . $baseName . '.webp');

        return File::exists($webpPath);
    }

    /**
     * Calculate total size of optimized images
     */
    protected function calculateOptimizedSize(array $optimizedUrls): int
    {
        $totalSize = 0;

        foreach ($optimizedUrls as $url) {
            $path = public_path(parse_url($url, PHP_URL_PATH));
            if (File::exists($path)) {
                $totalSize += File::size($path);
            }
        }

        return $totalSize;
    }

    /**
     * Display optimization results
     */
    protected function displayResults()
    {
        $this->newLine();
        $this->info('📊 OPTIMIZATION RESULTS');
        $this->info('========================');
        $this->line("📈 Images processed: {$this->stats['processed']}");
        $this->line("✅ Images optimized: {$this->stats['optimized']}");
        $this->line("⏭️  Images skipped: {$this->stats['skipped']}");
        $this->line("❌ Errors: {$this->stats['errors']}");
        $this->line("💾 Total space saved: " . $this->formatBytes($this->stats['space_saved']));

        if ($this->stats['optimized'] > 0) {
            $averageSavings = $this->stats['space_saved'] / $this->stats['optimized'];
            $this->line("📊 Average savings per image: " . $this->formatBytes($averageSavings));
        }

        $this->newLine();

        if ($this->stats['errors'] > 0) {
            $this->warn("⚠️  Some images failed to optimize. Check the error messages above.");
        } else {
            $this->info("🎉 All images processed successfully!");
        }
    }

    /**
     * Format bytes to human readable format
     */
    protected function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= (1 << (10 * $pow));

        return round($bytes, 2) . ' ' . $units[$pow];
    }
}
