<?php

namespace App\Console\Commands;

use App\Services\UrgentOrderNotificationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SendUrgentOrderNotifications extends Command
{
    protected $signature = 'orders:send-urgent-notifications';
    protected $description = 'Send notifications for urgent orders to admins and writers';

    protected UrgentOrderNotificationService $urgentOrderService;

    public function __construct(UrgentOrderNotificationService $urgentOrderService)
    {
        parent::__construct();
        $this->urgentOrderService = $urgentOrderService;
    }

    public function handle(): int
    {
        try {
            $this->info('Starting urgent order notification process...');
            
            $urgentOrders = $this->urgentOrderService->getUrgentOrders();
            $totalOrders = collect($urgentOrders)->flatten(1)->count();
            
            if ($totalOrders === 0) {
                $this->info('No urgent orders found.');
                return 0;
            }

            $this->info("Found {$totalOrders} urgent orders:");
            $this->info("- Critical (12h): " . count($urgentOrders['critical_12h'] ?? []));
            $this->info("- Urgent (24h): " . count($urgentOrders['urgent_24h'] ?? []));
            $this->info("- Warning (48h): " . count($urgentOrders['warning_48h'] ?? []));

            $this->urgentOrderService->sendUrgentOrderNotifications();
            
            $this->info('Urgent order notifications sent successfully!');
            
            return 0;
        } catch (\Exception $e) {
            $this->error('Failed to send urgent order notifications: ' . $e->getMessage());
            Log::error('Urgent order notification command failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return 1;
        }
    }
} 