<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BlogPost;
use App\Models\BlogCategory;
use App\Models\BlogAuthor;
use App\Models\BlogTag;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class BlogController extends Controller
{
    /**
     * Show blog dashboard
     */
    public function dashboard(): Response
    {
        $stats = [
            'total_posts' => BlogPost::count(),
            'published_posts' => BlogPost::where('status', 'published')->count(),
            'draft_posts' => BlogPost::where('status', 'draft')->count(),
            'total_categories' => BlogCategory::count(),
            'total_authors' => BlogAuthor::count(),
            'total_tags' => BlogTag::count(),
        ];

        $recent_posts = BlogPost::with(['author', 'category'])
            ->latest()
            ->limit(10)
            ->get();

        return Inertia::render('Admin/Blog/Dashboard', [
            'stats' => $stats,
            'recent_posts' => $recent_posts,
        ]);
    }

    /**
     * Display blog posts management page
     */
    public function posts(Request $request): Response
    {
        // Get sort parameters
        $sortBy = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');

        // Validate sort field
        $allowedSortFields = ['created_at', 'title', 'published_at', 'status'];
        if (!in_array($sortBy, $allowedSortFields)) {
            $sortBy = 'created_at';
        }

        // Validate sort direction
        $sortDirection = in_array($sortDirection, ['asc', 'desc']) ? $sortDirection : 'desc';

        // Build the query
        $posts = BlogPost::with(['author', 'category', 'tags'])
            ->when($request->search, function ($query, $search) {
                $query->where('title', 'like', "%{$search}%")
                    ->orWhere('excerpt', 'like', "%{$search}%");
            })
            ->when($request->status, function ($query, $status) {
                if ($status !== 'all') {
                    $query->where('status', $status);
                }
            })
            ->when($request->category, function ($query, $category) {
                if ($category !== 'all') {
                    $query->whereHas('category', function ($q) use ($category) {
                        $q->where('slug', $category);
                    });
                }
            })
            ->when($request->author, function ($query, $author) {
                if ($author !== 'all') {
                    $query->where('author_id', $author);
                }
            })
            ->when($request->difficulty, function ($query, $difficulty) {
                if ($difficulty !== 'all') {
                    $query->where('difficulty', $difficulty);
                }
            })
            ->orderBy($sortBy, $sortDirection)
            ->paginate(15);

        // Get categories and authors
        $categories = BlogCategory::orderBy('name')->get();
        $authors = BlogAuthor::orderBy('name')->get();

        // Calculate stats
        $stats = [
            'total' => BlogPost::count(),
            'published' => BlogPost::where('status', 'published')->count(),
            'draft' => BlogPost::where('status', 'draft')->count(),
            'archived' => BlogPost::where('status', 'archived')->count(),
        ];

        // Prepare filters object with all expected properties
        $filters = [
            'search' => $request->get('search', ''),
            'status' => $request->get('status', 'all'),
            'category' => $request->get('category', 'all'),
            'author' => $request->get('author', 'all'),
            'difficulty' => $request->get('difficulty', 'all'),
            'sort' => $sortBy,
            'direction' => $sortDirection,
        ];

        return Inertia::render('Admin/Blog/Posts', [
            'posts' => $posts,
            'categories' => $categories,
            'authors' => $authors,
            'filters' => $filters,
            'stats' => $stats,
        ]);
    }

    /**
     * Show create post form
     */
    public function createPost(): Response
    {
        $categories = BlogCategory::orderBy('name')->get();
        $authors = BlogAuthor::orderBy('name')->get();
        $tags = BlogTag::orderBy('name')->get();

        return Inertia::render('Admin/Blog/CreatePost', [
            'categories' => $categories,
            'authors' => $authors,
            'tags' => $tags,
        ]);
    }

    /**
     * Show edit post form
     */
    public function editPost($id): Response
    {
        $blogPost = BlogPost::findOrFail($id);
        $blogPost->load(['author', 'category', 'tags']);

        $categories = BlogCategory::orderBy('name')->get();
        $authors = BlogAuthor::orderBy('name')->get();
        $tags = BlogTag::orderBy('name')->get();

        return Inertia::render('Admin/Blog/EditPost', [
            'post' => $blogPost,
            'categories' => $categories,
            'authors' => $authors,
            'tags' => $tags,
        ]);
    }

    /**
     * Update blog post
     */
    public function updatePost(Request $request, $id)
    {
        $blogPost = BlogPost::findOrFail($id);

        \Log::info('=== BLOG POST UPDATE START ===', [
            'post_id' => $id,
            'post_title' => $blogPost->title,
        ]);

        \Log::info('RAW REQUEST DATA', [
            'all_input_keys' => array_keys($request->all()),
            'has_structured_data_in_request' => $request->has('structured_data'),
            'raw_title' => $request->input('title'),
            'raw_structured_data' => $request->input('structured_data'),
        ]);

        \Log::info('Structured data received', [
            'structured_data' => $request->input('structured_data'),
            'type' => gettype($request->input('structured_data')),
            'is_array' => is_array($request->input('structured_data')),
            'is_empty' => empty($request->input('structured_data')),
        ]);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug,' . $blogPost->id,
            'subtitle' => 'nullable|string',
            'excerpt' => 'required|string',
            'content' => 'required|string',
            'featured_image' => 'nullable|string',
            'featured_image_alt' => 'nullable|string',
            'category_id' => 'required|exists:blog_categories,id',
            'author_id' => 'required|exists:blog_authors,id',
            'status' => 'required|in:draft,published,archived',
            'published_at' => 'nullable|date',
            'read_time' => 'nullable|integer|min:1|max:60',
            'difficulty' => 'nullable|in:beginner,intermediate,advanced',
            'featured' => 'nullable|boolean',
            'enable_comments' => 'nullable|boolean',
            'enable_ratings' => 'nullable|boolean',
            'meta_data' => 'nullable|array',
            'structured_data' => 'nullable|array',
            'social_sharing' => 'nullable|array',
            'analytics' => 'nullable|array',
            'table_of_contents' => 'nullable|array',
            'tag_ids' => 'nullable|array',
            'tag_ids.*' => 'exists:blog_tags,id',
        ]);

        // Handle tag relationships
        $tagIds = $validated['tag_ids'] ?? [];
        unset($validated['tag_ids']);

        \Log::info('Validated data received', [
            'has_structured_data' => isset($validated['structured_data']),
            'structured_data' => $validated['structured_data'] ?? null,
            'structured_data_type' => isset($validated['structured_data']) ? gettype($validated['structured_data']) : 'not set',
        ]);

        // Preserve original published_at if post was already published
        // Only set published_at to now() if this is a first-time publish
        $isFirstTimePublish = ($validated['status'] === 'published' && !$blogPost->published_at);
        
        if ($validated['status'] === 'published' && $isFirstTimePublish) {
            // First time publishing - set to now()
            $validated['published_at'] = now();
            \Log::info('First time publish - setting published_at to now', [
                'new_published_at' => $validated['published_at'],
            ]);
        } elseif ($validated['status'] === 'published' && $blogPost->published_at) {
            // Already published - preserve original published_at
            $validated['published_at'] = $blogPost->published_at;
            \Log::info('Already published - preserving original published_at', [
                'original_published_at' => $blogPost->published_at,
            ]);
        } elseif ($validated['status'] === 'published' && !empty($validated['published_at'])) {
            // User has set a published_at date, but check for timezone issues
            $publishedTime = strtotime($validated['published_at']);
            $futureThreshold = now()->addMinutes(5)->timestamp;
            
            if ($publishedTime > $futureThreshold) {
                // Published_at is more than 5 minutes in the future - likely timezone bug
                // Set it to now()
                $validated['published_at'] = now();
                \Log::warning('Published_at was in the future - setting to now due to timezone bug', [
                    'original' => $validated['published_at'],
                    'new' => $validated['published_at'],
                ]);
            }
        }

        // Log data being sent to database
        \Log::info('About to update database', [
            'structured_data_in_update' => $validated['structured_data'] ?? 'NOT SET',
            'all_validated_keys' => array_keys($validated),
        ]);

        // Update the blog post
        $blogPost->update($validated);

        // Sync tags
        $blogPost->tags()->sync($tagIds);

        // Log after update - fetch fresh from DB
        $blogPost->refresh();
        \Log::info('Blog post updated successfully', [
            'post_id' => $blogPost->id,
            'slug' => $blogPost->slug,
            'status' => $blogPost->status,
            'structured_data_after_update' => $blogPost->structured_data,
            'structured_data_type' => gettype($blogPost->structured_data),
            'structured_data_empty' => empty($blogPost->structured_data),
        ]);

        \Log::info('=== BLOG POST UPDATE COMPLETE ===');

        // Redirect back to edit page instead of posts list
        return redirect()->route('admin.blog.posts.edit', $blogPost->id)
            ->with('message', 'Post updated successfully!');
    }

    /**
     * Delete blog post
     */
    public function deletePost($id)
    {
        $blogPost = BlogPost::findOrFail($id);
        $blogPost->delete();

        return redirect()->route('admin.blog.posts')
            ->with('success', 'Post deleted successfully!');
    }

    /**
     * Show blog categories management
     */
    public function categories(): Response
    {
        $categories = BlogCategory::withCount('posts')->get();

        return Inertia::render('Admin/Blog/Categories', [
            'categories' => $categories,
        ]);
    }

    /**
     * Show blog authors management
     */
    public function authors(): Response
    {
        $authors = BlogAuthor::withCount('posts')->get();

        return Inertia::render('Admin/Blog/Authors', [
            'authors' => $authors,
        ]);
    }

    /**
     * Show blog tags management
     */
    public function tags(): Response
    {
        $tags = BlogTag::withCount('posts')->get();

        return Inertia::render('Admin/Blog/Tags', [
            'tags' => $tags,
        ]);
    }

    /**
     * Show blog upload interface
     */
    public function upload(): Response
    {
        return Inertia::render('Admin/Blog/Upload');
    }

    /**
     * Store new blog post
     */
    public function storePost(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug',
            'excerpt' => 'required|string',
            'content' => 'required|string',
            'category_id' => 'required|exists:blog_categories,id',
            'author_id' => 'required|exists:blog_authors,id',
            'status' => 'required|in:draft,published,archived',
            'published_at' => 'nullable|date',
            'featured_image' => 'nullable|string',
            'meta_data' => 'nullable|array',
        ]);

        // Handle timezone for new posts
        if ($validated['status'] === 'published') {
            if (!empty($validated['published_at'])) {
                // Check if the provided published_at is more than 5 minutes in the future
                $publishedTime = strtotime($validated['published_at']);
                $futureThreshold = now()->addMinutes(5)->timestamp;
                
                if ($publishedTime > $futureThreshold) {
                    // Likely a timezone bug - set to now
                    $validated['published_at'] = now();
                    \Log::warning('New post published_at was in future - setting to now', [
                        'original' => $request->published_at,
                        'corrected' => $validated['published_at'],
                    ]);
                }
            } else {
                // No published_at provided but status is published - set to now
                $validated['published_at'] = now();
            }
        }

        $post = BlogPost::create($validated);

        return redirect()->route('admin.blog.posts')
            ->with('success', 'Post created successfully!');
    }


    /**
     * Store new category
     */
    public function storeCategory(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_categories,slug',
            'description' => 'nullable|string',
            'color' => 'nullable|string',
            'icon' => 'nullable|string',
        ]);

        BlogCategory::create($validated);

        return redirect()->route('admin.blog.categories')
            ->with('success', 'Category created successfully!');
    }

    /**
     * Update category
     */
    public function updateCategory(Request $request, BlogCategory $category)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_categories,slug,' . $category->id,
            'description' => 'nullable|string',
            'color' => 'nullable|string',
            'icon' => 'nullable|string',
        ]);

        $category->update($validated);

        return redirect()->route('admin.blog.categories')
            ->with('success', 'Category updated successfully!');
    }

    /**
     * Delete category
     */
    public function deleteCategory(BlogCategory $category)
    {
        $category->delete();

        return redirect()->route('admin.blog.categories')
            ->with('success', 'Category deleted successfully!');
    }

    /**
     * Store new author
     */
    public function storeAuthor(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:blog_authors,email',
            'bio' => 'nullable|string',
            'avatar' => 'nullable|string',
            'social_links' => 'nullable|array',
        ]);

        BlogAuthor::create($validated);

        return redirect()->route('admin.blog.authors')
            ->with('success', 'Author created successfully!');
    }

    /**
     * Update author
     */
    public function updateAuthor(Request $request, BlogAuthor $author)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:blog_authors,email,' . $author->id,
            'bio' => 'nullable|string',
            'avatar' => 'nullable|string',
            'social_links' => 'nullable|array',
        ]);

        $author->update($validated);

        return redirect()->route('admin.blog.authors')
            ->with('success', 'Author updated successfully!');
    }

    /**
     * Delete author
     */
    public function deleteAuthor(BlogAuthor $author)
    {
        $author->delete();

        return redirect()->route('admin.blog.authors')
            ->with('success', 'Author deleted successfully!');
    }

    /**
     * Store new tag
     */
    public function storeTag(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_tags,slug',
        ]);

        BlogTag::create($validated);

        return redirect()->route('admin.blog.tags')
            ->with('success', 'Tag created successfully!');
    }

    /**
     * Update tag
     */
    public function updateTag(Request $request, BlogTag $tag)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_tags,slug,' . $tag->id,
        ]);

        $tag->update($validated);

        return redirect()->route('admin.blog.tags')
            ->with('success', 'Tag updated successfully!');
    }

    /**
     * Delete tag
     */
    public function deleteTag(BlogTag $tag)
    {
        $tag->delete();

        return redirect()->route('admin.blog.tags')
            ->with('success', 'Tag deleted successfully!');
    }

    /**
     * Handle bulk actions
     */
    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:publish,unpublish,delete,archive',
            'post_ids' => 'required|array',
            'post_ids.*' => 'exists:blog_posts,id',
        ]);

        $posts = BlogPost::whereIn('id', $validated['post_ids']);

        switch ($validated['action']) {
            case 'publish':
                // Handle bulk publish properly - only set published_at for posts that don't have it
                $postsToUpdate = BlogPost::whereIn('id', $validated['post_ids'])->get();
                foreach ($postsToUpdate as $post) {
                    if (!$post->published_at) {
                        // First time publishing - set published_at
                        $post->update([
                            'status' => 'published',
                            'published_at' => now(),
                        ]);
                    } else {
                        // Already has published_at - preserve it
                        $post->update([
                            'status' => 'published',
                        ]);
                    }
                }
                break;
            case 'unpublish':
                $posts->update(['status' => 'draft']);
                break;
            case 'archive':
                $posts->update(['status' => 'archived']);
                break;
            case 'delete':
                $posts->delete();
                break;
        }

        return redirect()->route('admin.blog.posts')
            ->with('success', 'Bulk action completed successfully!');
    }

    /**
     * Upload featured image for blog posts
     */
    public function uploadFeaturedImage(Request $request)
    {
        $request->validate([
            'featured_image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120', // 5MB max
        ]);

        try {
            $file = $request->file('featured_image');
            $filename = time() . '_' . $file->getClientOriginalName();
            
            // Store the image directly in public/images/blog/
            $file->move(public_path('images/blog'), $filename);
            
            // Generate the URL in the expected format
            $imageUrl = '/images/blog/' . $filename;

            return response()->json([
                'success' => true,
                'message' => 'Featured image uploaded successfully',
                'image_url' => $imageUrl,
                'image_path' => 'images/blog/' . $filename
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload SEO/Open Graph image for blog posts
     */
    public function uploadSeoImage(Request $request)
    {
        $request->validate([
            'og_image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120', // 5MB max
        ]);

        try {
            $file = $request->file('og_image');
            $filename = 'og_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $file->getClientOriginalName());
            
            // Store in public/images/seo/ directory
            $directory = public_path('images/seo');
            if (!file_exists($directory)) {
                mkdir($directory, 0755, true);
            }
            
            $file->move($directory, $filename);
            
            // Generate the URL
            $imageUrl = '/images/seo/' . $filename;

            return response()->json([
                'success' => true,
                'message' => 'SEO image uploaded successfully',
                'image_url' => $imageUrl,
                'image_path' => 'images/seo/' . $filename
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload image: ' . $e->getMessage()
            ], 500);
        }
    }
}
