<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\User;
use Inertia\Inertia;
use App\Models\EssayTopic;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\WriterProfile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Notifications\AdminNotifiable;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Notification;
use App\Notifications\EssayReviewNotification;
use App\Notifications\AdminEssayReviewNotification;

class EssayReviewController extends Controller
{
    /**
     * Display a listing of the essays.
     */
    public function index(Request $request)
    {
        // Get search and filter parameters
        $search = $request->input('search', '');
        $status = $request->input('status', 'all');

        // Build the query using WriterProfile
        $query = WriterProfile::with(['user:id,name,email', 'essayTest'])
            ->join('users', 'writer_profiles.user_id', '=', 'users.id')
            ->join('writer_essay_tests', 'writer_profiles.id', '=', 'writer_essay_tests.writer_profile_id')
            ->select('writer_profiles.*')
            ->whereNotNull('writer_essay_tests.file_path')
            ->where('writer_essay_tests.file_path', '!=', '');

        // Apply search filter if provided
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('users.name', 'LIKE', "%{$search}%")
                    ->orWhere('users.email', 'LIKE', "%{$search}%");
            });
        }

        // Apply status filter if not 'all'
        if ($status !== 'all') {
            // Map status to essay_test_status
            $statusMap = [
                'pending' => 'essay_submitted',
                'approved' => 'passed',
                'rejected' => 'essay_failed'
            ];

            if (isset($statusMap[$status])) {
                $query->where('writer_essay_tests.essay_test_status', $statusMap[$status]);
            }
        }

        // Get paginated results
        $profiles = $query->orderBy('writer_essay_tests.updated_at', 'desc')
            ->paginate(10)
            ->withQueryString();

        // Transform data to match expected format
        $essays = $profiles->through(function ($profile) {
            $essayTest = $profile->essayTest;

            return [
                'id' => $profile->id,
                'writer' => [
                    'name' => $profile->user->name,
                    'email' => $profile->user->email,
                ],
                'submitted_at' => $essayTest ? $essayTest->essay_completed_at : null,
                'status' => $essayTest ? $this->mapStatusForFrontend($essayTest->essay_test_status) : 'pending',
                'topic' => [
                    'id' => null,
                    'title' => 'Writer Application Essay',
                ],
                'file_path' => $essayTest ? $essayTest->file_path : null,
            ];
        });

        return Inertia::render('Admin/Essays/Index', [
            'essays' => $essays,
            'filters' => [
                'search' => $search,
                'status' => $status,
            ],
        ]);
    }

    /**
     * Map internal status to frontend-friendly status
     */
    private function mapStatusForFrontend($status)
    {
        switch ($status) {
            case 'essay_submitted':
                return 'pending';
            case 'passed':
                return 'approved';
            case 'essay_failed':
                return 'rejected';
            default:
                return 'pending';
        }
    }

    /**
     * Map profile status to essay status
     */
    private function mapProfileStatusToEssayStatus($status)
    {
        return $this->mapStatusForFrontend($status);
    }

    /**
     * Display the specified essay.
     */
    public function show($id)
    {
        // Find the writer profile by ID
        $profile = WriterProfile::with(['user', 'essayTest', 'qualificationTest'])->findOrFail($id);
        $essayTest = $profile->essayTest;

        if (!$essayTest) {
            return redirect()->route('admin.essays.index')
                ->with('error', 'Essay test not found for this writer.');
        }

        // Get file information
        $fileContent = "";
        $fileType = $essayTest->file_type ?? '';
        $filePath = $essayTest->file_path ?? '';
        $fileExists = !empty($filePath) && Storage::exists($filePath);

        // For PDF files, we can't display content directly
        $isPdf = $fileType === 'application/pdf';
        $fileUrl = $fileExists ? route('admin.essays.download', $id) : null;

        // For DOCX files, we might need a preview service or just offer download
        $isDocx = $fileType === 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';

        // Transform the data to match what Show.jsx expects
        return Inertia::render('Admin/Essays/Show', [
            'essay' => [
                'id' => $profile->id,
                'content' => $isPdf || $isDocx ? 'This file type cannot be displayed directly. Please download the file to view its contents.' : $fileContent,
                'status' => $this->mapStatusForFrontend($essayTest->essay_test_status),
                'feedback' => $essayTest->essay_feedback,
                'submitted_at' => $essayTest->essay_completed_at,
                'reviewed_at' => $essayTest->updated_at,
                'file_info' => [
                    'exists' => $fileExists,
                    'type' => $fileType,
                    'name' => $essayTest->original_filename,
                    'size' => $essayTest->file_size,
                    'is_pdf' => $isPdf,
                    'is_docx' => $isDocx,
                    'download_url' => $fileUrl,
                ],
                'topic' => [
                    'id' => 1,
                    'title' => 'Writer Application Essay',
                    'description' => 'Essay submitted as part of the writer application process.',
                ],
            ],
            'writer' => [
                'id' => $profile->user->id,
                'name' => $profile->user->name,
                'email' => $profile->user->email,
                'writerProfile' => [
                    'country' => $profile->country ?? 'Not specified',
                    'education_level' => $profile->education_level ?? 'Not specified',
                    'experience_years' => $profile->experience_years ?? 0,
                    'questions_score' => $profile->qualificationTest ? $profile->qualificationTest->questions_score : null,
                    'questions_completed_at' => $profile->qualificationTest ? $profile->qualificationTest->questions_completed_at : null,
                ],
            ],
        ]);
    }

    /**
     * Download the essay file
     */
    public function download($id)
    {
        $profile = WriterProfile::with('essayTest')->findOrFail($id);
        $essayTest = $profile->essayTest;

        if (!$essayTest || !$essayTest->file_path || !Storage::exists($essayTest->file_path)) {
            return redirect()->route('admin.essays.show', $id)
                ->with('error', 'Essay file not found.');
        }

        return Storage::download(
            $essayTest->file_path,
            $essayTest->original_filename,
            ['Content-Type' => $essayTest->file_type]
        );
    }

    /**
     * Review an essay (approve or reject).
     */
    public function review(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:approved,rejected',
            'feedback' => $request->status === 'rejected' ? 'required|string|min:10' : 'nullable|string',
        ]);

        DB::beginTransaction();

        try {
            // Find the writer profile by ID
            $profile = WriterProfile::with(['user', 'essayTest'])->findOrFail($id);
            $user = $profile->user;
            $essayTest = $profile->essayTest;

            if (!$essayTest) {
                return back()->with('error', 'Essay test not found for this writer.');
            }

            // Update the essay test based on the review decision
            if ($request->status === 'approved') {
                $essayTest->essay_test_status = 'passed';
                $essayTest->essay_score = 100; // Default passing score
                $essayTest->essay_feedback = $request->feedback;
                $profile->is_available = true;
            } else {
                $essayTest->essay_test_status = 'essay_failed';
                $essayTest->essay_feedback = $request->feedback;
                $essayTest->last_failed_at = now();
                $essayTest->can_retry_after = now()->addMonths(3);
            }

            $essayTest->save();
            $profile->save();

            // Send notification to the writer
            $user->notify(new EssayReviewNotification(
                $request->status === 'approved',
                $request->feedback
            ));

            // Notify other admins about the essay review completion
            $adminEmail = env('ADMIN_EMAIL');
            if ($adminEmail && $adminEmail !== auth()->user()->email) {
                $admin = new AdminNotifiable($adminEmail);

                Notification::send($admin, new AdminEssayReviewNotification(
                    $user->name,
                    $user->email,
                    $request->status === 'approved',
                    $request->feedback,
                    auth()->user()->name,
                    'Writer Application Essay'
                ));
            }

            DB::commit();

            return redirect()->route('admin.essays.index')
                ->with('success', 'Essay review completed successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Essay review failed: ' . $e->getMessage());

            return back()->with('error', 'Failed to process the essay review: ' . $e->getMessage());
        }
    }
}
