<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class FineConfigurationController extends Controller
{
    /**
     * Display fine configuration dashboard
     */
    public function index()
    {
        $fineConfigurations = $this->getFineConfigurations();
        $defaultAmounts = $this->getDefaultFineAmounts();
        
        return view('admin.fine-configuration.index', compact('fineConfigurations', 'defaultAmounts'));
    }

    /**
     * Update fine configuration
     */
    public function update(Request $request)
    {
        $request->validate([
            'configurations' => 'required|array',
            'configurations.*.key' => 'required|string',
            'configurations.*.value' => 'required|numeric|min:0',
            'configurations.*.description' => 'nullable|string|max:255',
        ]);

        try {
            foreach ($request->configurations as $config) {
                AdminSetting::updateOrCreate(
                    ['key' => $config['key']],
                    [
                        'value' => $config['value'],
                        'description' => $config['description'] ?? $this->getDefaultDescription($config['key']),
                        'category' => 'fine_configuration',
                        'updated_at' => now(),
                    ]
                );
            }

            // Clear cache to ensure new values are used
            Cache::forget('fine_configurations');

            Log::info('Fine configurations updated', [
                'updated_by' => auth()->id(),
                'configurations' => $request->configurations
            ]);

            return redirect()->route('admin.fine-configuration.index')
                ->with('success', 'Fine configurations updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to update fine configurations', [
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to update fine configurations. Please try again.');
        }
    }

    /**
     * Reset fine configurations to defaults
     */
    public function resetToDefaults()
    {
        try {
            $defaultAmounts = $this->getDefaultFineAmounts();
            
            foreach ($defaultAmounts as $type => $severities) {
                foreach ($severities as $severity => $amount) {
                    $key = "fine_{$type}_{$severity}";
                    
                    AdminSetting::updateOrCreate(
                        ['key' => $key],
                        [
                            'value' => $amount,
                            'description' => $this->getDefaultDescription($key),
                            'category' => 'fine_configuration',
                            'updated_at' => now(),
                        ]
                    );
                }
            }

            // Clear cache
            Cache::forget('fine_configurations');

            Log::info('Fine configurations reset to defaults', [
                'reset_by' => auth()->id()
            ]);

            return redirect()->route('admin.fine-configuration.index')
                ->with('success', 'Fine configurations reset to defaults successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to reset fine configurations', [
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to reset fine configurations. Please try again.');
        }
    }

    /**
     * Get current fine configurations
     */
    private function getFineConfigurations(): array
    {
        return Cache::remember('fine_configurations', 3600, function () {
            $configurations = [];
            $defaultAmounts = $this->getDefaultFineAmounts();
            
            foreach ($defaultAmounts as $type => $severities) {
                foreach ($severities as $severity => $defaultAmount) {
                    $key = "fine_{$type}_{$severity}";
                    $setting = AdminSetting::where('key', $key)->first();
                    
                    $configurations[$type][$severity] = [
                        'key' => $key,
                        'value' => $setting ? (float) $setting->value : $defaultAmount,
                        'default' => $defaultAmount,
                        'description' => $setting ? $setting->description : $this->getDefaultDescription($key),
                        'is_customized' => $setting && (float) $setting->value !== $defaultAmount,
                    ];
                }
            }
            
            return $configurations;
        });
    }

    /**
     * Get default fine amounts
     */
    private function getDefaultFineAmounts(): array
    {
        return [
            'deadline_violation' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 35.00,
                'critical' => 75.00
            ],
            'quality_issue' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ],
            'communication_failure' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 30.00,
                'critical' => 60.00
            ],
            'policy_violation' => [
                'warning' => 15.00,
                'minor' => 35.00,
                'major' => 75.00,
                'critical' => 150.00
            ],
            'plagiarism' => [
                'warning' => 25.00,
                'minor' => 50.00,
                'major' => 100.00,
                'critical' => 200.00
            ],
            'instruction_non_compliance' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ],
            'unresponsive' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 30.00,
                'critical' => 60.00
            ],
            'unprofessional_behavior' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ]
        ];
    }

    /**
     * Get default description for a configuration key
     */
    private function getDefaultDescription(string $key): string
    {
        $descriptions = [
            'fine_deadline_violation_warning' => 'Fine for minor deadline violations (warnings)',
            'fine_deadline_violation_minor' => 'Fine for minor deadline violations',
            'fine_deadline_violation_major' => 'Fine for major deadline violations',
            'fine_deadline_violation_critical' => 'Fine for critical deadline violations',
            
            'fine_quality_issue_warning' => 'Fine for minor quality issues (warnings)',
            'fine_quality_issue_minor' => 'Fine for minor quality issues',
            'fine_quality_issue_major' => 'Fine for major quality issues',
            'fine_quality_issue_critical' => 'Fine for critical quality issues',
            
            'fine_communication_failure_warning' => 'Fine for minor communication failures (warnings)',
            'fine_communication_failure_minor' => 'Fine for minor communication failures',
            'fine_communication_failure_major' => 'Fine for major communication failures',
            'fine_communication_failure_critical' => 'Fine for critical communication failures',
            
            'fine_policy_violation_warning' => 'Fine for minor policy violations (warnings)',
            'fine_policy_violation_minor' => 'Fine for minor policy violations',
            'fine_policy_violation_major' => 'Fine for major policy violations',
            'fine_policy_violation_critical' => 'Fine for critical policy violations',
            
            'fine_plagiarism_warning' => 'Fine for minor plagiarism issues (warnings)',
            'fine_plagiarism_minor' => 'Fine for minor plagiarism issues',
            'fine_plagiarism_major' => 'Fine for major plagiarism issues',
            'fine_plagiarism_critical' => 'Fine for critical plagiarism issues',
            
            'fine_instruction_non_compliance_warning' => 'Fine for minor instruction non-compliance (warnings)',
            'fine_instruction_non_compliance_minor' => 'Fine for minor instruction non-compliance',
            'fine_instruction_non_compliance_major' => 'Fine for major instruction non-compliance',
            'fine_instruction_non_compliance_critical' => 'Fine for critical instruction non-compliance',
            
            'fine_unresponsive_warning' => 'Fine for minor unresponsive behavior (warnings)',
            'fine_unresponsive_minor' => 'Fine for minor unresponsive behavior',
            'fine_unresponsive_major' => 'Fine for major unresponsive behavior',
            'fine_unresponsive_critical' => 'Fine for critical unresponsive behavior',
            
            'fine_unprofessional_behavior_warning' => 'Fine for minor unprofessional behavior (warnings)',
            'fine_unprofessional_behavior_minor' => 'Fine for minor unprofessional behavior',
            'fine_unprofessional_behavior_major' => 'Fine for major unprofessional behavior',
            'fine_unprofessional_behavior_critical' => 'Fine for critical unprofessional behavior',
        ];

        return $descriptions[$key] ?? 'Fine configuration setting';
    }

    /**
     * Get fine configuration statistics
     */
    public function statistics()
    {
        $configurations = $this->getFineConfigurations();
        $statistics = [
            'total_configurations' => 0,
            'customized_configurations' => 0,
            'default_configurations' => 0,
            'total_fine_amount' => 0,
            'average_fine_amount' => 0,
            'highest_fine' => 0,
            'lowest_fine' => PHP_FLOAT_MAX,
        ];

        foreach ($configurations as $type => $severities) {
            foreach ($severities as $severity => $config) {
                $statistics['total_configurations']++;
                
                if ($config['is_customized']) {
                    $statistics['customized_configurations']++;
                } else {
                    $statistics['default_configurations']++;
                }
                
                $statistics['total_fine_amount'] += $config['value'];
                $statistics['highest_fine'] = max($statistics['highest_fine'], $config['value']);
                $statistics['lowest_fine'] = min($statistics['lowest_fine'], $config['value']);
            }
        }

        if ($statistics['total_configurations'] > 0) {
            $statistics['average_fine_amount'] = round($statistics['total_fine_amount'] / $statistics['total_configurations'], 2);
        }

        $statistics['lowest_fine'] = $statistics['lowest_fine'] === PHP_FLOAT_MAX ? 0 : $statistics['lowest_fine'];

        return response()->json($statistics);
    }

    /**
     * Export fine configuration to CSV
     */
    public function export()
    {
        $configurations = $this->getFineConfigurations();
        
        $filename = 'fine_configuration_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($configurations) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'Infraction Type', 'Severity Level', 'Current Amount', 'Default Amount', 
                'Is Customized', 'Description'
            ]);

            foreach ($configurations as $type => $severities) {
                foreach ($severities as $severity => $config) {
                    fputcsv($file, [
                        ucwords(str_replace('_', ' ', $type)),
                        ucfirst($severity),
                        '$' . number_format($config['value'], 2),
                        '$' . number_format($config['default'], 2),
                        $config['is_customized'] ? 'Yes' : 'No',
                        $config['description']
                    ]);
                }
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get fine configuration for API
     */
    public function api()
    {
        $configurations = $this->getFineConfigurations();
        
        return response()->json([
            'success' => true,
            'data' => $configurations,
            'last_updated' => now()->toISOString(),
        ]);
    }
} 