<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LoyaltyTier;
use App\Models\Coupon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class LoyaltyTierController extends Controller
{
    /**
     * Display a listing of loyalty tiers.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $query = LoyaltyTier::query();

        // Search filter
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Status filter
        if ($request->has('status') && $request->status !== 'all') {
            $status = $request->status === 'active';
            $query->where('is_active', $status);
        }

        // Sort by
        $sortField = $request->input('sort_field', 'min_orders');
        $sortDirection = $request->input('sort_direction', 'asc');
        $query->orderBy($sortField, $sortDirection);

        // Get paginated results with counts
        $tiers = $query->withCount('coupons')->paginate(10)->appends(request()->query());

        // Get counts for filter badges
        $counts = [
            'all' => LoyaltyTier::count(),
            'active' => LoyaltyTier::where('is_active', true)->count(),
            'inactive' => LoyaltyTier::where('is_active', false)->count(),
        ];

        return Inertia::render('Admin/LoyaltyTiers/Index', [
            'tiers' => $tiers,
            'counts' => $counts,
            'filters' => [
                'search' => $request->search ?? '',
                'status' => $request->status ?? 'all',
                'sort_field' => $sortField,
                'sort_direction' => $sortDirection,
            ],
        ]);
    }

    /**
     * Show the form for creating a new loyalty tier.
     *
     * @return \Inertia\Response
     */
    public function create()
    {
        return Inertia::render('Admin/LoyaltyTiers/Create', [
            'maxOrder' => LoyaltyTier::max('min_orders') ?? 0,
        ]);
    }

    /**
     * Store a newly created loyalty tier in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:loyalty_tiers,name',
            'min_orders' => 'required|integer|min:1',
            'discount_percentage' => 'required|numeric|min:0|max:100',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            // Check for overlap with existing tiers
            $existingTier = LoyaltyTier::where('min_orders', '<=', $validated['min_orders'])
                ->where('min_orders', '>', 0)
                ->orderByDesc('min_orders')
                ->first();

            if ($existingTier && $existingTier->min_orders === $validated['min_orders']) {
                return back()->withErrors(['min_orders' => 'A tier with this minimum order count already exists.']);
            }

            $loyaltyTier = LoyaltyTier::create($validated);

            DB::commit();

            return redirect()->route('admin.loyalty-tiers.index')
                ->with('success', "Loyalty tier '{$loyaltyTier->name}' created successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create loyalty tier: ' . $e->getMessage(), [
                'data' => $validated,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to create loyalty tier. Please try again.']);
        }
    }

    /**
     * Display the specified loyalty tier.
     *
     * @param LoyaltyTier $loyaltyTier
     * @return \Inertia\Response
     */
    public function show(LoyaltyTier $loyaltyTier)
    {
        // Load active coupons for this tier with counts
        $activeCoupons = Coupon::where('loyalty_tier_id', $loyaltyTier->id)
            ->where('is_active', true)
            ->where(function ($query) {
                $now = now();
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', $now);
            })
            ->where(function ($query) {
                $now = now();
                $query->whereNull('expiry_date')
                    ->orWhere('expiry_date', '>=', $now);
            })
            ->get();

        // Get tier usage statistics
        $statistics = [
            'total_coupons' => Coupon::where('loyalty_tier_id', $loyaltyTier->id)->count(),
            'active_coupons' => $activeCoupons->count(),
            'total_uses' => Coupon::where('loyalty_tier_id', $loyaltyTier->id)->sum('uses_count'),
            'eligible_users' => DB::table('users')
                ->join('orders', 'users.id', '=', 'orders.user_id')
                ->where('orders.payment_status', 'paid')
                ->groupBy('users.id')
                ->havingRaw('COUNT(DISTINCT orders.id) >= ?', [$loyaltyTier->min_orders])
                ->count(),
        ];

        // Get all coupons associated with this tier
        $coupons = Coupon::where('loyalty_tier_id', $loyaltyTier->id)
            ->withCount('userUsages')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return Inertia::render('Admin/LoyaltyTiers/Show', [
            'loyaltyTier' => $loyaltyTier,
            'statistics' => $statistics,
            'coupons' => $coupons,
            'activeCoupons' => $activeCoupons,
        ]);
    }

    /**
     * Show the form for editing the specified loyalty tier.
     *
     * @param LoyaltyTier $loyaltyTier
     * @return \Inertia\Response
     */
    public function edit(LoyaltyTier $loyaltyTier)
    {
        // Get adjacent tiers for validation context
        $nextTier = LoyaltyTier::where('min_orders', '>', $loyaltyTier->min_orders)
            ->orderBy('min_orders', 'asc')
            ->first();

        $previousTier = LoyaltyTier::where('min_orders', '<', $loyaltyTier->min_orders)
            ->orderBy('min_orders', 'desc')
            ->first();

        return Inertia::render('Admin/LoyaltyTiers/Edit', [
            'loyaltyTier' => $loyaltyTier,
            'nextTier' => $nextTier,
            'previousTier' => $previousTier,
            'couponCount' => Coupon::where('loyalty_tier_id', $loyaltyTier->id)->count(),
        ]);
    }

    /**
     * Update the specified loyalty tier in storage.
     *
     * @param Request $request
     * @param LoyaltyTier $loyaltyTier
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, LoyaltyTier $loyaltyTier)
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('loyalty_tiers')->ignore($loyaltyTier->id),
            ],
            'min_orders' => [
                'required',
                'integer',
                'min:1',
                Rule::unique('loyalty_tiers')->ignore($loyaltyTier->id),
            ],
            'discount_percentage' => 'required|numeric|min:0|max:100',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            // Check for overlap with existing tiers
            $existingTier = LoyaltyTier::where('id', '!=', $loyaltyTier->id)
                ->where('min_orders', $validated['min_orders'])
                ->exists();

            if ($existingTier) {
                return back()->withErrors(['min_orders' => 'A tier with this minimum order count already exists.']);
            }

            $loyaltyTier->update($validated);

            DB::commit();

            return redirect()->route('admin.loyalty-tiers.index')
                ->with('success', "Loyalty tier '{$loyaltyTier->name}' updated successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update loyalty tier: ' . $e->getMessage(), [
                'tier_id' => $loyaltyTier->id,
                'data' => $validated,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update loyalty tier. Please try again.']);
        }
    }

    /**
     * Remove the specified loyalty tier from storage.
     *
     * @param LoyaltyTier $loyaltyTier
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(LoyaltyTier $loyaltyTier)
    {
        try {
            DB::beginTransaction();

            // Check if there are coupons using this tier
            $couponCount = Coupon::where('loyalty_tier_id', $loyaltyTier->id)->count();
            if ($couponCount > 0) {
                return back()->withErrors([
                    'general' => "Cannot delete this loyalty tier. It's being used by {$couponCount} coupons."
                ]);
            }

            $tierName = $loyaltyTier->name;
            $loyaltyTier->delete();

            DB::commit();

            return redirect()->route('admin.loyalty-tiers.index')
                ->with('success', "Loyalty tier '{$tierName}' deleted successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to delete loyalty tier: ' . $e->getMessage(), [
                'tier_id' => $loyaltyTier->id,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to delete loyalty tier. Please try again.']);
        }
    }

    /**
     * Toggle the status of the specified loyalty tier.
     *
     * @param LoyaltyTier $loyaltyTier
     * @return \Illuminate\Http\RedirectResponse
     */
    public function toggleStatus(LoyaltyTier $loyaltyTier)
    {
        try {
            $loyaltyTier->is_active = !$loyaltyTier->is_active;
            $loyaltyTier->save();

            $status = $loyaltyTier->is_active ? 'activated' : 'deactivated';
            return back()->with('success', "Loyalty tier '{$loyaltyTier->name}' {$status} successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to toggle loyalty tier status: ' . $e->getMessage(), [
                'tier_id' => $loyaltyTier->id,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update loyalty tier status. Please try again.']);
        }
    }

    /**
     * Bulk toggle status for multiple loyalty tiers.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkToggleStatus(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:loyalty_tiers,id',
            'status' => 'required|boolean',
        ]);

        try {
            DB::beginTransaction();

            $count = LoyaltyTier::whereIn('id', $validated['ids'])
                ->update(['is_active' => $validated['status']]);

            DB::commit();

            $status = $validated['status'] ? 'activated' : 'deactivated';
            return back()->with('success', "{$count} loyalty tiers {$status} successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to bulk toggle loyalty tier status: ' . $e->getMessage(), [
                'ids' => $validated['ids'],
                'status' => $validated['status'],
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update loyalty tier statuses. Please try again.']);
        }
    }

    /**
     * Bulk delete multiple loyalty tiers.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:loyalty_tiers,id',
        ]);

        try {
            DB::beginTransaction();

            // Check for coupons using these tiers
            $couponCounts = Coupon::whereIn('loyalty_tier_id', $validated['ids'])
                ->selectRaw('loyalty_tier_id, count(*) as count')
                ->groupBy('loyalty_tier_id')
                ->get()
                ->keyBy('loyalty_tier_id');

            if ($couponCounts->isNotEmpty()) {
                $tierWithCoupons = LoyaltyTier::whereIn('id', $couponCounts->keys())->pluck('name');
                return back()->withErrors([
                    'general' => "Cannot delete loyalty tiers that are being used by coupons: " . $tierWithCoupons->implode(', ')
                ]);
            }

            $count = LoyaltyTier::whereIn('id', $validated['ids'])->delete();

            DB::commit();

            return back()->with('success', "{$count} loyalty tiers deleted successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to bulk delete loyalty tiers: ' . $e->getMessage(), [
                'ids' => $validated['ids'],
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to delete loyalty tiers. Please try again.']);
        }
    }

    /**
     * Get loyalty tier usage statistics.
     *
     * @return \Inertia\Response
     */
    public function showStatistics()
    {
        // Get tier usage data
        $tiers = LoyaltyTier::withCount('coupons')
            ->withSum('coupons', 'uses_count')
            ->orderBy('min_orders')
            ->get();

        // Get eligible user counts for each tier
        foreach ($tiers as $tier) {
            $tier->eligible_users_count = DB::table('users')
                ->join('orders', 'users.id', '=', 'orders.user_id')
                ->where('orders.payment_status', 'paid')
                ->groupBy('users.id')
                ->havingRaw('COUNT(DISTINCT orders.id) >= ?', [$tier->min_orders])
                ->count();
        }

        // Get overall statistics
        $statistics = [
            'total_tiers' => $tiers->count(),
            'active_tiers' => $tiers->where('is_active', true)->count(),
            'total_coupons' => $tiers->sum('coupons_count'),
            'total_redemptions' => $tiers->sum('coupons_sum_uses_count') ?? 0,
        ];

        return Inertia::render('Admin/LoyaltyTiers/Statistics', [
            'tiers' => $tiers,
            'statistics' => $statistics,
        ]);
    }
}
