<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\User;
use App\Notifications\RevisionQualityAssessedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class OrderRevisionController extends Controller
{
    /**
     * Display a listing of all revisions.
     */
    public function index(Request $request)
    {
        $query = OrderRevision::with(['order', 'requestedBy', 'assignedTo'])
            ->orderBy('created_at', 'desc');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by revision type
        if ($request->filled('revision_type')) {
            $query->where('revision_type', $request->revision_type);
        }

        // Filter by writer
        if ($request->filled('writer_id')) {
            $query->where('assigned_to', $request->writer_id);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search by order number
        if ($request->filled('search')) {
            $query->whereHas('order', function ($q) use ($request) {
                $q->where('order_number', 'like', '%' . $request->search . '%');
            });
        }

        $revisions = $query->paginate(20);
        $stats = $this->getRevisionStats();

        return view('admin.revisions.index', compact('revisions', 'stats'));
    }

    /**
     * Show the revision dashboard with overview statistics.
     */
    public function dashboard()
    {
        $stats = $this->getRevisionStats();
        $recentRevisions = OrderRevision::with(['order', 'requestedBy', 'assignedTo'])
            ->latest()
            ->take(10)
            ->get();
        
        $overdueRevisions = OrderRevision::overdue()
            ->with(['order', 'assignedTo'])
            ->take(5)
            ->get();

        $revisionTrends = $this->getRevisionTrends();

        return view('admin.revisions.dashboard', compact('stats', 'recentRevisions', 'overdueRevisions', 'revisionTrends'));
    }

    /**
     * Show the form for creating a new revision.
     */
    public function create()
    {
        // Allow revisions on orders that are under review, completed, or delivered to client
        $orders = Order::whereIn('order_status', ['under_review', 'completed', 'delivered_to_client'])
            ->with(['writer', 'user'])
            ->get();
        
        $writers = User::where('role', 'writer')
            ->where('is_active', true)
            ->get();

        return view('admin.revisions.create', compact('orders', 'writers'));
    }

    /**
     * Store a newly created revision.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'order_id' => 'required|exists:orders,id',
            'assigned_to' => 'required|exists:users,id',
            'revision_type' => 'required|in:quality,requirements,formatting,plagiarism,deadline,communication,other',
            'revision_reason' => 'required|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
            'due_date' => 'required|date|after:now',
            'revision_number' => 'required|integer|min:1',
            'is_final_revision' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            $order = Order::findOrFail($validated['order_id']);
            
            // Check if this is a valid revision request
            if (!$this->canRequestRevision($order, $validated['revision_number'])) {
                return back()->withErrors(['error' => 'Cannot request revision for this order at this time.']);
            }

            $revision = OrderRevision::create([
                'order_id' => $validated['order_id'],
                'requested_by' => auth()->id(),
                'assigned_to' => $validated['assigned_to'],
                'revision_type' => $validated['revision_type'],
                'revision_reason' => $validated['revision_reason'],
                'specific_instructions' => $validated['specific_instructions'],
                'status' => 'requested',
                'requested_at' => now(),
                'due_date' => $validated['due_date'],
                'revision_number' => $validated['revision_number'],
                'is_final_revision' => $validated['is_final_revision'] ?? false,
            ]);

            // Update order status if needed
            if ($order->order_status === 'under_review') {
                $order->update(['order_status' => 'revision_requested']);
            }

            // Create conversation thread for the revision
            $conversation = $this->createRevisionConversation($revision);
            $revision->update(['conversation_id' => $conversation->id]);

            DB::commit();

            // Send notification to assigned writer
            $this->notifyWriterOfRevision($revision);

            return redirect()->route('admin.revisions.show', $revision)
                ->with('success', 'Revision request created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to create revision. Please try again.']);
        }
    }

    /**
     * Display the specified revision.
     */
    public function show(OrderRevision $revision)
    {
        $revision->load(['order.writer', 'order.user', 'requestedBy', 'assignedTo', 'conversation.messages.user']);
        
        $revisionHistory = OrderRevision::where('order_id', $revision->order_id)
            ->orderBy('revision_number', 'asc')
            ->get();

        return view('admin.revisions.show', compact('revision', 'revisionHistory'));
    }

    /**
     * Show the form for editing the specified revision.
     */
    public function edit(OrderRevision $revision)
    {
        $writers = User::where('role', 'writer')
            ->where('is_active', true)
            ->get();

        return view('admin.revisions.edit', compact('revision', 'writers'));
    }

    /**
     * Update the specified revision.
     */
    public function update(Request $request, OrderRevision $revision)
    {
        $validated = $request->validate([
            'assigned_to' => 'nullable|exists:users,id',
            'revision_type' => ['required', Rule::in(['quality', 'requirements', 'formatting', 'plagiarism', 'deadline', 'communication', 'other'])],
            'revision_reason' => 'required|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
            'due_date' => 'required|date',
            'status' => ['required', Rule::in(['requested', 'in_progress', 'completed', 'rejected', 'escalated'])],
            'quality_rating' => 'nullable|integer|between:1,10',
            'quality_notes' => 'nullable|string|max:1000',
            'payment_delay_hours' => 'nullable|numeric|min:0',
            'quality_penalty' => 'nullable|numeric|min:0',
            'quality_bonus' => 'nullable|numeric|min:0',
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $oldStatus = $revision->status;
            $revision->update($validated);

            // Handle status transitions
            $this->handleStatusTransition($revision, $oldStatus, $validated['status']);

            // Update order status if revision is completed
            if ($validated['status'] === 'completed') {
                $this->handleRevisionCompletion($revision);
            }

            DB::commit();

            return redirect()->route('admin.revisions.show', $revision)
                ->with('success', 'Revision updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to update revision. Please try again.']);
        }
    }

    /**
     * Remove the specified revision.
     */
    public function destroy(OrderRevision $revision)
    {
        try {
            if ($revision->status !== 'requested') {
                return back()->withErrors(['error' => 'Only requested revisions can be deleted.']);
            }

            $revision->delete();

            return redirect()->route('admin.revisions.index')
                ->with('success', 'Revision deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to delete revision. Please try again.']);
        }
    }

    /**
     * Assign revision to a writer.
     */
    public function assignWriter(Request $request, OrderRevision $revision)
    {
        $validated = $request->validate([
            'writer_id' => 'required|exists:users,id',
            'due_date' => 'nullable|date|after:now',
        ]);

        try {
            $revision->update([
                'assigned_to' => $validated['writer_id'],
                'due_date' => $validated['due_date'] ?? $revision->due_date,
                'status' => 'requested',
            ]);

            // Send notification to assigned writer
            $this->notifyWriterOfRevision($revision);

            return back()->with('success', 'Revision assigned to writer successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to assign revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to assign revision. Please try again.']);
        }
    }

    /**
     * Handle client-initiated revision request.
     */
    public function clientRevisionRequest(Request $request)
    {
        $validated = $request->validate([
            'order_id' => 'required|exists:orders,id',
            'revision_type' => 'required|in:quality,requirements,formatting,plagiarism,deadline,communication,other',
            'revision_reason' => 'required|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
            'client_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $order = Order::findOrFail($validated['order_id']);
            
            // Check if this is a valid revision request
            $nextRevisionNumber = $order->revisions()->count() + 1;
            if (!$this->canRequestRevision($order, $nextRevisionNumber)) {
                return back()->withErrors(['error' => 'Cannot request revision for this order at this time.']);
            }

            // Create revision request (initially unassigned)
            $revision = OrderRevision::create([
                'order_id' => $validated['order_id'],
                'requested_by' => auth()->id(), // Admin who processed client request
                'assigned_to' => null, // Will be assigned later
                'revision_type' => $validated['revision_type'],
                'revision_reason' => $validated['revision_reason'],
                'specific_instructions' => $validated['specific_instructions'],
                'client_notes' => $validated['client_notes'],
                'status' => 'requested',
                'requested_at' => now(),
                'due_date' => now()->addHours(config('revisions.default_due_hours')),
                'revision_number' => $nextRevisionNumber,
                'is_final_revision' => false,
                'is_client_requested' => true, // Mark as client-initiated
            ]);

            // Update order status
            if ($order->order_status === 'delivered_to_client') {
                $order->update(['order_status' => 'revision_requested']);
            }

            DB::commit();

            return redirect()->route('admin.revisions.show', $revision)
                ->with('success', 'Client revision request created successfully. Please assign to a writer.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create client revision request: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to create client revision request. Please try again.']);
        }
    }

    /**
     * Escalate a revision.
     */
    public function escalate(Request $request, OrderRevision $revision)
    {
        $validated = $request->validate([
            'escalation_reason' => 'required|string|max:1000',
        ]);

        try {
            $revision->update([
                'status' => 'escalated',
                'escalation_reason' => $validated['escalation_reason'],
            ]);

            // Send escalation notification
            $this->notifyEscalation($revision);

            return back()->with('success', 'Revision escalated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to escalate revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to escalate revision. Please try again.']);
        }
    }

    /**
     * Get revision statistics for dashboard.
     */
    private function getRevisionStats(): array
    {
        $totalRevisions = OrderRevision::count();
        $pendingRevisions = OrderRevision::whereIn('status', ['requested', 'in_progress'])->count();
        $overdueRevisions = OrderRevision::overdue()->count();
        $completedRevisions = OrderRevision::where('status', 'completed')->count();
        $escalatedRevisions = OrderRevision::where('status', 'escalated')->count();

        $revisionTypes = OrderRevision::select('revision_type', DB::raw('count(*) as count'))
            ->groupBy('revision_type')
            ->pluck('count', 'revision_type')
            ->toArray();

        $avgCompletionTime = OrderRevision::where('status', 'completed')
            ->whereNotNull('started_at')
            ->whereNotNull('completed_at')
            ->avg(DB::raw('TIMESTAMPDIFF(HOUR, started_at, completed_at)'));

        return [
            'total' => $totalRevisions,
            'pending' => $pendingRevisions,
            'overdue' => $overdueRevisions,
            'completed' => $completedRevisions,
            'escalated' => $escalatedRevisions,
            'types' => $revisionTypes,
            'avg_completion_hours' => round($avgCompletionTime ?? 0, 1),
        ];
    }

    /**
     * Get revision trends for dashboard.
     */
    private function getRevisionTrends(): array
    {
        $last30Days = OrderRevision::select(
            DB::raw('DATE(created_at) as date'),
            DB::raw('count(*) as count')
        )
            ->where('created_at', '>=', now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->pluck('count', 'date')
            ->toArray();

        return $last30Days;
    }

    /**
     * Check if a revision can be requested for an order.
     */
    private function canRequestRevision(Order $order, int $revisionNumber): bool
    {
        // Check if order is in a valid state for revisions
        // Allow revisions on: under_review, completed, delivered_to_client
        if (!in_array($order->order_status, ['under_review', 'completed', 'delivered_to_client'])) {
            return false;
        }

        // Check revision limits
        $maxRevisions = config('revisions.max_per_order', 3);
        if ($revisionNumber > $maxRevisions) {
            return false;
        }

        // Check if previous revision is still pending
        $pendingRevision = $order->revisions()
            ->whereIn('status', ['requested', 'in_progress'])
            ->exists();

        if ($pendingRevision) {
            return false;
        }

        return true;
    }

    /**
     * Handle status transitions for revisions.
     */
    private function handleStatusTransition(OrderRevision $revision, string $oldStatus, string $newStatus): void
    {
        if ($oldStatus === $newStatus) {
            return;
        }

        switch ($newStatus) {
            case 'in_progress':
                $revision->update(['started_at' => now()]);
                break;
            case 'completed':
                $revision->update(['completed_at' => now()]);
                break;
            case 'rejected':
                $revision->update(['rejected_at' => now()]);
                break;
        }
    }

    /**
     * Handle revision completion.
     */
    private function handleRevisionCompletion(OrderRevision $revision): void
    {
        $order = $revision->order;
        
        // Update order status back to under review
        if ($order->order_status === 'revision_requested') {
            $order->update(['order_status' => 'under_review']);
        }

        // Check if this was the final revision
        if ($revision->is_final_revision) {
            $order->update(['order_status' => 'completed']);
        }
    }

    /**
     * Assess quality of completed revision and calculate payment impact.
     */
    public function assessQuality(Request $request, OrderRevision $revision)
    {
        // Only allow quality assessment on completed revisions
        if ($revision->status !== 'completed') {
            return back()->withErrors(['error' => 'Only completed revisions can be quality assessed.']);
        }

        $validated = $request->validate([
            'quality_rating' => 'required|integer|between:1,10',
            'quality_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            // Calculate payment impact based on quality rating
            $paymentImpact = $this->calculatePaymentImpact($validated['quality_rating']);
            
            // Update revision with quality assessment
            $revision->update([
                'quality_rating' => $validated['quality_rating'],
                'quality_notes' => $validated['quality_notes'],
                'quality_bonus' => $paymentImpact['bonus'],
                'quality_penalty' => $paymentImpact['penalty'],
                'payment_impact' => $paymentImpact['total'],
            ]);

            // Update order status if this was the final revision
            if ($revision->is_final_revision) {
                $revision->order->update(['order_status' => 'completed']);
            }

            // Notify writer of quality assessment
            try {
                $qualityData = [
                    'quality_rating' => $validated['quality_rating'],
                    'quality_notes' => $validated['quality_notes'],
                    'payment_impact' => $paymentImpact['total'],
                    'quality_bonus' => $paymentImpact['bonus'],
                    'quality_penalty' => $paymentImpact['penalty'],
                ];
                
                $revision->assignedTo->notify(new RevisionQualityAssessedNotification($revision, $qualityData));
            } catch (\Exception $e) {
                Log::error('Failed to send quality assessment notification: ' . $e->getMessage());
            }

            DB::commit();

            return back()->with('success', 'Quality assessment completed successfully. Payment impact: $' . number_format($paymentImpact['total'], 2));

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to assess revision quality: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to assess revision quality. Please try again.']);
        }
    }

    /**
     * Calculate payment impact based on quality rating.
     */
    private function calculatePaymentImpact(int $qualityRating): array
    {
        $config = config('revisions.payment_impact');
        
        if ($qualityRating >= 9) {
            // Excellent quality - bonus
            $bonus = $config['quality_bonus'];
            $penalty = 0;
        } elseif ($qualityRating >= 7) {
            // Good quality - no impact
            $bonus = 0;
            $penalty = 0;
        } elseif ($qualityRating >= 5) {
            // Poor quality - penalty
            $bonus = 0;
            $penalty = $config['quality_penalty'] * 0.6; // 60% of full penalty
        } else {
            // Very poor quality - full penalty
            $bonus = 0;
            $penalty = $config['quality_penalty'];
        }

        return [
            'bonus' => $bonus,
            'penalty' => $penalty,
            'total' => $bonus - $penalty, // Positive for bonus, negative for penalty
        ];
    }

    /**
     * Create conversation thread for revision.
     */
    private function createRevisionConversation(OrderRevision $revision): \App\Models\Conversation
    {
        $conversation = \App\Models\Conversation::create([
            'title' => "Revision #{$revision->revision_number} - Order #{$revision->order->order_number}",
            'type' => 'revision',
            'status' => 'active',
        ]);

        // Add participants
        $conversation->participants()->createMany([
            ['user_id' => $revision->requested_by, 'user_type' => 'admin'],
            ['user_id' => $revision->assigned_to, 'user_type' => 'writer'],
        ]);

        return $conversation;
    }

    /**
     * Notify writer of revision assignment.
     */
    private function notifyWriterOfRevision(OrderRevision $revision): void
    {
        try {
            $revision->assignedTo->notify(new \App\Notifications\RevisionAssignedNotification($revision));
        } catch (\Exception $e) {
            Log::error('Failed to send revision notification: ' . $e->getMessage());
        }
    }

    /**
     * Reject a revision and request additional work.
     */
    public function reject(Request $request, OrderRevision $revision)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            // Update revision status to rejected
            $revision->update([
                'status' => 'rejected',
                'rejected_at' => now(),
                'admin_notes' => $validated['rejection_reason'],
            ]);

            // Reset order status to under review
            $revision->order->update(['order_status' => 'under_review']);

            // Notify writer of rejection
            try {
                $revision->assignedTo->notify(new \App\Notifications\RevisionRejectedNotification($revision, $validated['rejection_reason']));
            } catch (\Exception $e) {
                Log::error('Failed to send rejection notification: ' . $e->getMessage());
            }

            DB::commit();

            return back()->with('success', 'Revision rejected successfully. Writer has been notified to make corrections.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to reject revision: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to reject revision. Please try again.']);
        }
    }

    /**
     * Notify escalation.
     */
    private function notifyEscalation(OrderRevision $revision): void
    {
        try {
            // Notify admins of escalation
            $admins = User::where('role', 'admin')->get();
            foreach ($admins as $admin) {
                $admin->notify(new \App\Notifications\RevisionEscalatedNotification($revision));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send escalation notification: ' . $e->getMessage());
        }
    }
} 