<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\OrderStatusService;
use App\Services\AdminNotificationService;
use App\Notifications\OrderStatusChangeNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class OrderStatusController extends Controller
{
    protected OrderStatusService $orderStatusService;

    public function __construct(OrderStatusService $orderStatusService)
    {
        $this->orderStatusService = $orderStatusService;
        $this->middleware('auth');
        $this->middleware('admin');
    }

    /**
     * Change order status (admin action).
     */
    public function changeStatus(Request $request, Order $order)
    {
        $request->validate([
            'new_status' => 'required|string',
            'reason' => 'nullable|string|max:500',
            'metadata' => 'nullable|array'
        ]);

        $newStatus = $request->input('new_status');
        $reason = $request->input('reason');
        $metadata = $request->input('metadata', []);

        try {
            // Use OrderStatusService to handle status transition
            $this->orderStatusService->changeStatus($order, $newStatus, $reason, $metadata);

            // Send notification to relevant parties
            $this->sendStatusChangeNotifications($order, $newStatus, $reason);

            return response()->json([
                'message' => 'Order status updated successfully',
                'order' => $order->fresh()->load(['user', 'writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to change order status: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'admin_id' => auth()->id(),
                'new_status' => $newStatus,
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to change order status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Complete admin review and send work to client.
     */
    public function completeAdminReview(Request $request, Order $order)
    {
        $request->validate([
            'review_notes' => 'nullable|string|max:1000',
            'client_files.*' => 'nullable|file|mimes:pdf,doc,docx,txt|max:10240'
        ]);

        // Verify the order is submitted
        if ($order->order_status !== 'submitted') {
            return response()->json(['message' => 'Order is not in submitted status'], 400);
        }

        try {
            // Handle file uploads for client
            $uploadedFiles = [];
            if ($request->hasFile('client_files')) {
                foreach ($request->file('client_files') as $file) {
                    $path = $file->store('client-deliveries', 'public');
                    
                    $orderFile = \App\Models\OrderFile::create([
                        'order_id' => $order->id,
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'file_type' => $file->getClientMimeType(),
                        'file_size' => $file->getSize(),
                        'uploaded_by' => auth()->id(),
                        'file_status' => 'delivered',
                        'upload_type' => 'admin_delivery'
                    ]);

                    $uploadedFiles[] = $orderFile;
                }
            }

            // Use OrderStatusService to handle status transition
            $this->orderStatusService->handleAdminReviewCompleted($order);

            // Update review notes if provided
            if ($request->filled('review_notes')) {
                $order->update([
                    'order_revision' => $request->review_notes
                ]);
            }

            // Send notifications
            $this->sendAdminReviewNotifications($order, $uploadedFiles);

            return response()->json([
                'message' => 'Admin review completed. Work sent to client for approval.',
                'order' => $order->fresh()->load(['user', 'writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to complete admin review: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'admin_id' => auth()->id(),
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to complete admin review: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle client approval/rejection.
     */
    public function handleClientDecision(Request $request, Order $order)
    {
        $request->validate([
            'decision' => 'required|in:approve,reject',
            'revision_notes' => 'required_if:decision,reject|string|max:1000'
        ]);

        $decision = $request->input('decision');

        try {
            if ($decision === 'approve') {
                $this->orderStatusService->handleClientApproval($order);
                $message = 'Order approved successfully';
            } else {
                $revisionNotes = $request->input('revision_notes');
                $this->orderStatusService->handleClientRejection($order, $revisionNotes);
                $message = 'Revision requested. Writer has been notified.';
            }

            // Send notifications
            $this->sendClientDecisionNotifications($order, $decision);

            return response()->json([
                'message' => $message,
                'order' => $order->fresh()->load(['user', 'writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to handle client decision: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'admin_id' => auth()->id(),
                'decision' => $decision,
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to handle client decision: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark order as under review when admin accesses writer-submitted files.
     */
    public function markAsUnderReview(Request $request, Order $order)
    {
        // Verify the order is submitted
        if ($order->order_status !== 'submitted') {
            return response()->json(['message' => 'Order is not in submitted status'], 400);
        }

        try {
            // Use OrderStatusService to handle status transition
            $this->orderStatusService->changeStatus($order, 'under_review', 'Admin accessed writer-submitted files for review', [
                'admin_id' => auth()->id(),
                'action' => 'file_access',
                'timestamp' => now()
            ]);

            // Send notification to relevant parties
            $this->sendStatusChangeNotifications($order, 'under_review', 'Admin accessed writer-submitted files for review');

            return response()->json([
                'message' => 'Order marked as under review',
                'order' => $order->fresh()->load(['user', 'writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to mark order as under review: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'admin_id' => auth()->id(),
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to mark order as under review: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get order status history.
     */
    public function getStatusHistory(Order $order)
    {
        $statusHistory = $order->statusHistory()
            ->with('changedByUser')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'status_history' => $statusHistory
        ]);
    }

    /**
     * Send notifications for status changes.
     */
    private function sendStatusChangeNotifications(Order $order, string $newStatus, ?string $reason)
    {
        try {
            // Notify the client
            $client = $order->user;
            if ($client) {
                $client->notify(new OrderStatusChangeNotification($order, $newStatus, $reason));
            }

            // Notify the writer if assigned
            $writer = $order->writer;
            if ($writer) {
                $writer->notify(new OrderStatusChangeNotification($order, $newStatus, $reason));
            }

        } catch (\Exception $e) {
            Log::error('Error sending status change notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'new_status' => $newStatus,
                'exception' => $e
            ]);
        }
    }

    /**
     * Send notifications for admin review completion.
     */
    private function sendAdminReviewNotifications(Order $order, array $uploadedFiles)
    {
        try {
            // Notify the client that work is ready for review
            $client = $order->user;
            if ($client) {
                $client->notify(new \App\Notifications\WorkReadyForReviewNotification($order, $uploadedFiles));
            }

            // Notify the writer that work has been delivered to client
            $writer = $order->writer;
            if ($writer) {
                // TODO: Create WorkDeliveredToClientNotification
                // $writer->notify(new WorkDeliveredToClientNotification($order, $uploadedFiles));
            }

        } catch (\Exception $e) {
            Log::error('Error sending admin review notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }

    /**
     * Send notifications for client decisions.
     */
    private function sendClientDecisionNotifications(Order $order, string $decision)
    {
        try {
            if ($decision === 'approve') {
                // Notify the writer about approval
                $writer = $order->writer;
                if ($writer) {
                    $writer->notify(new \App\Notifications\OrderApprovedNotification($order));
                }
            } else {
                // Notify the writer about revision request
                $writer = $order->writer;
                if ($writer) {
                    $writer->notify(new \App\Notifications\RevisionRequestedNotification($order, $order->revision_notes ?? ''));
                }
            }

        } catch (\Exception $e) {
            Log::error('Error sending client decision notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'decision' => $decision,
                'exception' => $e
            ]);
        }
    }
}
