<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use App\Models\PageContent;
use App\Models\PageSeo;
use App\Models\PageMedia;
use App\Services\MediaService;
use App\Services\PageService;
use App\Services\SEOService;
use App\Services\ServicePageScaffolder;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class PageController extends Controller
{
    protected PageService $pageService;
    protected SEOService $seoService;
    protected MediaService $mediaService;
    protected ServicePageScaffolder $servicePageScaffolder;

    public function __construct(
        PageService $pageService,
        SEOService $seoService,
        MediaService $mediaService,
        ServicePageScaffolder $servicePageScaffolder
    ) {
        $this->pageService = $pageService;
        $this->seoService = $seoService;
        $this->mediaService = $mediaService;
        $this->servicePageScaffolder = $servicePageScaffolder;
    }

    /**
     * Display a listing of pages
     */
    public function index(Request $request): Response
    {
        $filters = $request->only(['status', 'page_type', 'search']);
        $pages = $this->pageService->getAllPages($filters);

        // Generate stats
        $stats = [
            'totalPages' => Page::count(),
            'publishedPages' => Page::where('status', 'published')->count(),
            'draftPages' => Page::where('status', 'draft')->count(),
            'archivedPages' => Page::where('status', 'archived')->count(),
            'servicePages' => Page::where('page_type', 'service_page')->count(),
            'blogPages' => Page::where('page_type', 'blog_page')->count(),
            'totalContentBlocks' => PageContent::count(),
            'activeContentBlocks' => PageContent::where('is_active', true)->count(),
            'totalMedia' => PageMedia::count(),
            'optimizedMedia' => PageMedia::where('is_optimized', true)->count(),
            'seoScore' => $this->calculateAverageSeoScore(),
        ];

        return Inertia::render('Admin/CMS/Index', [
            'pages' => $pages,
            'stats' => $stats,
            'filters' => $filters,
            'pageTypes' => $this->getPageTypes(),
            'statuses' => ['draft', 'published', 'archived']
        ]);
    }

    /**
     * Edit a page by slug (convenience route for specific editors)
     */
    public function editBySlug(string $slug): Response
    {
        $page = Page::with(['contentBlocks', 'seo', 'media'])->where('slug', $slug)->firstOrFail();

        // Provide auxiliary data similar to edit()
        $pageTypes = ['service', 'landing', 'static', 'blog'];
        $templates = ['default', $page->template];
        $parentPages = Page::whereNull('parent_id')->where('id', '!=', $page->id)->get(['id', 'title', 'slug']);

        return Inertia::render('Admin/CMS/Pages/Edit', [
            'page' => [
                'id' => $page->id,
                'title' => $page->title,
                'slug' => $page->slug,
                'page_type' => $page->page_type,
                'status' => $page->status,
                'parent_id' => $page->parent_id,
                'template' => $page->template,
                'seo_priority' => $page->seo_priority,
                'published_at' => optional($page->published_at)?->toISOString(),
                'updated_at' => optional($page->updated_at)?->toISOString(),
                'content_blocks' => $page->contentBlocks,
                'seo' => $page->seo,
                'media' => $page->media,
            ],
            'pageTypes' => $pageTypes,
            'templates' => $templates,
            'parentPages' => $parentPages,
        ]);
    }

    /**
     * Dedicated editor for Affordable Essay Writing page
     */
    public function editAffordable(): Response
    {
        $page = Page::with(['contentBlocks', 'seo', 'media'])->where('slug', 'affordable-essay-writing')->firstOrFail();

        // Process content blocks to ensure proper JSON structure
        $contentBlocks = $page->contentBlocks->map(function ($block) {
            return [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content, // Already cast to array by model
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        });

        return Inertia::render('Admin/CMS/Services/AffordableEditor', [
            'page' => [
                'id' => $page->id,
                'title' => $page->title,
                'slug' => $page->slug,
                'content_blocks' => $contentBlocks,
                'seo' => $page->seo,
            ],
        ]);
    }

    /**
     * Dedicated editor for Buy Essays Online page
     */
    public function editBuy(): Response
    {
        $page = Page::with(['contentBlocks', 'seo', 'media'])->where('slug', 'buy-essays-online')->firstOrFail();

        // Process content blocks to ensure proper JSON structure
        $contentBlocks = $page->contentBlocks->map(function ($block) {
            return [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content, // Already cast to array by model
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        });

        return Inertia::render('Admin/CMS/Services/BuyEditor', [
            'page' => [
                'id' => $page->id,
                'title' => $page->title,
                'slug' => $page->slug,
                'content_blocks' => $contentBlocks,
                'seo' => $page->seo,
            ],
        ]);
    }

    /**
     * Dedicated editor for Cheap Essays Online page
     */
    public function editCheap(): Response
    {
        $page = Page::with(['contentBlocks', 'seo', 'media'])->where('slug', 'cheap-essays-online')->firstOrFail();

        // Process content blocks to ensure proper JSON structure
        $contentBlocks = $page->contentBlocks->map(function ($block) {
            return [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content, // Already cast to array by model
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        });

        return Inertia::render('Admin/CMS/Services/CheapEditor', [
            'page' => [
                'id' => $page->id,
                'title' => $page->title,
                'slug' => $page->slug,
                'content_blocks' => $contentBlocks,
                'seo' => $page->seo,
            ],
        ]);
    }

    /**
     * Show the form for creating a new page
     */
    public function create(): Response
    {
        return Inertia::render('Admin/CMS/Pages/Create', [
            'pageTypes' => $this->getPageTypes(),
            'templates' => $this->getTemplates(),
            'parentPages' => Page::select('id', 'title', 'slug')->get()
        ]);
    }

    /**
     * Store a newly created page
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug',
            'page_type' => 'required|string|in:homepage,service,blog,contact,about,landing,pricing,service_page,blog_page',
            'status' => 'required|string|in:draft,published,archived',
            'parent_id' => 'nullable|exists:pages,id',
            'template' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|integer|min:0|max:100',
            'published_at' => 'nullable|date',
            'content_blocks' => 'nullable|array',
            'content_blocks.*.block_type' => 'required|string',
            'content_blocks.*.block_key' => 'nullable|string',
            'content_blocks.*.content' => 'required|array',
            'content_blocks.*.order' => 'nullable|integer',
            'content_blocks.*.is_active' => 'nullable|boolean',
            'seo_data' => 'nullable|array',
            'media_files' => 'nullable|array',
            'use_template' => 'nullable|boolean',
        ]);

        $useTemplate = $request->boolean('use_template');
        $parentId = $validated['parent_id'] ?? null;
        if ($parentId === '') {
            $parentId = null;
        }
        $validated['parent_id'] = $parentId;
        unset($validated['use_template']);

        if ($useTemplate && ($validated['page_type'] === 'service' || $validated['page_type'] === 'service_page')) {
            try {
                $rawSlug = $validated['slug'] ?: $validated['title'];
                $slug = Str::slug($rawSlug);

                if ($slug === '') {
                    throw new \InvalidArgumentException('Unable to derive a valid slug from the provided title.');
                }

                $status = $validated['status'] ?? 'draft';
                $publishedAt = $validated['published_at'] ?? null;

                if ($status === 'published') {
                    $publishedAt = $publishedAt
                        ? Carbon::parse($publishedAt)
                        : now();
                } elseif ($publishedAt) {
                    $publishedAt = Carbon::parse($publishedAt);
                }

                $pageOverrides = array_filter([
                    'page_type' => $validated['page_type'] === 'service_page' ? 'service' : $validated['page_type'],
                    'status' => $status,
                    'template' => 'Services/DynamicService',
                    'parent_id' => $parentId,
                    'seo_priority' => $validated['seo_priority'] ?? null,
                    'published_at' => $publishedAt,
                ], static function ($value) {
                    return $value !== null;
                });

                $result = $this->servicePageScaffolder->scaffold($slug, [
                    'title' => $validated['title'],
                    'page' => $pageOverrides,
                ]);

                return redirect()
                    ->route('admin.cms.pages.edit', $result['page'])
                    ->with('success', 'Service page scaffolded using the canonical template.');
            } catch (\Throwable $e) {
                return back()->withErrors([
                    'error' => 'Failed to scaffold service page: ' . $e->getMessage(),
                ]);
            }
        }

        try {
            $page = $this->pageService->createPage(
                $validated,
                $validated['content_blocks'] ?? [],
                $validated['seo_data'] ?? [],
                $validated['media_files'] ?? []
            );

            return redirect()->route('admin.cms.pages.show', $page)
                ->with('success', 'Page created successfully.');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to create page: ' . $e->getMessage()]);
        }
    }

    /**
     * Display the specified page
     */
    public function show(Page $page): Response
    {
        $page->load(['contentBlocks', 'seo', 'media', 'relationships.relatedPage', 'parent', 'children']);

        return Inertia::render('Admin/CMS/Pages/Show', [
            'page' => $page
        ]);
    }

    /**
     * Show the form for editing the specified page
     */
    public function edit(Page $page): Response
    {
        $page->load(['contentBlocks', 'seo', 'media', 'relationships.relatedPage']);

        return Inertia::render('Admin/CMS/Pages/Edit', [
            'page' => $page,
            'pageTypes' => $this->getPageTypes(),
            'templates' => $this->getTemplates(),
            'parentPages' => Page::where('id', '!=', $page->id)->select('id', 'title', 'slug')->get()
        ]);
    }

    /**
     * Update the specified page
     */
    public function update(Request $request, Page $page)
    {
        // Log incoming data for debugging
        \Log::info('=== PAGE UPDATE START ===', [
            'page_id' => $page->id,
            'page_slug' => $page->slug,
            'has_content_blocks' => $request->has('content_blocks'),
            'content_blocks_count' => $request->has('content_blocks') ? count($request->input('content_blocks', [])) : 0,
            'has_seo_data' => $request->has('seo_data'),
            'seo_data_keys' => $request->has('seo_data') ? array_keys($request->input('seo_data', [])) : [],
        ]);

        // Log the raw SEO data received
        if ($request->has('seo_data')) {
            $seoData = $request->input('seo_data');
            \Log::info('Raw SEO data received', [
                'seo_data' => $seoData,
                'has_meta_title' => isset($seoData['meta_title']),
                'has_meta_description' => isset($seoData['meta_description']),
                'has_structured_data' => isset($seoData['structured_data']),
                'structured_data_type' => isset($seoData['structured_data']) ? gettype($seoData['structured_data']) : 'not set',
                'structured_data_value' => isset($seoData['structured_data']) ? $seoData['structured_data'] : null,
            ]);
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug,' . $page->id,
            'page_type' => 'required|string|in:homepage,service,blog,contact,about,landing,pricing,service_page,blog_page',
            'status' => 'required|string|in:draft,published,archived',
            'parent_id' => 'nullable|exists:pages,id',
            'template' => 'nullable|string|max:255',
            'seo_priority' => 'nullable|integer|min:0|max:100',
            'published_at' => 'nullable|date',
            'content_blocks' => 'nullable|array',
            'content_blocks.*.block_type' => 'required|string',
            'content_blocks.*.block_key' => 'nullable|string',
            'content_blocks.*.content' => 'required|array',
            'content_blocks.*.order' => 'nullable|integer',
            'content_blocks.*.is_active' => 'nullable|boolean',
            'seo_data' => 'nullable|array',
            'seo_data.meta_title' => 'nullable|string|max:255',
            'seo_data.meta_description' => 'nullable|string',
            'seo_data.meta_keywords' => 'nullable|string',
            'seo_data.canonical_url' => 'nullable|string',
            'seo_data.robots' => 'nullable|string',
            'seo_data.language' => 'nullable|string',
            'seo_data.og_title' => 'nullable|string',
            'seo_data.og_description' => 'nullable|string',
            'seo_data.og_image' => 'nullable|string',
            'seo_data.og_image_alt' => 'nullable|string',
            'seo_data.og_type' => 'nullable|string',
            'seo_data.og_url' => 'nullable|string',
            'seo_data.og_site_name' => 'nullable|string',
            'seo_data.og_locale' => 'nullable|string',
            'seo_data.twitter_card' => 'nullable|string',
            'seo_data.twitter_site' => 'nullable|string',
            'seo_data.twitter_creator' => 'nullable|string',
            'seo_data.twitter_title' => 'nullable|string',
            'seo_data.twitter_description' => 'nullable|string',
            'seo_data.twitter_image' => 'nullable|string',
            'seo_data.twitter_image_alt' => 'nullable|string',
            'seo_data.structured_data' => 'nullable|array',
            'media_files' => 'nullable|array'
        ]);

        \Log::info('PageController validated data', [
            'content_blocks_count' => count($validated['content_blocks'] ?? []),
            'first_block' => !empty($validated['content_blocks']) ? [
                'block_type' => $validated['content_blocks'][0]['block_type'] ?? 'N/A',
                'block_key' => $validated['content_blocks'][0]['block_key'] ?? 'N/A',
                'content_keys' => !empty($validated['content_blocks'][0]['content']) ? array_keys($validated['content_blocks'][0]['content']) : []
            ] : 'No blocks',
            'has_validated_seo_data' => isset($validated['seo_data']),
            'validated_seo_data_keys' => isset($validated['seo_data']) ? array_keys($validated['seo_data']) : [],
            'validated_structured_data' => isset($validated['seo_data']['structured_data']) ? $validated['seo_data']['structured_data'] : 'not present',
        ]);

        try {
            \Log::info('About to call PageService updatePage', [
                'page_id' => $page->id,
                'seo_data_being_passed' => $validated['seo_data'] ?? 'none',
                'seo_data_keys' => isset($validated['seo_data']) ? array_keys($validated['seo_data']) : [],
            ]);
            
            $updatedPage = $this->pageService->updatePage(
                $page,
                $validated,
                $validated['content_blocks'] ?? [],
                $validated['seo_data'] ?? [],
                $validated['media_files'] ?? []
            );

            \Log::info('PageController update successful', [
                'page_id' => $updatedPage->id,
                'content_blocks_after' => $updatedPage->contentBlocks->count(),
                'seo_after_update' => $updatedPage->seo ? $updatedPage->seo->toArray() : 'no SEO data',
            ]);

            // Clear page cache after update
            $this->pageService->clearPageCache($updatedPage->slug);

            \Log::info('=== PAGE UPDATE COMPLETE ===');

            return redirect()->route('admin.cms.pages.edit', $updatedPage)
                ->with('success', 'Page updated successfully.');

        } catch (\Exception $e) {
            \Log::error('PageController update failed', [
                'page_id' => $page->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->withErrors(['error' => 'Failed to update page: ' . $e->getMessage()]);
        }
    }

    /**
     * Remove the specified page
     */
    public function destroy(Page $page)
    {
        try {
            $this->pageService->deletePage($page);

            return redirect()->route('admin.cms.pages.index')
                ->with('success', 'Page deleted successfully.');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to delete page: ' . $e->getMessage()]);
        }
    }

    /**
     * Duplicate a page
     */
    public function duplicate(Request $request, Page $page): JsonResponse
    {
        $validated = $request->validate([
            'new_title' => 'required|string|max:255'
        ]);

        try {
            $duplicatedPage = $this->pageService->duplicatePage($page, $validated['new_title']);

            return response()->json([
                'success' => true,
                'message' => 'Page duplicated successfully',
                'page' => $duplicatedPage
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to duplicate page: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle page status
     */
    public function toggleStatus(Page $page)
    {
        try {
            $newStatus = $page->status === 'published' ? 'draft' : 'published';
            $page->update(['status' => $newStatus]);

            return back()->with('success', 'Page status updated successfully.');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to update page status: ' . $e->getMessage()]);
        }
    }

    /**
     * Get page content for frontend
     */
    public function getPageContent(string $slug): JsonResponse
    {
        $page = $this->pageService->getPageWithContent($slug);

        if (!$page) {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        }

        // Generate SEO data
        $seoData = $this->seoService->generateOpenGraphData($page->toArray(), $page->seo->toArray() ?? []);
        $twitterData = $this->seoService->generateTwitterCardData($page->toArray(), $page->seo->toArray() ?? []);

        return response()->json([
            'success' => true,
            'page' => $page,
            'seo' => $page->seo,
            'openGraph' => $seoData,
            'twitterCard' => $twitterData
        ]);
    }

    /**
     * Upload SEO/Open Graph image for pages
     */
    public function uploadSeoImage(Request $request): JsonResponse
    {
        $request->validate([
            'og_image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120', // 5MB max
        ]);

        try {
            $file = $request->file('og_image');
            $filename = 'og_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $file->getClientOriginalName());
            
            // Store in public/images/seo/ directory
            $directory = public_path('images/seo');
            if (!file_exists($directory)) {
                mkdir($directory, 0755, true);
            }
            
            $file->move($directory, $filename);
            
            // Generate the URL
            $imageUrl = '/images/seo/' . $filename;

            return response()->json([
                'success' => true,
                'message' => 'SEO image uploaded successfully',
                'image_url' => $imageUrl,
                'image_path' => 'images/seo/' . $filename
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get available page types
     */
    private function getPageTypes(): array
    {
        return [
            ['value' => 'homepage', 'label' => 'Homepage'],
            ['value' => 'service', 'label' => 'Service Page'],
            ['value' => 'blog', 'label' => 'Blog Post'],
            ['value' => 'contact', 'label' => 'Contact Page'],
            ['value' => 'about', 'label' => 'About Page'],
            ['value' => 'landing', 'label' => 'Landing Page']
        ];
    }

    /**
     * Get available templates
     */
    private function getTemplates(): array
    {
        return [
            ['value' => 'default', 'label' => 'Default Template'],
            ['value' => 'service', 'label' => 'Service Template'],
            ['value' => 'landing', 'label' => 'Landing Page Template'],
            ['value' => 'blog', 'label' => 'Blog Post Template'],
            ['value' => 'contact', 'label' => 'Contact Page Template']
        ];
    }

    /**
     * Calculate average SEO score across all pages
     */
    private function calculateAverageSeoScore(): int
    {
        $pagesWithSeo = Page::whereHas('seo')->count();
        if ($pagesWithSeo === 0) {
            return 0;
        }

        // Simple calculation based on pages with complete SEO data
        $pagesWithCompleteSeo = Page::whereHas('seo', function ($query) {
            $query->whereNotNull('meta_title')
                  ->whereNotNull('meta_description')
                  ->whereNotNull('canonical_url');
        })->count();

        return (int) round(($pagesWithCompleteSeo / $pagesWithSeo) * 100);
    }
}
