<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use Inertia\Inertia;
use App\Models\Payment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class PaymentAnalyticsController extends Controller
{
    /**
     * Time in seconds to cache dashboard data
     */
    const CACHE_DURATION = 300; // 5 minutes

    /**
     * Display payment analytics dashboard
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $this->authorize('viewPaymentAnalytics', Payment::class);

        // Parse date filters with defaults
        $endDate = $request->input('end_date')
            ? Carbon::parse($request->input('end_date'))
            : Carbon::now();

        $startDate = $request->input('start_date')
            ? Carbon::parse($request->input('start_date'))
            : $endDate->copy()->subDays(30);

        $period = $request->input('period', 'monthly');

        // Ensure valid dates
        if ($startDate->gt($endDate)) {
            $startDate = $endDate->copy()->subDays(30);
        }

        // Cache key includes filters for proper cache invalidation
        $cacheKey = "payment_analytics_{$startDate->format('Y-m-d')}_{$endDate->format('Y-m-d')}_{$period}";

        // Get analytics data with caching
        $analyticsData = Cache::remember($cacheKey, self::CACHE_DURATION, function () use ($startDate, $endDate, $period) {
            return [
                'summary' => $this->getSummaryMetrics($startDate, $endDate),
                'trends' => $this->getPaymentTrends($startDate, $endDate, $period),
                'paymentMethods' => $this->getPaymentMethodBreakdown($startDate, $endDate),
                'recentTransactions' => $this->getRecentTransactions(),
                'topClients' => $this->getTopClients($startDate, $endDate),
                'statusDistribution' => $this->getStatusDistribution($startDate, $endDate),
            ];
        });

        return Inertia::render('Admin/Payments/Analytics', [
            'analytics' => $analyticsData,
            'filters' => [
                'startDate' => $startDate->format('Y-m-d'),
                'endDate' => $endDate->format('Y-m-d'),
                'period' => $period,
            ],
        ]);
    }

    /**
     * Get summary metrics
     *
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return array
     */
    private function getSummaryMetrics(Carbon $startDate, Carbon $endDate): array
    {
        // Total payments in period
        $totalPayments = Payment::whereBetween('payments.created_at', [$startDate, $endDate])
            ->where('payments.status', Payment::STATUS_COMPLETED)
            ->count();

        // Total revenue in period
        $totalRevenue = Payment::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', Payment::STATUS_COMPLETED)
            ->sum('amount');

        // Failed transactions
        $failedPayments = Payment::whereBetween('created_at', [$startDate, $endDate])
            ->where('status', Payment::STATUS_FAILED)
            ->count();

        // Success rate
        $allPaymentAttempts = Payment::whereBetween('created_at', [$startDate, $endDate])
            ->whereIn('status', [Payment::STATUS_COMPLETED, Payment::STATUS_FAILED])
            ->count();

        $successRate = $allPaymentAttempts > 0
            ? round(($totalPayments / $allPaymentAttempts) * 100, 2)
            : 0;

        // Growth comparison with previous period
        $previousStartDate = $startDate->copy()->subDays($endDate->diffInDays($startDate));
        $previousEndDate = $startDate->copy()->subDay();

        $previousRevenue = Payment::whereBetween('created_at', [$previousStartDate, $previousEndDate])
            ->where('status', Payment::STATUS_COMPLETED)
            ->sum('amount');

        $revenueGrowth = $previousRevenue > 0
            ? round((($totalRevenue - $previousRevenue) / $previousRevenue) * 100, 2)
            : 100;

        return [
            'totalPayments' => $totalPayments,
            'totalRevenue' => $totalRevenue,
            'failedPayments' => $failedPayments,
            'successRate' => $successRate,
            'revenueGrowth' => $revenueGrowth,
            'previousPeriodRevenue' => $previousRevenue,
        ];
    }

    /**
     * Get payment trends over time
     *
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @param string $period
     * @return array
     */
    private function getPaymentTrends(Carbon $startDate, Carbon $endDate, string $period): array
    {
        $format = $period === 'daily' ? 'Y-m-d' : 'Y-m';
        $groupBy = $period === 'daily' ? 'date' : 'month';

        // Get payment amounts by date
        $trends = Payment::whereBetween('payments.created_at', [$startDate, $endDate])
            ->where('payments.status', Payment::STATUS_COMPLETED)
            ->select(
                DB::raw("DATE_FORMAT(created_at, '%" . ($period === 'daily' ? 'Y-%m-%d' : 'Y-%m') . "') as {$groupBy}"),
                DB::raw('SUM(amount) as revenue'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy($groupBy)
            ->orderBy($groupBy)
            ->get();

        // Format results
        $labels = [];
        $revenue = [];
        $count = [];

        // Fill in gaps in the data with zeros
        $currentDate = $startDate->copy();
        while ($currentDate->lte($endDate)) {
            $key = $currentDate->format($format);
            $labels[] = $key;

            $dataPoint = $trends->firstWhere($groupBy, $key);
            $revenue[] = $dataPoint ? floatval($dataPoint->revenue) : 0;
            $count[] = $dataPoint ? intval($dataPoint->count) : 0;

            if ($period === 'daily') {
                $currentDate->addDay();
            } else {
                $currentDate->addMonth();
            }
        }

        return [
            'labels' => $labels,
            'revenue' => $revenue,
            'count' => $count,
        ];
    }

    /**
     * Get payment method breakdown
     *
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return array
     */
    private function getPaymentMethodBreakdown(Carbon $startDate, Carbon $endDate): array
    {
        $paymentMethods = Payment::whereBetween('payments.created_at', [$startDate, $endDate])
            ->where('payments.status', Payment::STATUS_COMPLETED)
            ->select(
                'payment_method',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(amount) as revenue')
            )
            ->groupBy('payment_method')
            ->get();

        return $paymentMethods->map(function ($item) {
            return [
                'name' => $item->payment_method,
                'count' => $item->count,
                'revenue' => floatval($item->revenue),
            ];
        })->toArray();
    }

    /**
     * Get recent transactions
     *
     * @param int $limit
     * @return array
     */
    private function getRecentTransactions(int $limit = 10): array
    {
        return Payment::with(['order', 'order.user'])
            ->latest()
            ->limit($limit)
            ->get()
            ->map(function ($payment) {
                return [
                    'id' => $payment->id,
                    'date' => $payment->created_at,
                    'amount' => $payment->amount,
                    'status' => $payment->status,
                    'method' => $payment->payment_method,
                    'order_id' => $payment->order_id,
                    'order_number' => $payment->order ? $payment->order->order_number : null,
                    'client_name' => $payment->order && $payment->order->user ? $payment->order->user->name : 'Unknown',
                    'transaction_id' => $payment->transaction_id,
                ];
            })->toArray();
    }

    /**
     * Get top clients by payment volume
     *
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @param int $limit
     * @return array
     */
    private function getTopClients(Carbon $startDate, Carbon $endDate, int $limit = 5): array
    {
        return Payment::whereBetween('payments.created_at', [$startDate, $endDate])  // Specify table name
            ->where('payments.status', Payment::STATUS_COMPLETED)  // Specify table name
            ->join('orders', 'payments.order_id', '=', 'orders.id')
            ->join('users', 'orders.user_id', '=', 'users.id')
            ->select(
                'users.id',
                'users.name',
                DB::raw('COUNT(payments.id) as payment_count'),
                DB::raw('SUM(payments.amount) as total_spent')
            )
            ->groupBy('users.id', 'users.name')
            ->orderBy('total_spent', 'desc')
            ->limit($limit)
            ->get()
            ->map(function ($client) {
                return [
                    'id' => $client->id,
                    'name' => $client->name,
                    'payment_count' => $client->payment_count,
                    'total_spent' => floatval($client->total_spent),
                ];
            })->toArray();
    }


    /**
     * Get payment status distribution
     *
     * @param Carbon $startDate
     * @param Carbon $endDate
     * @return array
     */
    private function getStatusDistribution(Carbon $startDate, Carbon $endDate): array
    {
        $statuses = Payment::whereBetween('payments.created_at', [$startDate, $endDate])
            ->select(
                'payments.status',
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('status')
            ->get();

        return $statuses->map(function ($item) {
            return [
                'status' => $item->status,
                'count' => $item->count,
            ];
        })->toArray();
    }
}
