<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\Payments\PaymentServiceFactory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class PaymentController extends Controller
{
    /**
     * Display a listing of all payments.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $query = Payment::with('order.user')
            ->orderBy('created_at', 'desc');

        // Apply filters if any
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        if ($request->has('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('transaction_id', 'like', "%{$request->search}%")
                    ->orWhereHas('order', function ($q2) use ($request) {
                        $q2->where('order_number', 'like', "%{$request->search}%");
                    });
            });
        }

        $payments = $query->paginate(15);

        return Inertia::render('Admin/Payments/Index', [
            'payments' => $payments,
            'filters' => $request->only(['status', 'payment_method', 'search']),
            'paymentMethods' => PaymentMethod::select('name', 'display_name')->get(),
        ]);
    }

    /**
     * Display pending payments.
     *
     * @return \Inertia\Response
     */
    public function pending()
    {
        $pendingPayments = Payment::with('order.user')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return Inertia::render('Admin/Payments/Pending', [
            'pendingPayments' => $pendingPayments,
        ]);
    }

    /**
     * Display failed payments.
     *
     * @return \Inertia\Response
     */
    public function failed()
    {
        $failedPayments = Payment::with('order.user')
            ->where('status', 'failed')
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return Inertia::render('Admin/Payments/Failed', [
            'failedPayments' => $failedPayments,
        ]);
    }

    /**
     * Display the specified payment details.
     *
     * @param  \App\Models\Payment  $payment
     * @return \Inertia\Response
     */
    public function show(Payment $payment)
    {
        $payment->load('order.user');

        return Inertia::render('Admin/Payments/Show', [
            'payment' => $payment,
        ]);
    }

    /**
     * Update payment status manually.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Payment  $payment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateStatus(Request $request, Payment $payment)
    {
        $validated = $request->validate([
            'status' => 'required|string|in:completed,pending,failed,refunded',
            'admin_notes' => 'nullable|string',
        ]);

        $oldStatus = $payment->status;
        $payment->status = $validated['status'];

        // Update metadata with admin notes
        $metadata = json_decode($payment->metadata, true) ?: [];
        $metadata['admin_notes'] = $validated['admin_notes'] ?? null;
        $metadata['status_updated_by_admin'] = true;
        $metadata['status_updated_at'] = now()->toDateTimeString();
        $payment->metadata = json_encode($metadata);

        $payment->save();

        // Log the status change
        Log::info('Payment status updated by admin', [
            'payment_id' => $payment->id,
            'old_status' => $oldStatus,
            'new_status' => $payment->status,
            'updated_by' => $request->user()->id,
        ]);

        return redirect()->back()
            ->with('success', 'Payment status updated successfully.');
    }

    /**
     * Process a refund.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Payment  $payment
     * @return \Illuminate\Http\JsonResponse
     */
    public function processRefund(Request $request, Payment $payment)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01|max:' . $payment->amount,
            'reason' => 'required|string',
        ]);

        try {
            // Get payment service
            $paymentService = PaymentServiceFactory::create($payment->payment_method);

            // Process refund
            $result = $paymentService->processRefund($payment, $validated['amount'], $validated['reason']);

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'message' => 'Refund processed successfully.',
                    'data' => $result['data'] ?? null,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result['error'] ?? 'Refund processing failed.',
                ], 400);
            }
        } catch (\Exception $e) {
            Log::error('Refund processing error', [
                'payment_id' => $payment->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing the refund: ' . $e->getMessage(),
            ], 500);
        }
    }
}
