<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use Inertia\Inertia;
use App\Models\Payment;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Response;
use League\Csv\Writer;

class PaymentHistoryController extends Controller
{
    /**
     * Display a paginated list of payment transactions.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', Payment::class);

        // Get filter parameters
        $search = $request->input('search');
        $status = $request->input('status');
        $method = $request->input('method');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        $perPage = $request->input('per_page', 15);

        // Query payments with filters
        $paymentsQuery = Payment::query()
            ->with(['order', 'order.user'])
            ->when($search, function ($query, $search) {
                return $query->where(function ($q) use ($search) {
                    $q->where('transaction_id', 'like', "%{$search}%")
                        ->orWhereHas('order', function ($orderQuery) use ($search) {
                            $orderQuery->where('order_number', 'like', "%{$search}%");
                        })
                        ->orWhereHas('order.user', function ($userQuery) use ($search) {
                            $userQuery->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%");
                        });
                });
            })
            ->when($status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($method, function ($query, $method) {
                return $query->where('payment_method', $method);
            })
            ->when($startDate, function ($query, $startDate) {
                return $query->where('created_at', '>=', Carbon::parse($startDate)->startOfDay());
            })
            ->when($endDate, function ($query, $endDate) {
                return $query->where('created_at', '<=', Carbon::parse($endDate)->endOfDay());
            })
            ->latest();

        // Get unique payment methods for filter dropdown
        $paymentMethods = Payment::select('payment_method')
            ->distinct()
            ->pluck('payment_method');

        // Paginate the results
        $payments = $paymentsQuery->paginate($perPage)
            ->withQueryString()
            ->through(function ($payment) {
                return [
                    'id' => $payment->id,
                    'transaction_id' => $payment->transaction_id,
                    'amount' => $payment->amount,
                    'currency' => $payment->currency ?? 'USD',
                    'status' => $payment->status,
                    'payment_method' => $payment->payment_method,
                    'created_at' => $payment->created_at,
                    'payment_date' => $payment->payment_date,
                    'order_id' => $payment->order_id,
                    'order_number' => $payment->order->order_number ?? null,
                    'client' => $payment->order && $payment->order->user ? [
                        'id' => $payment->order->user->id,
                        'name' => $payment->order->user->name,
                        'email' => $payment->order->user->email,
                    ] : null,
                    'metadata' => $payment->metadata,
                ];
            });

        return Inertia::render('Admin/Payments/History', [
            'payments' => $payments,
            'filters' => [
                'search' => $search,
                'status' => $status,
                'method' => $method,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'per_page' => $perPage,
            ],
            'paymentMethods' => $paymentMethods,
            'statuses' => [
                Payment::STATUS_PENDING,
                Payment::STATUS_COMPLETED,
                Payment::STATUS_FAILED,
                Payment::STATUS_REFUNDED,
            ],
        ]);
    }

    /**
     * Display detailed information for a specific payment.
     *
     * @param Payment $payment
     * @return \Inertia\Response
     */
    public function show(Payment $payment)
    {
        $this->authorize('view', $payment);

        $payment->load(['order', 'order.user']);

        return Inertia::render('Admin/Payments/Detail', [
            'payment' => [
                'id' => $payment->id,
                'transaction_id' => $payment->transaction_id,
                'amount' => $payment->amount,
                'currency' => $payment->currency ?? 'USD',
                'status' => $payment->status,
                'payment_method' => $payment->payment_method,
                'created_at' => $payment->created_at,
                'payment_date' => $payment->payment_date,
                'metadata' => $payment->metadata,
                'order' => $payment->order ? [
                    'id' => $payment->order->id,
                    'order_number' => $payment->order->order_number,
                    'title' => $payment->order->title,
                    'order_status' => $payment->order->order_status,
                    'payment_status' => $payment->order->payment_status,
                ] : null,
                'client' => $payment->order && $payment->order->user ? [
                    'id' => $payment->order->user->id,
                    'name' => $payment->order->user->name,
                    'email' => $payment->order->user->email,
                ] : null,
            ],
        ]);
    }

    /**
     * Export payment history to CSV.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function export(Request $request)
    {
        $this->authorize('export', Payment::class);

        // Get filter parameters (same as index method)
        $search = $request->input('search');
        $status = $request->input('status');
        $method = $request->input('method');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        // Query payments with filters (without pagination)
        $payments = Payment::with(['order', 'order.user'])
            ->when($search, function ($query, $search) {
                return $query->where(function ($q) use ($search) {
                    $q->where('transaction_id', 'like', "%{$search}%")
                        ->orWhereHas('order', function ($orderQuery) use ($search) {
                            $orderQuery->where('order_number', 'like', "%{$search}%");
                        })
                        ->orWhereHas('order.user', function ($userQuery) use ($search) {
                            $userQuery->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%");
                        });
                });
            })
            ->when($status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($method, function ($query, $method) {
                return $query->where('payment_method', $method);
            })
            ->when($startDate, function ($query, $startDate) {
                return $query->where('created_at', '>=', Carbon::parse($startDate)->startOfDay());
            })
            ->when($endDate, function ($query, $endDate) {
                return $query->where('created_at', '<=', Carbon::parse($endDate)->endOfDay());
            })
            ->latest()
            ->get();

        // Create CSV
        $csv = Writer::createFromString('');

        // Add headers
        $csv->insertOne([
            'ID',
            'Transaction ID',
            'Amount',
            'Currency',
            'Status',
            'Payment Method',
            'Date',
            'Order Number',
            'Client Name',
            'Client Email'
        ]);

        // Add payment data
        foreach ($payments as $payment) {
            $csv->insertOne([
                $payment->id,
                $payment->transaction_id,
                $payment->amount,
                $payment->currency ?? 'USD',
                $payment->status,
                $payment->payment_method,
                $payment->created_at->format('Y-m-d H:i:s'),
                $payment->order->order_number ?? 'N/A',
                $payment->order->user->name ?? 'N/A',
                $payment->order->user->email ?? 'N/A',
            ]);
        }

        // Generate filename
        $filename = 'payment_history_' . Carbon::now()->format('Y-m-d_His') . '.csv';

        // Return CSV as download
        return Response::make($csv->toString(), 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}
