<?php

namespace App\Http\Controllers\Admin;

use Inertia\Inertia;
use App\Models\Payment;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Services\Payments\PaymentServiceFactory;

class PaymentMethodController extends Controller
{
    /**
     * Display a listing of payment methods.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        $paymentMethods = PaymentMethod::all();

        return Inertia::render('Admin/PaymentMethods/Index', [
            'paymentMethods' => $paymentMethods->map(function ($method) {
                return [
                    'id' => $method->id,
                    'name' => $method->name,
                    'display_name' => $method->display_name,
                    'is_active' => $method->is_active,
                    'created_at' => $method->created_at,
                    'updated_at' => $method->updated_at,
                ];
            }),
        ]);
    }

    /**
     * Show the form for creating a new payment method.
     *
     * @return \Inertia\Response
     */
    public function create()
    {
        $this->authorize('create', PaymentMethod::class);
        try {
            // Get available payment service types
            $availableServiceTypes = [
                PaymentMethod::PAYPAL => 'PayPal',
                PaymentMethod::STRIPE => 'Stripe',
                PaymentMethod::PAYPAL_DIRECT => 'PayPal Direct',
            ];

            return Inertia::render('Admin/PaymentMethods/Create', [
                'availableServiceTypes' => $availableServiceTypes
            ]);
        } catch (\Exception $e) {
            Log::error('Error loading payment method creation form', [
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Failed to load payment method creation form.');
        }
    }

    /**
     * Store a newly created payment method.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->authorize('create', PaymentMethod::class);
        $validated = $request->validate([
            'name' => 'required|string|unique:payment_methods,name',
            'display_name' => 'required|string',
            'is_active' => 'boolean',
            'config_data' => 'required|array',
        ]);

        // $validated['config_data'] = json_encode($validated['config_data']);

        PaymentMethod::create($validated);

        return redirect()->route('admin.payment-methods.index')
            ->with('success', 'Payment method created successfully.');
    }

    /**
     * Show the form for editing the specified payment method.
     *
     * @param  \App\Models\PaymentMethod  $paymentMethod
     * @return \Inertia\Response
     */
    public function edit(PaymentMethod $paymentMethod)
    {
        $this->authorize('update', $paymentMethod);
        return Inertia::render('Admin/PaymentMethods/Edit', [
            'paymentMethod' => [
                'id' => $paymentMethod->id,
                'name' => $paymentMethod->name,
                'display_name' => $paymentMethod->display_name,
                'is_active' => $paymentMethod->is_active,
                'config_data' => $paymentMethod->config_data,
            ],
        ]);
    }

    /**
     * Update the specified payment method.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PaymentMethod  $paymentMethod
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, PaymentMethod $paymentMethod)
    {

        $this->authorize('update', $paymentMethod);
        $validated = $request->validate([
            'display_name' => 'required|string',
            'is_active' => 'boolean',
            'config_data' => 'required|array',
        ]);

        // $validated['config_data'] = json_encode($validated['config_data']);

        $paymentMethod->update($validated);

        return redirect()->route('admin.payment-methods.index')
            ->with('success', 'Payment method updated successfully.');
    }

    /**
     * Toggle the active status of a payment method.
     *
     * @param  \App\Models\PaymentMethod  $paymentMethod
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleActive($id)
    {
        $paymentMethod = PaymentMethod::findOrFail($id);
        $this->authorize('toggleStatus', $paymentMethod);

        try {
            $paymentMethod->is_active = !$paymentMethod->is_active;
            $paymentMethod->save();

            return response()->json([
                'success' => true,
                'is_active' => $paymentMethod->is_active,
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling payment method status', [
                'payment_method_id' => $id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle payment method status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test connection with the payment gateway.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PaymentMethod  $paymentMethod
     * @return \Illuminate\Http\JsonResponse
     */
    public function testConnection(Request $request, $id)
    {
        try {
            // Explicitly find the payment method by ID
            $paymentMethod = PaymentMethod::findOrFail($id);
            $this->authorize('testConnection', $paymentMethod);

            // Now we can safely use $paymentMethod->name
            $paymentService = PaymentServiceFactory::create($paymentMethod->name);

            // Test connection
            $result = $paymentService->testConnection();

            return response()->json([
                'success' => true,
                'message' => 'Connection successful!',
                'details' => $result,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Connection failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified payment method from storage.
     *
     * @param  \App\Models\PaymentMethod  $paymentMethod
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(PaymentMethod $paymentMethod)
    {
        try {
            $this->authorize('delete', $paymentMethod);

            // Check if the payment method is in use
            $paymentsCount = Payment::where('payment_method', $paymentMethod->name)->count();
            if ($paymentsCount > 0) {
                return redirect()->route('admin.payment-methods.index')
                    ->with('message', 'This payment method cannot be deleted because it has associated payment records. Consider deactivating it instead.')
                    ->with('type', 'error'); // Add a type to distinguish between success and error messages
            }

            // Store the name for the success message
            $name = $paymentMethod->display_name;

            // Delete the payment method
            $paymentMethod->delete();

            return redirect()->route('admin.payment-methods.index')
                ->with('success', "Payment method '{$name}' has been deleted successfully.");
        } catch (\Exception $e) {
            Log::error('Error deleting payment method', [
                'id' => $paymentMethod->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->route('admin.payment-methods.index')
                ->with('error', 'Failed to delete payment method: ' . $e->getMessage());
        }
    }
}
