<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterInfraction;
use App\Models\WriterFine;
use App\Models\User;
use App\Services\FineCalculationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class ProgressiveDisciplineController extends Controller
{
    protected $fineCalculationService;

    public function __construct(FineCalculationService $fineCalculationService)
    {
        $this->fineCalculationService = $fineCalculationService;
    }

    /**
     * Display the progressive discipline dashboard
     */
    public function index(Request $request)
    {
        $query = WriterInfraction::with(['writer', 'order', 'reportedBy'])
            ->orderBy('created_at', 'desc');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by severity
        if ($request->filled('severity')) {
            $query->where('severity_level', $request->severity);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('infraction_type', $request->type);
        }

        // Filter by escalated status
        if ($request->filled('escalated')) {
            $query->where('escalated', $request->escalated === 'true');
        }

        $infractions = $query->paginate(20);

        $statistics = $this->getDisciplineStatistics();

        return view('admin.progressive-discipline.index', compact('infractions', 'statistics'));
    }

    /**
     * Show infraction details and management options
     */
    public function show(WriterInfraction $infraction)
    {
        $infraction->load(['writer', 'order', 'reportedBy', 'reviewedBy', 'fines']);
        
        $relatedInfractions = WriterInfraction::where('writer_id', $infraction->writer_id)
            ->where('id', '!=', $infraction->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $fineStatistics = $this->fineCalculationService->getWriterFineStatistics($infraction->writer_id);

        return view('admin.progressive-discipline.show', compact(
            'infraction', 
            'relatedInfractions', 
            'fineStatistics'
        ));
    }

    /**
     * Review and confirm an infraction
     */
    public function review(Request $request, WriterInfraction $infraction)
    {
        $request->validate([
            'status' => ['required', Rule::in(['confirmed', 'dismissed', 'under_review'])],
            'admin_notes' => 'nullable|string|max:1000',
            'severity_level' => ['nullable', Rule::in(['warning', 'minor', 'major', 'critical'])],
            'estimated_cost_impact' => 'nullable|numeric|min:0',
            'client_satisfaction_impact' => 'nullable|integer|min:1|max:5',
        ]);

        try {
            DB::beginTransaction();

            $infraction->update([
                'status' => $request->status,
                'reviewed_by' => auth()->id(),
                'reviewed_at' => now(),
                'admin_notes' => $request->admin_notes,
                'severity_level' => $request->severity_level ?? $infraction->severity_level,
                'estimated_cost_impact' => $request->estimated_cost_impact,
                'client_satisfaction_impact' => $request->client_satisfaction_impact,
            ]);

            // If confirmed, check if it should trigger automatic fine
            if ($request->status === 'confirmed' && $this->fineCalculationService->shouldTriggerAutomaticFine($infraction)) {
                $fine = $this->fineCalculationService->createAutomaticFine($infraction);
                
                if ($fine) {
                    Log::info('Automatic fine created during review', [
                        'infraction_id' => $infraction->id,
                        'fine_id' => $fine->id
                    ]);
                }
            }

            // Check if escalation is needed
            if ($infraction->shouldTriggerEscalation()) {
                $this->escalateInfraction($infraction);
            }

            DB::commit();

            return redirect()->route('admin.progressive-discipline.show', $infraction)
                ->with('success', 'Infraction reviewed successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to review infraction', [
                'infraction_id' => $infraction->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to review infraction. Please try again.');
        }
    }

    /**
     * Escalate an infraction to higher severity
     */
    public function escalate(Request $request, WriterInfraction $infraction)
    {
        $request->validate([
            'escalation_notes' => 'required|string|max:1000',
            'new_severity_level' => ['required', Rule::in(['minor', 'major', 'critical'])],
        ]);

        try {
            DB::beginTransaction();

            $oldSeverity = $infraction->severity_level;
            
            $infraction->update([
                'severity_level' => $request->new_severity_level,
                'escalated' => true,
                'escalation_notes' => $request->escalation_notes,
            ]);

            // Create new fine if escalation triggers automatic fine
            if ($this->fineCalculationService->shouldTriggerAutomaticFine($infraction)) {
                $fine = $this->fineCalculationService->createAutomaticFine($infraction);
                
                if ($fine) {
                    Log::info('Fine created due to escalation', [
                        'infraction_id' => $infraction->id,
                        'fine_id' => $fine->id,
                        'old_severity' => $oldSeverity,
                        'new_severity' => $request->new_severity_level
                    ]);
                }
            }

            // Apply progressive discipline measures
            $this->applyProgressiveDiscipline($infraction);

            DB::commit();

            return redirect()->route('admin.progressive-discipline.show', $infraction)
                ->with('success', 'Infraction escalated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to escalate infraction', [
                'infraction_id' => $infraction->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to escalate infraction. Please try again.');
        }
    }

    /**
     * Apply progressive discipline measures
     */
    private function applyProgressiveDiscipline(WriterInfraction $infraction): void
    {
        $writer = $infraction->writer;
        $repeatCount = $infraction->repeat_count;

        // Warning level - no restrictions
        if ($infraction->severity_level === 'warning') {
            return;
        }

        // Minor level - restrict to lower-value orders
        if ($infraction->severity_level === 'minor' && $repeatCount >= 2) {
            $writer->update([
                'max_order_value' => 100, // Restrict to orders under $100
                'restriction_notes' => 'Restricted due to minor infractions'
            ]);
        }

        // Major level - probationary period
        if ($infraction->severity_level === 'major' && $repeatCount >= 2) {
            $writer->update([
                'account_status' => 'probationary',
                'probation_start_date' => now(),
                'probation_end_date' => now()->addDays(30),
                'restriction_notes' => 'Probationary period due to major infractions'
            ]);
        }

        // Critical level - suspension
        if ($infraction->severity_level === 'critical') {
            $writer->update([
                'account_status' => 'suspended',
                'suspension_start_date' => now(),
                'suspension_end_date' => now()->addDays(90),
                'restriction_notes' => 'Account suspended due to critical infraction'
            ]);
        }
    }

    /**
     * Handle infraction appeals
     */
    public function handleAppeal(Request $request, WriterInfraction $infraction)
    {
        $request->validate([
            'appeal_decision' => ['required', Rule::in(['approved', 'rejected'])],
            'appeal_decision_notes' => 'required|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $infraction->update([
                'status' => $request->appeal_decision === 'approved' ? 'appeal_approved' : 'appeal_rejected',
                'appeal_decision_notes' => $request->appeal_decision_notes,
                'resolved_at' => now(),
            ]);

            // If appeal is approved, reverse any fines
            if ($request->appeal_decision === 'approved') {
                $this->reverseFines($infraction);
            }

            DB::commit();

            return redirect()->route('admin.progressive-discipline.show', $infraction)
                ->with('success', 'Appeal handled successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to handle appeal', [
                'infraction_id' => $infraction->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to handle appeal. Please try again.');
        }
    }

    /**
     * Reverse fines when appeal is approved
     */
    private function reverseFines(WriterInfraction $infraction): void
    {
        $fines = $infraction->fines()->where('status', 'applied')->get();

        foreach ($fines as $fine) {
            $fine->update([
                'status' => 'reversed',
                'review_notes' => 'Fine reversed due to approved appeal',
                'resolved_at' => now(),
            ]);
        }
    }

    /**
     * Get discipline statistics for dashboard
     */
    private function getDisciplineStatistics(): array
    {
        $totalInfractions = WriterInfraction::count();
        $pendingReview = WriterInfraction::whereIn('status', ['reported', 'under_review'])->count();
        $escalated = WriterInfraction::where('escalated', true)->count();
        $appealsPending = WriterInfraction::where('status', 'appeal_submitted')->count();

        $severityBreakdown = WriterInfraction::selectRaw('severity_level, COUNT(*) as count')
            ->groupBy('severity_level')
            ->pluck('count', 'severity_level')
            ->toArray();

        $typeBreakdown = WriterInfraction::selectRaw('infraction_type, COUNT(*) as count')
            ->groupBy('infraction_type')
            ->pluck('count', 'infraction_type')
            ->toArray();

        return [
            'total_infractions' => $totalInfractions,
            'pending_review' => $pendingReview,
            'escalated' => $escalated,
            'appeals_pending' => $appealsPending,
            'severity_breakdown' => $severityBreakdown,
            'type_breakdown' => $typeBreakdown,
        ];
    }

    /**
     * Get writers with multiple infractions
     */
    public function repeatOffenders()
    {
        $repeatOffenders = WriterInfraction::selectRaw('writer_id, COUNT(*) as infraction_count')
            ->groupBy('writer_id')
            ->having('infraction_count', '>', 1)
            ->orderBy('infraction_count', 'desc')
            ->with('writer')
            ->paginate(20);

        return view('admin.progressive-discipline.repeat-offenders', compact('repeatOffenders'));
    }

    /**
     * Get performance improvement plans
     */
    public function performancePlans()
    {
        $writersOnProbation = User::where('account_status', 'probationary')
            ->where('role', 'writer')
            ->with(['infractions' => function ($query) {
                $query->orderBy('created_at', 'desc');
            }])
            ->paginate(20);

        return view('admin.progressive-discipline.performance-plans', compact('writersOnProbation'));
    }

    /**
     * Update performance improvement plan
     */
    public function updatePerformancePlan(Request $request, User $writer)
    {
        $request->validate([
            'probation_end_date' => 'required|date|after:today',
            'restriction_notes' => 'nullable|string|max:1000',
            'account_status' => ['required', Rule::in(['probationary', 'active', 'suspended'])],
        ]);

        try {
            $writer->update([
                'probation_end_date' => $request->probation_end_date,
                'restriction_notes' => $request->restriction_notes,
                'account_status' => $request->account_status,
            ]);

            return redirect()->route('admin.progressive-discipline.performance-plans')
                ->with('success', 'Performance plan updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to update performance plan', [
                'writer_id' => $writer->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Failed to update performance plan. Please try again.');
        }
    }

    /**
     * Export discipline data
     */
    public function export(Request $request)
    {
        $query = WriterInfraction::with(['writer', 'order', 'reportedBy'])
            ->orderBy('created_at', 'desc');

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $infractions = $query->get();

        // Generate CSV export
        $filename = 'discipline_report_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($infractions) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'ID', 'Writer', 'Order', 'Type', 'Severity', 'Status', 
                'Reported By', 'Reported At', 'Resolved At', 'Notes'
            ]);

            foreach ($infractions as $infraction) {
                fputcsv($file, [
                    $infraction->id,
                    $infraction->writer->name ?? 'N/A',
                    $infraction->order->order_number ?? 'N/A',
                    $infraction->infraction_type,
                    $infraction->severity_level,
                    $infraction->status,
                    $infraction->reportedBy->name ?? 'N/A',
                    $infraction->reported_at?->format('Y-m-d H:i:s') ?? 'N/A',
                    $infraction->resolved_at?->format('Y-m-d H:i:s') ?? 'N/A',
                    $infraction->admin_notes ?? 'N/A'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
} 