<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Services\RefundService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RefundController extends Controller
{
    /**
     * @var RefundService
     */
    protected $refundService;

    /**
     * Create a new controller instance.
     *
     * @param RefundService $refundService
     * @return void
     */
    public function __construct(RefundService $refundService)
    {
        $this->refundService = $refundService;
    }

    /**
     * Process a refund request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $paymentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function process(Request $request, $paymentId)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'reason' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $payment = Payment::findOrFail($paymentId);

        // Authorization check
        $this->authorize('refund', $payment);

        // Process the refund using our RefundService
        $result = $this->refundService->processRefund(
            $payment,
            (float) $validated['amount'],
            $validated['reason'],
            $validated['notes'],
            Auth::id()
        );

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => $result['message'],
                'refund' => [
                    'id' => $result['refund']->id,
                    'amount' => $result['refund']->amount,
                    'status' => $result['refund']->status,
                    'processed_at' => $result['refund']->processed_at,
                ],
                'payment' => [
                    'id' => $result['payment']->id,
                    'status' => $result['payment']->status,
                    'refunded_amount' => $result['payment']->getRefundedAmount(),
                    'refundable_amount' => $result['payment']->getRefundableAmount()
                ],
                'order' => $result['order'] ? [
                    'id' => $result['order']->id,
                    'payment_status' => $result['order']->payment_status
                ] : null
            ]);
        } else {
            $status = isset($result['error_type']) && $result['error_type'] === 'validation' ? 422 : 500;

            return response()->json([
                'success' => false,
                'message' => $result['message'],
                'refund_id' => $result['refund_id'] ?? null,
                'error_type' => $result['error_type'] ?? 'unknown'
            ], $status);
        }
    }

    /**
     * Get refund history for a payment.
     *
     * @param  int  $paymentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function history($paymentId)
    {
        $payment = Payment::findOrFail($paymentId);

        // Authorization check
        $this->authorize('view', $payment);

        // Get refund history from the RefundService
        $history = $this->refundService->getRefundHistory($payment);

        return response()->json(array_merge(['success' => true], $history));
    }
}
