<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\User;
use App\Services\RevisionService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Inertia\Response;

class RevisionController extends Controller
{
    protected $revisionService;

    public function __construct(RevisionService $revisionService)
    {
        $this->revisionService = $revisionService;
        $this->middleware(['auth', 'admin']);
    }

    /**
     * Display a listing of revisions.
     */
    public function index(Request $request): Response
    {
        $query = OrderRevision::with(['order', 'requestedBy', 'assignedTo'])
            ->orderBy('created_at', 'desc');

        // Apply filters
        if ($request->filled('status')) {
            $query->where('revision_status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->where('revision_priority', $request->priority);
        }

        if ($request->filled('type')) {
            $query->where('revision_type', $request->type);
        }

        if ($request->filled('overdue')) {
            if ($request->overdue === 'true') {
                $query->overdue();
            }
        }

        $revisions = $query->paginate(20);

        return Inertia::render('Admin/Revisions/Index', [
            'revisions' => $revisions,
            'filters' => $request->only(['status', 'priority', 'type', 'overdue']),
            'statuses' => [
                OrderRevision::STATUS_REQUESTED => 'Requested',
                OrderRevision::STATUS_ACKNOWLEDGED => 'Acknowledged',
                OrderRevision::STATUS_IN_PROGRESS => 'In Progress',
                OrderRevision::STATUS_SUBMITTED => 'Submitted',
                OrderRevision::STATUS_REVIEWED => 'Reviewed',
            ],
            'priorities' => [
                OrderRevision::PRIORITY_LOW => 'Low',
                OrderRevision::PRIORITY_MEDIUM => 'Medium',
                OrderRevision::PRIORITY_HIGH => 'High',
            ],
            'types' => [
                OrderRevision::REVISION_TYPE_CONTENT => 'Content',
                OrderRevision::REVISION_TYPE_FORMATTING => 'Formatting',
                OrderRevision::REVISION_TYPE_RESEARCH => 'Research',
                OrderRevision::REVISION_TYPE_CITATION => 'Citation',
            ],
        ]);
    }

    /**
     * Show the form for creating a new revision.
     */
    public function create(Request $request): Response
    {
        $order = null;
        if ($request->filled('order_id')) {
            $order = Order::with(['writer', 'user'])->findOrFail($request->order_id);
        }

        $writers = User::whereHas('roles', function ($query) {
            $query->where('name', 'writer');
        })->get(['id', 'name', 'email']);

        return Inertia::render('Admin/Revisions/Create', [
            'order' => $order,
            'writers' => $writers,
            'types' => [
                OrderRevision::REVISION_TYPE_CONTENT => 'Content',
                OrderRevision::REVISION_TYPE_FORMATTING => 'Formatting',
                OrderRevision::REVISION_TYPE_RESEARCH => 'Research',
                OrderRevision::REVISION_TYPE_CITATION => 'Citation',
            ],
            'priorities' => [
                OrderRevision::PRIORITY_LOW => 'Low',
                OrderRevision::PRIORITY_MEDIUM => 'Medium',
                OrderRevision::PRIORITY_HIGH => 'High',
            ],
        ]);
    }

    /**
     * Store a newly created revision.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|exists:orders,id',
            'revision_type' => 'required|in:' . implode(',', [
                OrderRevision::REVISION_TYPE_CONTENT,
                OrderRevision::REVISION_TYPE_FORMATTING,
                OrderRevision::REVISION_TYPE_RESEARCH,
                OrderRevision::REVISION_TYPE_CITATION,
            ]),
            'revision_priority' => 'required|in:' . implode(',', [
                OrderRevision::PRIORITY_LOW,
                OrderRevision::PRIORITY_MEDIUM,
                OrderRevision::PRIORITY_HIGH,
            ]),
            'revision_section' => 'nullable|string|max:1000',
            'revision_reason' => 'required|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
            'assigned_to' => 'nullable|exists:users,id',
            'revision_deadline' => 'nullable|date|after:now',
            'revision_admin_notes' => 'nullable|string|max:1000',
            'is_client_requested' => 'boolean',
            'client_notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $order = Order::findOrFail($request->order_id);
            
            $revision = $this->revisionService->createRevision($order, $request->all());

            return response()->json([
                'success' => true,
                'message' => 'Revision created successfully',
                'revision' => $revision->load(['order', 'requestedBy', 'assignedTo']),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to create revision', [
                'request' => $request->all(),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create revision: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified revision.
     */
    public function show(OrderRevision $revision): Response
    {
        $revision->load(['order', 'requestedBy', 'assignedTo', 'conversation']);

        $statistics = $this->revisionService->getRevisionStatistics($revision->order);

        return Inertia::render('Admin/Revisions/Show', [
            'revision' => $revision,
            'statistics' => $statistics,
            'nextPossibleStatuses' => $revision->getNextPossibleStatuses(),
        ]);
    }

    /**
     * Show the form for editing the specified revision.
     */
    public function edit(OrderRevision $revision): Response
    {
        $revision->load(['order', 'requestedBy', 'assignedTo']);

        $writers = User::whereHas('roles', function ($query) {
            $query->where('name', 'writer');
        })->get(['id', 'name', 'email']);

        return Inertia::render('Admin/Revisions/Edit', [
            'revision' => $revision,
            'writers' => $writers,
            'types' => [
                OrderRevision::REVISION_TYPE_CONTENT => 'Content',
                OrderRevision::REVISION_TYPE_FORMATTING => 'Formatting',
                OrderRevision::REVISION_TYPE_RESEARCH => 'Research',
                OrderRevision::REVISION_TYPE_CITATION => 'Citation',
            ],
            'priorities' => [
                OrderRevision::PRIORITY_LOW => 'Low',
                OrderRevision::PRIORITY_MEDIUM => 'Medium',
                OrderRevision::PRIORITY_HIGH => 'High',
            ],
            'statuses' => [
                OrderRevision::STATUS_REQUESTED => 'Requested',
                OrderRevision::STATUS_ACKNOWLEDGED => 'Acknowledged',
                OrderRevision::STATUS_IN_PROGRESS => 'In Progress',
                OrderRevision::STATUS_SUBMITTED => 'Submitted',
                OrderRevision::STATUS_REVIEWED => 'Reviewed',
            ],
        ]);
    }

    /**
     * Update the specified revision.
     */
    public function update(Request $request, OrderRevision $revision): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'revision_type' => 'sometimes|in:' . implode(',', [
                OrderRevision::REVISION_TYPE_CONTENT,
                OrderRevision::REVISION_TYPE_FORMATTING,
                OrderRevision::REVISION_TYPE_RESEARCH,
                OrderRevision::REVISION_TYPE_CITATION,
            ]),
            'revision_priority' => 'sometimes|in:' . implode(',', [
                OrderRevision::PRIORITY_LOW,
                OrderRevision::PRIORITY_MEDIUM,
                OrderRevision::PRIORITY_HIGH,
            ]),
            'revision_section' => 'nullable|string|max:1000',
            'revision_reason' => 'sometimes|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
            'revision_deadline' => 'nullable|date',
            'revision_admin_notes' => 'nullable|string|max:1000',
            'client_notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $revision->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Revision updated successfully',
                'revision' => $revision->fresh()->load(['order', 'requestedBy', 'assignedTo']),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to update revision', [
                'revision_id' => $revision->id,
                'request' => $request->all(),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update revision: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update revision status.
     */
    public function updateStatus(Request $request, OrderRevision $revision): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:' . implode(',', [
                OrderRevision::STATUS_REQUESTED,
                OrderRevision::STATUS_ACKNOWLEDGED,
                OrderRevision::STATUS_IN_PROGRESS,
                OrderRevision::STATUS_SUBMITTED,
                OrderRevision::STATUS_REVIEWED,
            ]),
            'metadata' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $success = $this->revisionService->updateRevisionStatus(
                $revision,
                $request->status,
                $request->metadata ?? []
            );

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'Revision status updated successfully',
                    'revision' => $revision->fresh()->load(['order', 'requestedBy', 'assignedTo']),
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Failed to update revision status',
            ], 500);

        } catch (\Exception $e) {
            Log::error('Failed to update revision status', [
                'revision_id' => $revision->id,
                'status' => $request->status,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update revision status: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Assign revision to a writer.
     */
    public function assignToWriter(Request $request, OrderRevision $revision): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'writer_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $writer = User::findOrFail($request->writer_id);
            
            $success = $this->revisionService->assignRevisionToWriter($revision, $writer);

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'Revision assigned to writer successfully',
                    'revision' => $revision->fresh()->load(['order', 'requestedBy', 'assignedTo']),
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign revision to writer',
            ], 500);

        } catch (\Exception $e) {
            Log::error('Failed to assign revision to writer', [
                'revision_id' => $revision->id,
                'writer_id' => $request->writer_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign revision to writer: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified revision.
     */
    public function destroy(OrderRevision $revision): JsonResponse
    {
        try {
            // Only allow deletion of requested revisions that haven't been assigned
            if ($revision->revision_status !== OrderRevision::STATUS_REQUESTED || $revision->assigned_to) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete revision that has been assigned or is in progress',
                ], 422);
            }

            $revision->delete();

            return response()->json([
                'success' => true,
                'message' => 'Revision deleted successfully',
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to delete revision', [
                'revision_id' => $revision->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete revision: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get revision statistics.
     */
    public function statistics(): JsonResponse
    {
        try {
            $overallStats = [
                'total_revisions' => OrderRevision::count(),
                'active_revisions' => OrderRevision::active()->count(),
                'overdue_revisions' => OrderRevision::overdue()->count(),
                'high_priority_revisions' => OrderRevision::highPriority()->count(),
                'completed_revisions' => OrderRevision::completed()->count(),
                'revisions_by_type' => OrderRevision::selectRaw('revision_type, COUNT(*) as count')
                    ->groupBy('revision_type')
                    ->pluck('count', 'revision_type'),
                'revisions_by_priority' => OrderRevision::selectRaw('revision_priority, COUNT(*) as count')
                    ->groupBy('revision_priority')
                    ->pluck('count', 'revision_priority'),
            ];

            return response()->json([
                'success' => true,
                'statistics' => $overallStats,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to get revision statistics', [
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to get revision statistics: ' . $e->getMessage(),
            ], 500);
        }
    }
}
