<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterTestQuestion;
use Illuminate\Http\Request;
use Inertia\Inertia;

class WriterTestQuestionController extends Controller
{
    /**
     * Display a listing of the test questions.
     */
    public function index(Request $request)
    {
        $query = WriterTestQuestion::query();

        // Apply search filter if provided
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('question', 'like', "%{$search}%")
                    ->orWhere('category', 'like', "%{$search}%");
            });
        }

        // Apply category filter if provided
        if ($request->has('category') && $request->category !== '') {
            $query->where('category', $request->category);
        }

        // Apply status filter if provided
        if ($request->has('status')) {
            $status = $request->status === 'active';
            $query->where('is_active', $status);
        }

        $questions = $query->orderBy('id', 'desc')
            ->paginate(10)
            ->appends($request->query());

        // Get unique categories for filter dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->pluck('category');

        return Inertia::render('Admin/WriterTest/Index', [
            'questions' => $questions,
            'categories' => $categories,
            'filters' => [
                'search' => $request->search ?? '',
                'category' => $request->category ?? '',
                'status' => $request->status ?? '',
            ],
        ]);
    }

    /**
     * Show the form for creating a new test question.
     */
    public function create()
    {
        // Get unique categories for dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->pluck('category');

        return Inertia::render('Admin/WriterTest/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created test question in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:1000',
            'option_a' => 'required|string|max:255',
            'option_b' => 'required|string|max:255',
            'option_c' => 'required|string|max:255',
            'option_d' => 'required|string|max:255',
            'correct_answer' => 'required|in:a,b,c,d',
            'category' => 'required|string|max:100',
            'is_active' => 'boolean',
        ]);

        WriterTestQuestion::create($validated);

        return redirect()->route('admin.writer-test-questions.index')
            ->with('message', 'Question created successfully.');
    }

    /**
     * Show the form for editing the specified test question.
     */
    public function edit(WriterTestQuestion $writerTestQuestion)
    {
        // Get unique categories for dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->pluck('category');

        return Inertia::render('Admin/WriterTest/Questions/Edit', [
            'question' => $writerTestQuestion,
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified test question in storage.
     */
    public function update(Request $request, WriterTestQuestion $writerTestQuestion)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:1000',
            'option_a' => 'required|string|max:255',
            'option_b' => 'required|string|max:255',
            'option_c' => 'required|string|max:255',
            'option_d' => 'required|string|max:255',
            'correct_answer' => 'required|in:a,b,c,d',
            'category' => 'required|string|max:100',
            'is_active' => 'boolean',
        ]);

        $writerTestQuestion->update($validated);

        return redirect()->route('admin.writer-test-questions.index')
            ->with('message', 'Question updated successfully.');
    }

    /**
     * Remove the specified test question from storage.
     */
    public function destroy(WriterTestQuestion $writerTestQuestion)
    {
        $writerTestQuestion->delete();

        return redirect()->route('admin.writer-test-questions.index')
            ->with('message', 'Question deleted successfully.');
    }

    /**
     * Toggle the active status of the specified test question.
     */
    public function toggleStatus(WriterTestQuestion $writerTestQuestion)
    {
        $writerTestQuestion->update([
            'is_active' => !$writerTestQuestion->is_active,
        ]);

        return redirect()->back()
            ->with('message', 'Question status updated successfully.');
    }
}
