<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\WriterProfile;
use App\Models\WriterTestResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;

class WriterTestReviewController extends Controller
{
    /**
     * Display a listing of writer test submissions.
     */
    public function index(Request $request)
    {
        // Get filter parameters
        $search = $request->input('search', '');
        $status = $request->input('status', 'all');
        $sortField = $request->input('sort_field', 'updated_at');
        $sortDirection = $request->input('sort_direction', 'desc');

        // Build the query for writer profiles with test data
        $query = WriterProfile::with(['user:id,name,email'])
            ->join('writer_qualification_tests', 'writer_profiles.id', '=', 'writer_qualification_tests.writer_profile_id')
            ->leftJoin('writer_essay_tests', 'writer_profiles.id', '=', 'writer_essay_tests.writer_profile_id')
            ->select('writer_profiles.*', 
                'writer_qualification_tests.questions_test_status',
                'writer_qualification_tests.questions_score',
                'writer_qualification_tests.questions_completed_at',
                'writer_qualification_tests.last_failed_at as questions_last_failed_at',
                'writer_qualification_tests.can_retry_after as questions_can_retry_after',
                'writer_essay_tests.essay_test_status',
                'writer_essay_tests.essay_score',
                'writer_essay_tests.essay_completed_at',
                'writer_essay_tests.last_failed_at as essay_last_failed_at',
                'writer_essay_tests.can_retry_after as essay_can_retry_after'
            )
            ->whereIn('writer_qualification_tests.questions_test_status', [
                'questions_failed',
                'passed'
            ])
            ->whereNotNull('writer_qualification_tests.questions_completed_at');

        // Apply search filter if provided
        if (!empty($search)) {
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                    ->orWhere('email', 'LIKE', "%{$search}%");
            });
        }

        // Apply status filter if not 'all'
        if ($status !== 'all') {
            if ($status === 'passed') {
                $query->where('writer_qualification_tests.questions_score', '>=', 90);
            } elseif ($status === 'failed') {
                $query->where('writer_qualification_tests.questions_score', '<', 90);
            }
        }

        // Apply sorting
        if ($sortField === 'name') {
            $query->join('users', 'writer_profiles.user_id', '=', 'users.id')
                ->orderBy('users.name', $sortDirection);
        } elseif ($sortField === 'score') {
            $query->orderBy('writer_qualification_tests.questions_score', $sortDirection);
        } else {
            $query->orderBy($sortField, $sortDirection);
        }

        // Get paginated results
        $writers = $query->paginate(10)
            ->withQueryString();

        // Transform the data for the frontend
        $writers->transform(function ($profile) {
            return [
                'id' => $profile->id,
                'user_id' => $profile->user_id,
                'name' => $profile->user->name,
                'email' => $profile->user->email,
                'question_test_status' => $profile->questions_test_status,
                'essay_test_status' => $profile->essay_test_status,
                'score' => $profile->questions_score,
                'essay_score' => $profile->essay_score,
                'completed_at' => $profile->questions_completed_at,
                'essay_completed_at' => $profile->essay_completed_at,
                'can_retry_after' => $profile->questions_can_retry_after,
                'essay_can_retry_after' => $profile->essay_can_retry_after,
                'last_failed_at' => $profile->questions_last_failed_at,
                'essay_last_failed_at' => $profile->essay_last_failed_at,
            ];
        });

        return Inertia::render('Admin/WriterTests/Index', [
            'writers' => $writers,
            'filters' => [
                'search' => $search,
                'status' => $status,
                'sort_field' => $sortField,
                'sort_direction' => $sortDirection,
            ],
        ]);
    }

    /**
     * Display the specified writer test results.
     */
    public function show($id)
    {
        // Get the writer profile with test data
        $profile = WriterProfile::with(['user:id,name,email'])
            ->join('writer_qualification_tests', 'writer_profiles.id', '=', 'writer_qualification_tests.writer_profile_id')
            ->leftJoin('writer_essay_tests', 'writer_profiles.id', '=', 'writer_essay_tests.writer_profile_id')
            ->select('writer_profiles.*', 
                'writer_qualification_tests.questions_test_status',
                'writer_qualification_tests.questions_score',
                'writer_qualification_tests.questions_completed_at',
                'writer_qualification_tests.last_failed_at as questions_last_failed_at',
                'writer_qualification_tests.can_retry_after as questions_can_retry_after',
                'writer_essay_tests.essay_test_status',
                'writer_essay_tests.essay_score',
                'writer_essay_tests.essay_completed_at',
                'writer_essay_tests.last_failed_at as essay_last_failed_at',
                'writer_essay_tests.can_retry_after as essay_can_retry_after'
            )
            ->findOrFail($id);

        // Get the test responses
        $responses = WriterTestResponse::where('user_id', $profile->user_id)
            ->with('question')
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('created_at->date')
            ->first();

        if (!$responses) {
            return redirect()->route('admin.writers.tests')
                ->with('error', 'No test responses found for this writer.');
        }

        // Calculate statistics
        $totalQuestions = $responses->count();
        $correctAnswers = $responses->where('is_correct', true)->count();
        $scorePercentage = ($correctAnswers / $totalQuestions) * 100;
        $passed = $scorePercentage >= 90;

        // Group questions by category for detailed feedback
        $categorizedResponses = $responses->groupBy(function ($response) {
            return $response->question->category;
        })->map(function ($categoryResponses) {
            $total = $categoryResponses->count();
            $correct = $categoryResponses->where('is_correct', true)->count();

            return [
                'total' => $total,
                'correct' => $correct,
                'percentage' => ($correct / $total) * 100,
            ];
        });

        // Format the responses for display
        $formattedResponses = $responses->map(function ($response) {
            return [
                'id' => $response->id,
                'question' => $response->question->question,
                'category' => $response->question->category,
                'selected_answer' => $response->selected_answer,
                'correct_answer' => $response->question->correct_answer,
                'is_correct' => $response->is_correct,
                'options' => [
                    'a' => $response->question->option_a,
                    'b' => $response->question->option_b,
                    'c' => $response->question->option_c,
                    'd' => $response->question->option_d,
                ],
                'explanation' => $response->question->explanation,
            ];
        });

        return Inertia::render('Admin/WriterTests/Show', [
            'writer' => [
                'id' => $profile->id,
                'user_id' => $profile->user_id,
                'name' => $profile->user->name,
                'email' => $profile->user->email,
                'questions_test_status' => $profile->questions_test_status,
                'essay_test_status' => $profile->essay_test_status,
                'questions_score' => $profile->questions_score,
                'essay_score' => $profile->essay_score,
                'questions_completed_at' => $profile->questions_completed_at,
                'essay_completed_at' => $profile->essay_completed_at,
                'questions_can_retry_after' => $profile->questions_can_retry_after,
                'essay_can_retry_after' => $profile->essay_can_retry_after,
                'questions_last_failed_at' => $profile->questions_last_failed_at,
                'essay_last_failed_at' => $profile->essay_last_failed_at,
            ],
            'score' => [
                'total' => $totalQuestions,
                'correct' => $correctAnswers,
                'percentage' => $scorePercentage,
                'passed' => $passed,
            ],
            'categorizedResults' => $categorizedResponses,
            'responses' => $formattedResponses,
        ]);
    }
}
