<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterWithdrawalRequest;
use App\Models\WriterPaymentRecord;
use App\Models\User;
use App\Services\WriterPaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class WriterWithdrawalController extends Controller
{
    protected $writerPaymentService;

    public function __construct(WriterPaymentService $writerPaymentService)
    {
        $this->middleware('auth');
        $this->middleware('admin');
        $this->writerPaymentService = $writerPaymentService;
    }

    /**
     * Display the withdrawal management dashboard.
     */
    public function index(Request $request)
    {
        $status = $request->get('status', 'all');
        $date = $request->get('date');
        $writer = $request->get('writer');

        $query = WriterWithdrawalRequest::with(['writer', 'approver']);

        // Filter by status
        if ($status !== 'all') {
            $query->where('status', $status);
        }

        // Filter by date
        if ($date) {
            $query->whereDate('withdrawal_date', $date);
        }

        // Filter by writer
        if ($writer) {
            $query->whereHas('writer', function ($q) use ($writer) {
                $q->where('name', 'like', "%{$writer}%")
                  ->orWhere('email', 'like', "%{$writer}%");
            });
        }

        $withdrawals = $query->orderBy('created_at', 'desc')->paginate(20);

        // Get summary statistics
        $stats = $this->getWithdrawalStats();

        return Inertia::render('Admin/WriterWithdrawals/Index', [
            'withdrawals' => $withdrawals,
            'stats' => $stats,
            'filters' => [
                'status' => $status,
                'date' => $date,
                'writer' => $writer,
            ]
        ]);
    }

    /**
     * Show a specific withdrawal request.
     */
    public function show(WriterWithdrawalRequest $withdrawal)
    {
        $withdrawal->load(['writer', 'approver', 'paymentRecords.order']);

        return Inertia::render('Admin/WriterWithdrawals/Show', [
            'withdrawal' => $withdrawal
        ]);
    }

    /**
     * Approve a withdrawal request.
     */
    public function approve(Request $request, WriterWithdrawalRequest $withdrawal)
    {
        $request->validate([
            'notes' => 'nullable|string|max:1000',
            'payment_method_details' => 'nullable|array'
        ]);

        try {
            DB::transaction(function () use ($withdrawal, $request) {
                // Approve the withdrawal
                $withdrawal->approve(auth()->id());

                // Update payment records to processing status
                $withdrawal->paymentRecords()->update([
                    'status' => WriterPaymentRecord::STATUS_PROCESSING
                ]);

                // Log the approval
                Log::info('Withdrawal request approved', [
                    'withdrawal_id' => $withdrawal->id,
                    'writer_id' => $withdrawal->writer_id,
                    'amount' => $withdrawal->requested_amount,
                    'approved_by' => auth()->id(),
                    'notes' => $request->notes
                ]);
            });

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal request approved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to approve withdrawal request: ' . $e->getMessage(), [
                'withdrawal_id' => $withdrawal->id,
                'admin_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to approve withdrawal request'
            ], 500);
        }
    }

    /**
     * Reject a withdrawal request.
     */
    public function reject(Request $request, WriterWithdrawalRequest $withdrawal)
    {
        $request->validate([
            'rejection_reason' => 'required|string|max:1000'
        ]);

        try {
            DB::transaction(function () use ($withdrawal, $request) {
                // Reject the withdrawal
                $withdrawal->reject($request->rejection_reason, auth()->id());

                // Revert payment records to available status
                $withdrawal->paymentRecords()->update([
                    'status' => WriterPaymentRecord::STATUS_AVAILABLE
                ]);

                // Log the rejection
                Log::info('Withdrawal request rejected', [
                    'withdrawal_id' => $withdrawal->id,
                    'writer_id' => $withdrawal->writer_id,
                    'amount' => $withdrawal->requested_amount,
                    'rejected_by' => auth()->id(),
                    'reason' => $request->rejection_reason
                ]);
            });

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal request rejected successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to reject withdrawal request: ' . $e->getMessage(), [
                'withdrawal_id' => $withdrawal->id,
                'admin_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to reject withdrawal request'
            ], 500);
        }
    }

    /**
     * Process approved withdrawals (mark as processing).
     */
    public function process(Request $request, WriterWithdrawalRequest $withdrawal)
    {
        if ($withdrawal->status !== WriterWithdrawalRequest::STATUS_APPROVED) {
            return response()->json([
                'success' => false,
                'message' => 'Only approved withdrawals can be processed'
            ], 400);
        }

        try {
            $withdrawal->markAsProcessing();

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal marked as processing'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to mark withdrawal as processing: ' . $e->getMessage(), [
                'withdrawal_id' => $withdrawal->id,
                'admin_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark withdrawal as processing'
            ], 500);
        }
    }

    /**
     * Mark withdrawal as paid.
     */
    public function markAsPaid(Request $request, WriterWithdrawalRequest $withdrawal)
    {
        $request->validate([
            'transaction_reference' => 'required|string|max:255',
            'payment_notes' => 'nullable|string|max:1000'
        ]);

        try {
            DB::transaction(function () use ($withdrawal, $request) {
                // Mark withdrawal as paid
                $withdrawal->markAsPaid($request->transaction_reference);

                // Update payment records to withdrawn status
                $withdrawal->paymentRecords()->update([
                    'status' => WriterPaymentRecord::STATUS_WITHDRAWN,
                    'withdrawn_at' => now(),
                    'transaction_reference' => $request->transaction_reference
                ]);

                // Log the payment
                Log::info('Withdrawal marked as paid', [
                    'withdrawal_id' => $withdrawal->id,
                    'writer_id' => $withdrawal->writer_id,
                    'amount' => $withdrawal->requested_amount,
                    'admin_id' => auth()->id(),
                    'transaction_reference' => $request->transaction_reference
                ]);
            });

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal marked as paid successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to mark withdrawal as paid: ' . $e->getMessage(), [
                'withdrawal_id' => $withdrawal->id,
                'admin_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark withdrawal as paid'
            ], 500);
        }
    }

    /**
     * Bulk approve multiple withdrawal requests.
     */
    public function bulkApprove(Request $request)
    {
        $request->validate([
            'withdrawal_ids' => 'required|array',
            'withdrawal_ids.*' => 'exists:writer_withdrawal_requests,id'
        ]);

        $approvedCount = 0;
        $failedCount = 0;

        foreach ($request->withdrawal_ids as $withdrawalId) {
            try {
                $withdrawal = WriterWithdrawalRequest::find($withdrawalId);
                
                if ($withdrawal && $withdrawal->status === WriterWithdrawalRequest::STATUS_REQUESTED) {
                    $withdrawal->approve(auth()->id());
                    $withdrawal->paymentRecords()->update([
                        'status' => WriterPaymentRecord::STATUS_PROCESSING
                    ]);
                    $approvedCount++;
                }
            } catch (\Exception $e) {
                $failedCount++;
                Log::error('Failed to bulk approve withdrawal: ' . $e->getMessage(), [
                    'withdrawal_id' => $withdrawalId,
                    'admin_id' => auth()->id()
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => "Bulk approval completed. {$approvedCount} approved, {$failedCount} failed.",
            'approved_count' => $approvedCount,
            'failed_count' => $failedCount
        ]);
    }

    /**
     * Get withdrawal statistics.
     */
    private function getWithdrawalStats(): array
    {
        $total = WriterWithdrawalRequest::count();
        $pending = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_REQUESTED)->count();
        $approved = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_APPROVED)->count();
        $processing = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_PROCESSING)->count();
        $paid = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_PAID)->count();
        $rejected = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_REJECTED)->count();

        $totalAmount = WriterWithdrawalRequest::where('status', WriterWithdrawalRequest::STATUS_PAID)
            ->sum('requested_amount');

        return [
            'total' => $total,
            'pending' => $pending,
            'approved' => $approved,
            'processing' => $processing,
            'paid' => $paid,
            'rejected' => $rejected,
            'total_amount_paid' => $totalAmount
        ];
    }

    /**
     * Export withdrawal data.
     */
    public function export(Request $request)
    {
        $status = $request->get('status', 'all');
        $date = $request->get('date');

        $query = WriterWithdrawalRequest::with(['writer', 'approver']);

        if ($status !== 'all') {
            $query->where('status', $status);
        }

        if ($date) {
            $query->whereDate('withdrawal_date', $date);
        }

        $withdrawals = $query->get();

        // Generate CSV
        $filename = 'withdrawals_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($withdrawals) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'ID', 'Writer', 'Email', 'Amount', 'Status', 'Requested Date', 
                'Payment Method', 'Approved By', 'Approved Date', 'Paid Date'
            ]);

            // CSV data
            foreach ($withdrawals as $withdrawal) {
                fputcsv($file, [
                    $withdrawal->id,
                    $withdrawal->writer->name ?? 'N/A',
                    $withdrawal->writer->email ?? 'N/A',
                    $withdrawal->requested_amount,
                    $withdrawal->status,
                    $withdrawal->withdrawal_date,
                    $withdrawal->payment_method,
                    $withdrawal->approver->name ?? 'N/A',
                    $withdrawal->approved_at,
                    $withdrawal->paid_at
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
} 