<?php

namespace App\Http\Controllers;

use App\Models\User;
use Inertia\Inertia;
use App\Models\Order;
use App\Models\Refund;
use App\Models\Payment;
use Illuminate\Http\Request;
use App\Services\RefundService;
use App\Services\UrgentOrderNotificationService;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{

    /**
     * @var RefundService
     */
    protected $refundService;
    protected $urgentOrderService;

    /**
     * Create a new controller instance.
     *
     * @param RefundService $refundService
     * @param UrgentOrderNotificationService $urgentOrderService
     * @return void
     */
    public function __construct(RefundService $refundService, UrgentOrderNotificationService $urgentOrderService)
    {
        $this->refundService = $refundService;
        $this->urgentOrderService = $urgentOrderService;
    }



    public function dashboard()
    {
        $totalClients = User::where('user_type', 'client')->count();
        $totalWriters = User::where('user_type', 'writer')->count();
        $totalOrders = Order::count();

        // Get refund statistics
        $totalRefundedAmount = $this->calculateTotalRefundedAmount();
        $refundedOrdersCount = Order::anyRefund()->count();
        $refundRate = $totalOrders > 0 ? round(($refundedOrdersCount / $totalOrders) * 100, 2) : 0;

        $clients = User::where('user_type', 'client')
            ->withCount('orders')
            ->paginate(5, ['id', 'name', 'email']);

        // This will load all clients and their associated orders
        $users = User::where('user_type', 'client')
            ->withCount('orders as order_count')
            ->limit(10) // Optionally limit to top 10 clients
            ->orderBy('order_count', 'desc')
            ->get(['id', 'name', 'orders_count']);

        // Format data for the chart
        $clientsOrders = $users->map(function ($client) {
            return [
                'name' => $client->name,
                'order_count' => $client->order_count
            ];
        });

        // Calculate monthly data with refund adjustments
        $monthlyData = $this->getMonthlyRevenueData();

        // Calculate gross revenue (paid orders)
        $grossRevenue = Order::whereIn('payment_status', ['paid', 'partially_refunded', 'refunded'])
            ->sum('net_amount');

        // Calculate net revenue (after refunds)
        $netRevenue = $grossRevenue - $totalRefundedAmount;

        // Get urgent orders for admin dashboard
        $urgentOrders = $this->urgentOrderService->getUrgentOrdersForUser(auth()->user());
        $urgentOrdersCount = $this->urgentOrderService->getUrgentOrdersCount(auth()->user());

        return Inertia::render('Admin/Dashboard', [
            'clients' => $clients,
            'totalClients' => $totalClients,
            'totalWriters' => $totalWriters,
            'totalOrders' => $totalOrders,
            'chartData' => $monthlyData,
            'clientsOrders' => $clientsOrders,
            'grossRevenue' => $grossRevenue,
            'totalRevenue' => $netRevenue,
            'urgentOrders' => $urgentOrders,
            'urgentOrdersCount' => $urgentOrdersCount,
            'refundStats' => [
                'totalRefunded' => $totalRefundedAmount,
                'refundedOrdersCount' => $refundedOrdersCount,
                'refundRate' => $refundRate,
                'partiallyRefundedCount' => Order::partiallyRefunded()->count(),
                'fullyRefundedCount' => Order::refunded()->count(),
            ]
        ]);
    }


    /**
     * Calculate the total refunded amount across all orders
     *
     * @return float
     */
    private function calculateTotalRefundedAmount(): float
    {
        return (float) Refund::where('status', Refund::STATUS_COMPLETED)
            ->sum('amount');
    }

    /**
     * Get monthly revenue data with refund adjustments
     *
     * @return \Illuminate\Support\Collection
     */
    private function getMonthlyRevenueData()
    {
        // Get monthly order data
        $monthlyOrders = DB::table('orders')
            ->selectRaw('MONTH(dateposted) as month, COUNT(*) as orders, SUM(net_amount) as gross_revenue')
            ->whereYear('dateposted', date('Y'))
            ->groupBy('month')
            ->get();

        // Get monthly refund data
        $monthlyRefunds = DB::table('refunds')
            ->join('payments', 'refunds.payment_id', '=', 'payments.id')
            ->join('orders', 'payments.order_id', '=', 'orders.id')
            ->selectRaw('MONTH(refunds.processed_at) as month, SUM(refunds.amount) as refunded_amount')
            ->where('refunds.status', Refund::STATUS_COMPLETED)
            ->whereYear('refunds.processed_at', date('Y'))
            ->groupBy('month')
            ->get()
            ->keyBy('month');

        // Combine order and refund data
        return $monthlyOrders->map(function ($item) use ($monthlyRefunds) {
            $month = $item->month;
            $refundedAmount = $monthlyRefunds->has($month) ? $monthlyRefunds[$month]->refunded_amount : 0;
            $netRevenue = $item->gross_revenue - $refundedAmount;

            return [
                'name' => date('M', mktime(0, 0, 0, $month, 1)),
                'orders' => $item->orders,
                'gross_revenue' => $item->gross_revenue,
                'refunded' => $refundedAmount,
                'revenue' => $netRevenue
            ];
        });
    }

    public function userOrders($userId)
    {
        $user = User::find($userId);

        if (!$user) {
            return redirect()->route('admin.dashboard')->with('error', 'User not found');
        }

        // Get the user's orders with pagination and refund information
        $orders = $user->orders()
            ->orderBy('dateposted', 'desc')
            ->with(['payments.refunds' => function ($query) {
                $query->where('status', Refund::STATUS_COMPLETED);
            }])
            ->paginate(4)
            ->through(function ($order) {
                // Add refund information to each order
                return array_merge($order->toArray(), [
                    'is_refunded' => $order->isRefunded(),
                    'is_partially_refunded' => $order->isPartiallyRefunded(),
                    'refunded_amount' => $order->getRefundedAmount(),
                    'remaining_amount' => $order->getRemainingAmount()
                ]);
            });

        return Inertia::render('Admin/Users/UserOrders', [
            'user' => $user,
            'orders' => $orders
        ]);
    }
}
