<?php

namespace App\Http\Controllers;

use App\Models\User;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Notifications\AdminUserRegisterNotification;


class AdminUserController extends Controller
{

    /**
     * Display a paginated list of users.
     */
    public function index(): Response
    {
        $users = User::orderBy('created_at', 'asc')->paginate(6); // Paginate 10 users per page
        $userCount = User::count();
        $writerCount = User::where('user_type', 'writer')->count();
        $clientCount = User::where('user_type', 'client')->count();
        $adminCount = User::where('user_type', 'admin')->count();


        return Inertia::render('Admin/Users/Index', [
            'users' => $users,
            'userCount' => $userCount,
            'writerCount' => $writerCount,
            'clientCount' => $clientCount,
            'adminCount' => $adminCount,
        ]);
    }

    /**
     * Search for users by name or email.
     */
    public function search(Request $request): Response
    {
        $query = $request->input('query');
        $userType = $request->input('userType');
        $status = $request->input('status');

        $users = User::query()
            ->when($query, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                });
            })
            ->when($userType !== null, function ($query) use ($userType) {
                $query->where('user_type', $userType);
            })
            ->when($status !== null, function ($query) use ($status) {
                $query->where('status', $status);
            })
            ->paginate(6)
            ->appends($request->query());
        // ->withQueryString();

        $userCount = User::count();
        $writerCount = User::where('user_type', 'writer')->count();
        $clientCount = User::where('user_type', 'client')->count();
        $adminCount = User::where('user_type', 'admin')->count();

        return Inertia::render('Admin/Users/Index', [
            'users' => $users,
            'query' => $query,
            'filters' => [
                'userType' => $userType,
                'status' => $status
            ],
            'userCount' => $userCount,
            'writerCount' => $writerCount,
            'clientCount' => $clientCount,
            'adminCount' => $adminCount,
        ]);
    }

    /**
     * Show the form for editing a user.
     */
    public function show($id): Response
    {
        $user = User::findOrFail($id);
        return Inertia::render('Admin/Users/Show', [
            'user' => $user,
        ]);
    }

    public function create(): Response
    {
        return Inertia::render('Admin/Users/Create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'user_type' => 'required|in:admin,writer,client',
            'password' => 'required|string|min:6 ',
            'password_confirmation' => 'required|same:password',
            'status' => 'required|in:active,inactive',
        ]);

        // Store the plain text password temporarily to include in the notification
        $plainPassword = $request->password;

        // Create the user with hashed password
        $user = new User();
        $user->name = $request->name;
        $user->email = $request->email;
        $user->user_type = $request->user_type;
        $user->status = $request->status;
        $user->password = Hash::make($plainPassword);
        $user->save();

        // Send notification to the newly created user
        $user->notify(new AdminUserRegisterNotification($user, $plainPassword));

        return redirect()->route('admin.users.index')->with('message', 'User created successfully.');
    }



    public function edit($id): Response
    {
        $user = User::findOrFail($id);

        return Inertia::render('Admin/Users/Edit', [
            'user' => $user,
        ]);
    }

    /**
     * Update user details.
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'user_type' => 'required|in:admin,writer,client',
            'status' => 'required|in:active,inactive',
            'password' => 'nullable|string|min:6|confirmed',
        ]);

        // Get all the fields you want to update
        $userData = $request->only(['name', 'email', 'user_type', 'status']);

        $passwordChanged = false;
        $newPassword = null;

        // Only update password if it's provided
        if ($request->filled('password')) {
            $newPassword = $request->password;
            $userData['password'] = Hash::make($newPassword);
            $passwordChanged = true;
        }

        $user->update($userData);

        // Notify user if password was changed
        if ($passwordChanged) {
            $user->notify(new \App\Notifications\AdminUserRegisterNotification($user, $newPassword));
        }

        return redirect()->route('admin.users.index')->with('message', 'User updated successfully.');
    }

    /**
     * Deactivate a user (soft delete).
     */
    public function deactivate($id)
    {
        $user = User::findOrFail($id);
        $user->update(['status' => 'inactive']);

        return redirect()->route('admin.users.show', $id)->with('message', 'User deactivated.');
    }

    public function activate($id)
    {
        $user = User::findOrFail($id);
        $user->update(['status' => 'active']);
        return redirect()->route('admin.users.show', $id)->with('message', 'User activated.');
    }

    /**
     * Delete a user permanently.
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        return redirect()->route('admin.users.index')->with('message', 'User Deleted Successfully.');
    }
}
