<?php

namespace App\Http\Controllers\Auth;

use App\Models\User;
use Inertia\Inertia;
use Inertia\Response;
use App\Models\Country;
use App\Models\Subject;
use App\Models\Language;
use Illuminate\Http\Request;
use App\Models\WriterProfile;
use App\Models\EducationLevel;
use App\Models\AcademicSubject;
use App\Models\WriterEssayTest;
use Illuminate\Validation\Rules;
use App\Models\WriterCitationStyle;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use Illuminate\Auth\Events\Registered;
use App\Models\WriterQualificationTest;
use App\Notifications\WriterRegistered;
use App\Services\NicknameGeneratorService;

class WriterRegisteredUserController extends Controller
{
    /**
     * Display the writer registration view.
     */
    public function create(): Response
    {
        // Fetch all countries with dialing codes
        $countries = Country::select('id', 'name', 'dialing_code')
            ->orderBy('name')
            ->get();

        // Fetch all subjects
        $subjects = Subject::select('id', 'name')
            ->orderBy('name')
            ->get();

        // Fetch all citation styles
        $citationStyles = WriterCitationStyle::select('id', 'name')
            ->orderBy('name')
            ->get();

        // Fetch all education levels
        $educationLevels = EducationLevel::select('id', 'name')
            ->orderBy('name')
            ->get();

        // Fetch all languages
        $languages = Language::select('id', 'name')
            ->orderBy('name')
            ->get();

        return Inertia::render('Auth/WriterRegister', [
            'countries' => $countries,
            'subjects' => $subjects,
            'citationStyles' => $citationStyles,
            'educationLevels' => $educationLevels,
            'languages' => $languages,
            'maxSubjects' => 5,
            'maxLanguages' => 3,
            'maxCitationStyles' => 5,
        ]);
    }

    /**
     * Handle an incoming writer registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|lowercase|email|max:255|unique:' . User::class,
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'country_id' => 'required|exists:countries,id',
            'phone' => 'required|string|max:50',
            'education_level_id' => 'required|exists:writer_education_levels,id',
            'subjects' => 'required|array|min:1|max:5',
            'subjects.*' => 'exists:subjects,id',
            'languages' => 'required|array|min:1|max:3', // Updated to max 3
            'languages.*' => 'exists:languages,id',
            'citation_styles' => 'required|array|min:1|max:5', // New field for multiple citation styles
            'citation_styles.*' => 'exists:writer_citation_styles,id',
            'bio' => 'required|string|max:1000',
            'experience_years' => 'required|integer|min:0',
        ]);

        // Create the user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'user_type' => 'writer',
            'password' => Hash::make($request->password),
            'status' => 'active',
            'country_id' => $request->country_id,
            'phone' => $request->phone,
        ]);

        // Generate automatic nickname
        $nickname = NicknameGeneratorService::generateNickname($user->id, $request->name);

        // Create the writer profile
        $writerProfile = WriterProfile::create([
            'user_id' => $user->id,
            'nickname' => $nickname,
            'education_level_id' => $request->education_level_id,
            'bio' => $request->bio,
            'experience_years' => $request->experience_years,
            'is_available' => 0,
        ]);

        // Attach subjects to the writer profile
        $writerProfile->subjects()->attach($request->subjects);

        // Attach languages to the writer profile (max 3)
        $writerProfile->languages()->attach($request->languages);

        // Attach citation styles to the writer profile (max 5)
        $writerProfile->citationStyles()->attach($request->citation_styles);

        // Create initial qualification test record
        WriterQualificationTest::create([
            'writer_profile_id' => $writerProfile->id,
            'questions_test_status' => 'not_started',
        ]);

        // Create initial essay test record
        WriterEssayTest::create([
            'writer_profile_id' => $writerProfile->id,
            'essay_test_status' => 'not_started',
        ]);

        event(new Registered($user));

        // Send notification to the writer
        $user->notify(new WriterRegistered($user));

        Auth::login($user);

        return redirect(route('writer.dashboard', absolute: false));
    }
}
