<?php

namespace App\Http\Controllers;

use App\Services\BlogFileProcessor;
use App\Services\BlogImportService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class BlogProcessingController extends Controller
{
    private BlogFileProcessor $fileProcessor;
    private BlogImportService $importService;

    public function __construct(BlogFileProcessor $fileProcessor, BlogImportService $importService)
    {
        $this->fileProcessor = $fileProcessor;
        $this->importService = $importService;
    }

    /**
     * Process uploaded blog files
     */
    public function processFiles(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'upload_id' => 'required|string|uuid',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'details' => $validator->errors(),
                ], 422);
            }

            $uploadId = $request->input('upload_id');

            // Process the files
            $result = $this->fileProcessor->processUploadedFiles($uploadId);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'error' => 'File processing failed',
                    'details' => $result,
                ], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Files processed successfully',
                'data' => $result,
            ]);
        } catch (\Exception $e) {
            Log::error('Blog file processing failed', [
                'upload_id' => $request->input('upload_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Processing failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Import processed files to database
     */
    public function importFiles(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'upload_id' => 'required|string|uuid',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'details' => $validator->errors(),
                ], 422);
            }

            $uploadId = $request->input('upload_id');

            // Import the processed files
            $result = $this->importService->importProcessedFiles($uploadId);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'error' => 'Import failed',
                    'details' => $result,
                ], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Files imported successfully',
                'data' => $result,
            ]);
        } catch (\Exception $e) {
            Log::error('Blog import failed', [
                'upload_id' => $request->input('upload_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Import failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get processing status
     */
    public function getProcessingStatus(string $uploadId): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $uploadDisk = Storage::disk($config['storage']['upload_disk']);

            // Check if upload exists
            $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;
            if (!$uploadDisk->exists($uploadPath)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Upload not found',
                ], 404);
            }

            // Check processing status
            $processedPath = $config['storage']['directories']['processed'] . '/' . $uploadId;
            $processedFiles = $uploadDisk->files($processedPath);

            $uploadedFiles = $uploadDisk->files($uploadPath);

            $status = [
                'upload_id' => $uploadId,
                'total_files' => count($uploadedFiles),
                'processed_files' => count($processedFiles),
                'processing_complete' => count($processedFiles) === count($uploadedFiles),
                'processing_percentage' => count($uploadedFiles) > 0
                    ? round((count($processedFiles) / count($uploadedFiles)) * 100, 2)
                    : 0,
            ];

            return response()->json([
                'success' => true,
                'status' => $status,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get processing status: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get import statistics
     */
    public function getImportStats(string $uploadId): JsonResponse
    {
        try {
            $stats = $this->importService->getImportStats($uploadId);

            return response()->json([
                'success' => true,
                'stats' => $stats,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get import stats: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Process and import files in one operation
     */
    public function processAndImport(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'upload_id' => 'required|string|uuid',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'details' => $validator->errors(),
                ], 422);
            }

            $uploadId = $request->input('upload_id');

            // Step 1: Process files
            $processResult = $this->fileProcessor->processUploadedFiles($uploadId);

            if (!$processResult['success']) {
                return response()->json([
                    'success' => false,
                    'error' => 'File processing failed',
                    'details' => $processResult,
                ], 500);
            }

            // Step 2: Import to database
            $importResult = $this->importService->importProcessedFiles($uploadId);

            if (!$importResult['success']) {
                return response()->json([
                    'success' => false,
                    'error' => 'Import failed',
                    'details' => $importResult,
                ], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Files processed and imported successfully',
                'data' => [
                    'processing' => $processResult,
                    'import' => $importResult,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Blog process and import failed', [
                'upload_id' => $request->input('upload_id'),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Process and import failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get processed file content
     */
    public function getProcessedFile(string $uploadId, string $filename): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $uploadDisk = Storage::disk($config['storage']['upload_disk']);

            $filePath = $config['storage']['directories']['processed'] . '/' . $uploadId . '/' . $filename;

            if (!$uploadDisk->exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Processed file not found',
                ], 404);
            }

            $content = $uploadDisk->get($filePath);
            $data = json_decode($content, true);

            return response()->json([
                'success' => true,
                'filename' => $filename,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get processed file: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Clean up upload files
     */
    public function cleanupUpload(string $uploadId): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $uploadDisk = Storage::disk($config['storage']['upload_disk']);

            // Delete upload directory
            $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;
            if ($uploadDisk->exists($uploadPath)) {
                $uploadDisk->deleteDirectory($uploadPath);
            }

            // Delete processed directory
            $processedPath = $config['storage']['directories']['processed'] . '/' . $uploadId;
            if ($uploadDisk->exists($processedPath)) {
                $uploadDisk->deleteDirectory($processedPath);
            }

            // Delete backup directory
            $backupPath = $config['storage']['directories']['backups'] . '/' . $uploadId;
            if ($uploadDisk->exists($backupPath)) {
                $uploadDisk->deleteDirectory($backupPath);
            }

            Log::info('Upload cleanup completed', [
                'upload_id' => $uploadId,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Upload cleaned up successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('Upload cleanup failed', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Cleanup failed: ' . $e->getMessage(),
            ], 500);
        }
    }
}
