<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderRevision;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class ClientRevisionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('client');
    }

    /**
     * Display a listing of revisions for a specific order.
     */
    public function index(Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        $revisions = $order->revisions()
            ->with(['assignedTo', 'requestedBy'])
            ->orderBy('created_at', 'desc')
            ->get();

        return Inertia::render('Client/Revisions/Index', [
            'order' => $order->load(['writer', 'orderFiles']),
            'revisions' => $revisions,
            'canRequestRevision' => $this->canRequestRevision($order),
            'revisionTypes' => config('revisions.types'),
        ]);
    }

    /**
     * Show the form for requesting a new revision.
     */
    public function create(Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        // Check if order is eligible for revision
        if (!$this->canRequestRevision($order)) {
            abort(400, 'This order is not eligible for revision');
        }

        $revisionTypes = config('revisions.types');
        $existingRevisions = $order->revisions()->count();

        return Inertia::render('Client/Revisions/Create', [
            'order' => $order->load(['writer', 'orderFiles']),
            'revisionTypes' => $revisionTypes,
            'existingRevisions' => $existingRevisions,
            'maxRevisions' => config('revisions.max_per_order'),
        ]);
    }

    /**
     * Store a newly requested revision.
     */
    public function store(Request $request, Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        // Check if order is eligible for revision
        if (!$this->canRequestRevision($order)) {
            return response()->json([
                'message' => 'This order is not eligible for revision'
            ], 400);
        }

        $request->validate([
            'revision_type' => 'required|in:' . implode(',', array_keys(config('revisions.types'))),
            'revision_reason' => 'required|string|max:1000',
            'specific_instructions' => 'nullable|string|max:2000',
        ]);

        try {
            $revisionNumber = $order->revisions()->count() + 1;
            $isFinalRevision = $revisionNumber >= config('revisions.max_per_order');

            $revision = OrderRevision::create([
                'order_id' => $order->id,
                'requested_by' => Auth::id(),
                'revision_type' => $request->revision_type,
                'revision_reason' => $request->revision_reason,
                'specific_instructions' => $request->specific_instructions,
                'status' => 'requested',
                'requested_at' => now(),
                'due_date' => now()->addHours(config('revisions.default_due_hours')),
                'revision_number' => $revisionNumber,
                'is_final_revision' => $isFinalRevision,
                'is_client_requested' => true,
                'client_notes' => $request->revision_reason,
            ]);

            // Create conversation thread for the revision
            $conversation = $this->createRevisionConversation($revision);
            $revision->update(['conversation_id' => $conversation->id]);

            // Update order status if needed
            if ($order->order_status === 'delivered_to_client') {
                $order->update(['order_status' => 'revision_requested']);
            }

            // Send notification to admin
            $this->notifyAdminOfRevision($revision);

            return response()->json([
                'message' => 'Revision requested successfully',
                'revision' => $revision->load(['order', 'requestedBy']),
                'redirect' => route('client.revisions.index', $order)
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to create revision: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'client_id' => Auth::id(),
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to create revision: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified revision.
     */
    public function show(Order $order, OrderRevision $revision)
    {
        // Verify the client owns this order and revision
        if ($order->user_id !== Auth::id() || $revision->order_id !== $order->id) {
            abort(403, 'Unauthorized');
        }

        $revision->load(['assignedTo', 'requestedBy', 'conversation.messages.user']);

        return Inertia::render('Client/Revisions/Show', [
            'order' => $order->load(['writer', 'orderFiles']),
            'revision' => $revision,
            'canRequestRevision' => $this->canRequestRevision($order),
        ]);
    }

    /**
     * Get revision statistics for the client dashboard.
     */
    public function dashboard()
    {
        $clientId = Auth::id();
        
        $stats = [
            'total_revisions' => OrderRevision::whereHas('order', function ($query) use ($clientId) {
                $query->where('user_id', $clientId);
            })->count(),
            
            'pending_revisions' => OrderRevision::whereHas('order', function ($query) use ($clientId) {
                $query->where('user_id', $clientId);
            })->whereIn('status', ['requested', 'in_progress'])->count(),
            
            'completed_revisions' => OrderRevision::whereHas('order', function ($query) use ($clientId) {
                $query->where('user_id', $clientId);
            })->where('status', 'completed')->count(),
        ];

        $recentRevisions = OrderRevision::whereHas('order', function ($query) use ($clientId) {
            $query->where('user_id', $clientId);
        })
        ->with(['order', 'assignedTo'])
        ->latest()
        ->take(5)
        ->get();

        return Inertia::render('Client/Revisions/Dashboard', [
            'stats' => $stats,
            'recentRevisions' => $recentRevisions,
        ]);
    }

    /**
     * Check if the order can have a revision requested.
     */
    private function canRequestRevision(Order $order): bool
    {
        // Check if order status allows revisions
        if (!in_array($order->order_status, ['under_review', 'completed', 'delivered_to_client'])) {
            return false;
        }

        // Check revision limit
        $currentRevisions = $order->revisions()->count();
        if ($currentRevisions >= config('revisions.max_per_order')) {
            return false;
        }

        // Check if there are no active revisions
        $activeRevisions = $order->revisions()
            ->whereIn('status', ['requested', 'in_progress'])
            ->count();

        return $activeRevisions === 0;
    }

    /**
     * Create a conversation thread for the revision.
     */
    private function createRevisionConversation(OrderRevision $revision): \App\Models\Conversation
    {
        $conversation = \App\Models\Conversation::create([
            'title' => "Revision #{$revision->revision_number} - Order #{$revision->order->order_number}",
            'type' => 'revision',
            'is_private' => false,
        ]);

        // Add participants
        $conversation->participants()->createMany([
            ['user_id' => $revision->order->user_id, 'role' => 'client'],
            ['user_id' => $revision->order->writer_id, 'role' => 'writer'],
        ]);

        return $conversation;
    }

    /**
     * Notify admin of new revision request.
     */
    private function notifyAdminOfRevision(OrderRevision $revision): void
    {
        try {
            // Get admin users
            $admins = \App\Models\User::where('role', 'admin')->get();
            
            foreach ($admins as $admin) {
                $admin->notify(new \App\Notifications\AdminRevisionRequestedNotification($revision));
            }
        } catch (\Exception $e) {
            Log::error('Error sending admin revision notification: ' . $e->getMessage(), [
                'revision_id' => $revision->id,
                'exception' => $e
            ]);
        }
    }
} 