<?php

namespace App\Http\Controllers;

use Inertia\Inertia;
use Illuminate\Http\Request;
use App\Models\ContactMessage;
use App\Models\Page;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;
use App\Notifications\ContactMessageNotification;
use App\Notifications\ContactMessageAdminNotification;
use Illuminate\Validation\ValidationException;

class ContactController extends Controller
{
    /**
     * Display the contact page
     */
    public function index()
    {
        // Try to load from CMS first
        $cmsPage = Page::where('slug', 'contact')
            ->with(['contentBlocks' => function($query) {
                $query->where('is_active', true)->orderBy('order');
            }, 'seo'])
            ->first();

        if ($cmsPage && $cmsPage->seo) {
            // Use CMS data
            return Inertia::render('Contact', [
                'page' => $cmsPage,
                'contentBlocks' => $cmsPage->contentBlocks,
                'seoData' => [
                    'title' => $cmsPage->seo->meta_title,
                    'description' => $cmsPage->seo->meta_description,
                    'canonicalUrl' => $cmsPage->seo->canonical_url,
                    'ogUrl' => $cmsPage->seo->canonical_url,
                    'ogTitle' => $cmsPage->seo->og_title,
                    'ogDescription' => $cmsPage->seo->og_description,
                    'ogImage' => $cmsPage->seo->og_image,
                ],
                'openGraph' => [
                    'title' => $cmsPage->seo->og_title,
                    'description' => $cmsPage->seo->og_description,
                    'image' => $cmsPage->seo->og_image,
                ],
                'structuredData' => $cmsPage->seo->structured_data ?? [],
                'isCMS' => true
            ]);
        }

        // Fallback to static data
        return Inertia::render('Contact', [
            'seoData' => [
                'title' => 'Contact Us - Academic Scribe',
                'description' => 'Get in touch with Academic Scribe. We\'re here to help with your academic writing needs.',
                'canonicalUrl' => url('/contact'),
                'ogUrl' => url('/contact'),
            ],
            'isCMS' => false
        ]);
    }

    /**
     * Store a new contact message
     */
    public function store(Request $request)
    {
        try {
            // Validate the request
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'subject' => 'required|string|max:255',
                'message' => 'required|string|max:5000',
                'phone' => 'nullable|string|max:20',
            ], [
                'name.required' => 'Please enter your name.',
                'email.required' => 'Please enter your email address.',
                'email.email' => 'Please enter a valid email address.',
                'subject.required' => 'Please enter a subject.',
                'message.required' => 'Please enter your message.',
                'message.max' => 'Message cannot exceed 5000 characters.',
            ]);

            // Create the contact message
            $contactMessage = ContactMessage::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'subject' => $validated['subject'],
                'message' => $validated['message'],
                'phone' => $validated['phone'] ?? null,
                'status' => 'new',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            // Send notification to user (confirmation email)
            try {
                Notification::route('mail', $contactMessage->email)
                    ->notify(new ContactMessageNotification($contactMessage));
            } catch (\Exception $e) {
                Log::error('Failed to send contact confirmation email: ' . $e->getMessage(), [
                    'contact_message_id' => $contactMessage->id,
                    'email' => $contactMessage->email
                ]);
            }

            // Send notification to admin
            try {
                // Get admin email from settings or use default
                $adminEmail = config('ADMIN_EMAIL', 'academicscribe77@gmail.com');
                Notification::route('mail', $adminEmail)
                    ->notify(new ContactMessageAdminNotification($contactMessage));
            } catch (\Exception $e) {
                Log::error('Failed to send contact admin notification: ' . $e->getMessage(), [
                    'contact_message_id' => $contactMessage->id
                ]);
            }

            // Log successful contact submission
            Log::info('Contact message submitted successfully', [
                'contact_message_id' => $contactMessage->id,
                'email' => $contactMessage->email,
                'subject' => $contactMessage->subject
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Thank you for contacting us! We have received your message and will get back to you soon.',
                'contact_id' => $contactMessage->id
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Please check your input and try again.',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Contact form submission failed: ' . $e->getMessage(), [
                'request_data' => $request->except(['_token']),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Sorry, there was an error sending your message. Please try again later.'
            ], 500);
        }
    }

    /**
     * Rate limiting helper - check if user has submitted too many messages
     */
    private function checkRateLimit(Request $request): bool
    {
        $ipAddress = $request->ip();
        $recentMessages = ContactMessage::where('ip_address', $ipAddress)
            ->where('created_at', '>=', now()->subHour())
            ->count();

        return $recentMessages >= 3; // Max 3 messages per hour per IP
    }
}
