<?php

namespace App\Http\Controllers;

use App\Models\Bid;
use App\Models\User;
use Inertia\Inertia;
use App\Models\Order;
use App\Models\Payment;
use Illuminate\Support\Carbon;
use App\Services\CouponService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;


class DashboardController extends Controller
{
    protected $couponService;

    /**
     * Create a new controller instance.
     *
     * @param CouponService $couponService
     * @return void
     */
    public function __construct(CouponService $couponService)
    {
        $this->couponService = $couponService;
        $this->middleware(['auth', 'verified', 'client']);
    }


    /**
     * Display the user dashboard
     *
     * @return \Inertia\Response
     */
    public function index()
    {

        try {
            // Get authenticated user
            $user = Auth::user();

            if (!$user) {
                return redirect()->route('login');
            }

            // Gather all dashboard data
            return Inertia::render('Dashboard', [
                'orderStats' => $this->getOrderStatistics($user),
                'recentOrders' => $this->getRecentOrders($user),
                'availableCoupons' => $this->getAvailableCoupons($user),
                'loyaltyInfo' => $this->getLoyaltyInformation($user),
                'preferredWriters' => $this->getPreferredWriters($user),
                'pendingBids' => $this->getPendingBids($user),
                'financialSummary' => $this->getFinancialSummary($user),
                'orderActivityData' => $this->getOrderActivityData($user),
            ]);
        } catch (\Exception $e) {
            // Log the error
            Log::error('Dashboard error: ' . $e->getMessage(), [
                'user_id' => Auth::id() ?? 'unauthenticated',
                'trace' => $e->getTraceAsString()
            ]);

            // Return dashboard with error message
            return Inertia::render('Dashboard', [
                'error' => 'There was an error loading your dashboard. Please try again later.',
                'orderStats' => [],
                'recentOrders' => [],
                'availableCoupons' => [],
                'loyaltyInfo' => null,
                'preferredWriters' => [],
                'pendingBids' => [],
                'financialSummary' => null,
                'orderActivityData' => [],
            ]);
        }
    }

    /**
     * Get order statistics for the user.
     *
     * @param User $user
     * @return array
     */
    private function getOrderStatistics(User $user): array
    {
        $orders = $user->orders();

        // Calculate total spent from payments table (source of truth) instead of orders table
        // This ensures consistency with Financial Summary
        $totalSpent = Payment::whereHas('order', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })
            ->where('status', Payment::STATUS_COMPLETED)
            ->sum('amount');

        return [
            'total' => $orders->count(),
            'approved' => $orders->where('order_status', 'approved')->count(),
            'completed' => $orders->where('order_status', 'completed')->count(),
            'pending' => $orders->whereIn('order_status', ['pending', 'bidding'])->count(),
            'cancelled' => $orders->where('order_status', 'cancelled')->count(),
            'totalSpent' => $totalSpent,
        ];
    }

    /**
     * Get recent orders for the user.
     *
     * @param User $user
     * @return array
     */
    private function getRecentOrders(User $user): array
    {
        return $user->orders()
            ->latest('dateposted')
            ->take(5)
            ->with(['bids' => function ($query) {
                $query->count();
            }])
            ->get()
            ->map(function ($order) {
                $deadline = $order->deadline ? Carbon::parse($order->deadline) : null;

                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'title' => $order->title,
                    'type_of_paper' => $order->type_of_paper,
                    'status' => $order->order_status,
                    'payment_status' => $order->payment_status,
                    'deadline' => $deadline ? $deadline->format('M d, Y H:i') : null,
                    'deadline_timestamp' => $deadline ? $deadline->timestamp : null,
                    'net_amount' => $order->net_amount,
                    'currency' => $order->currency ?? 'USD',
                    'bids_count' => $order->bids->count(),
                    'is_late' => $deadline ? $deadline->isPast() : false,
                    'days_remaining' => $deadline ? now()->diffInDays($deadline, false) : null,
                ];
            })
            ->toArray();
    }

    /**
     * Get available coupons for the user.
     *
     * @param User $user
     * @return array
     */
    private function getAvailableCoupons(User $user): array
    {
        // Get order context for coupon applicability
        $orderContext = [
            'order_total' => 0, // Default
            'pages' => 1,       // Default
            'previous_orders' => $user->orders()->count()
        ];

        // Get applicable coupons using the coupon service
        $coupons = $this->couponService->getApplicableCoupons($user, $orderContext);

        return $coupons->map(function ($coupon) {
            $expiryDate = $coupon->expiry_date ? Carbon::parse($coupon->expiry_date) : null;

            return [
                'id' => $coupon->id,
                'code' => $coupon->coupon_code,
                'type' => $coupon->coupon_type,
                'discount_type' => $coupon->discount_type,
                'discount_amount' => $coupon->discount_amount,
                'description' => $this->getCouponDescription($coupon),
                'expiry_date' => $expiryDate ? $expiryDate->format('M d, Y') : null,
                'days_remaining' => $expiryDate ? now()->diffInDays($expiryDate, false) : null,
                'is_expiring_soon' => $expiryDate ? now()->diffInDays($expiryDate) <= 7 : false,
            ];
        })->toArray();
    }

    /**
     * Generate a human-readable description for a coupon.
     *
     * @param \App\Models\Coupon $coupon
     * @return string
     */
    protected function getCouponDescription($coupon): string
    {
        $amount = $coupon->discount_amount;
        $type = $coupon->discount_type === 'percentage' ? '%' : '$';

        switch ($coupon->coupon_type) {
            case 'first_order':
                return "First order discount: {$amount}{$type} off your first order!";
            case 'flash':
                $expiry = $coupon->expiry_date ? ' until ' . $coupon->expiry_date->format('M d') : '';
                return "Flash sale: {$amount}{$type} off{$expiry}!";
            case 'referral':
                return "Referral bonus: {$amount}{$type} off your order!";
            case 'loyalty':
                if ($coupon->loyaltyTier) {
                    return "{$coupon->loyaltyTier->name} customer: {$amount}{$type} off!";
                }
                return "Loyalty discount: {$amount}{$type} off!";
            case 'anniversary':
                return "Anniversary special: {$amount}{$type} off to celebrate!";
            case 'seasonal':
                if ($coupon->seasonalType) {
                    return "{$coupon->seasonalType->name}: {$amount}{$type} off!";
                }
                return "Seasonal discount: {$amount}{$type} off!";
            case 'volume':
                $pages = $coupon->min_pages ?? 0;
                return "Volume discount: {$amount}{$type} off orders with {$pages}+ pages!";
            default:
                return "Special offer: {$amount}{$type} off your order!";
        }
    }

    /**
     * Get order activity data for the chart (past 6 months)
     *
     * @param User $user
     * @return array
     */
    private function getOrderActivityData(User $user): array
    {
        try {
            // Get the date range (last 6 months)
            $endDate = Carbon::now();
            $startDate = Carbon::now()->subMonths(5)->startOfMonth(); // 6 months including current

            // Initialize result array with all months (even those with zero orders)
            $result = [];
            $currentDate = $startDate->copy();

            while ($currentDate->lte($endDate)) {
                $monthKey = $currentDate->format('M Y');
                $result[$monthKey] = [
                    'month' => $currentDate->format('M'),
                    'year' => $currentDate->format('Y'),
                    'count' => 0,
                    'completed' => 0,
                    'amount' => 0,
                ];
                $currentDate->addMonth();
            }

            // Get orders within date range
            $orders = Order::where('user_id', $user->id)
                ->whereBetween('dateposted', [$startDate, $endDate])
                ->get();

            // Process the orders and aggregate by month
            foreach ($orders as $order) {
                $monthKey = Carbon::parse($order->dateposted)->format('M Y');

                if (isset($result[$monthKey])) {
                    $result[$monthKey]['count']++;

                    if ($order->order_status === 'completed') {
                        $result[$monthKey]['completed']++;
                    }

                    // Add order amount if payment status is paid
                    if ($order->payment_status === 'paid') {
                        $result[$monthKey]['amount'] += $order->net_amount ?? $order->order_amount;
                    }
                }
            }

            // Convert to indexed array for the frontend
            return array_values($result);
        } catch (\Exception $e) {
            Log::error('Error fetching order activity data: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);

            // Return empty array on error
            return [];
        }
    }


    /**
     * Get loyalty information for the user.
     *
     * @param User $user
     * @return array|null
     */
    private function getLoyaltyInformation(User $user): ?array
    {
        // Get current loyalty tier from coupon service
        $loyaltyTier = $this->couponService->getUserLoyaltyTier($user);

        if (!$loyaltyTier) {
            return null;
        }

        // Get completed orders count
        $completedOrdersCount = $user->orders()
            ->where('payment_status', 'paid')
            ->count();

        // Get next tier if available
        $nextTier = null;
        if ($loyaltyTier->required_orders !== null) {
            $nextTier = \App\Models\LoyaltyTier::where('required_orders', '>', $loyaltyTier->required_orders)
                ->orderBy('required_orders')
                ->first();
        }

        return [
            'current_tier' => [
                'id' => $loyaltyTier->id,
                'name' => $loyaltyTier->name,
                'description' => $loyaltyTier->description,
                'required_orders' => $loyaltyTier->required_orders,
            ],
            'completed_orders' => $completedOrdersCount,
            'next_tier' => $nextTier ? [
                'id' => $nextTier->id,
                'name' => $nextTier->name,
                'description' => $nextTier->description,
                'required_orders' => $nextTier->required_orders,
                'orders_needed' => $nextTier->required_orders - $completedOrdersCount,
            ] : null,
            'progress_percentage' => $nextTier
                ? min(100, round(($completedOrdersCount / $nextTier->required_orders) * 100))
                : 100,
        ];
    }

    /**
     * Get preferred writers for the user (using nicknames only).
     *
     * @param User $user
     * @return array
     */
    private function getPreferredWriters(User $user): array
    {
        // Get orders with assigned writers that are completed
        $completedOrders = $user->orders()
            ->whereNotNull('writer_id')
            ->where('order_status', 'completed')
            ->with('writer:id,name')
            ->get();

        // Group by writer and count orders
        $writers = [];
        $writerCounts = [];

        foreach ($completedOrders as $order) {
            if (!$order->writer_id) continue;

            $writerId = $order->writer_id;

            if (!isset($writerCounts[$writerId])) {
                // Use nickname instead of real name (for privacy)
                $nickname = "Writer #" . substr(md5($writerId), 0, 6);

                $writers[$writerId] = [
                    'id' => $writerId,
                    'nickname' => $nickname,
                    'completed_orders' => 0,
                    'last_order_date' => null,
                ];
                $writerCounts[$writerId] = 0;
            }

            $writerCounts[$writerId]++;
            $writers[$writerId]['completed_orders'] = $writerCounts[$writerId];

            // Track most recent order date
            $orderDate = $order->dateposted;
            if (
                !$writers[$writerId]['last_order_date'] ||
                $orderDate > $writers[$writerId]['last_order_date']
            ) {
                $writers[$writerId]['last_order_date'] = $orderDate;
            }
        }

        // Sort by number of completed orders (descending)
        usort($writers, function ($a, $b) {
            return $b['completed_orders'] - $a['completed_orders'];
        });

        // Return top 3 most used writers
        return array_slice(array_values($writers), 0, 3);
    }

    /**
     * Get pending bids for the user's orders.
     *
     * @param User $user
     * @return array
     */
    private function getPendingBids(User $user): array
    {
        $pendingBids = Bid::whereHas('order', function ($query) use ($user) {
            $query->where('user_id', $user->id)
                ->where('order_status', 'bidding');
        })
            ->with(['order:id,order_number,title'])
            ->get()
            ->groupBy('order_id')
            ->map(function ($bids, $orderId) {
                $order = $bids->first()->order;

                return [
                    'order_id' => $orderId,
                    'order_number' => $order->order_number,
                    'order_title' => $order->title,
                    'bids_count' => $bids->count(),
                ];
            })
            ->values()
            ->toArray();

        return [
            'total_orders_with_bids' => count($pendingBids),
            'total_bids' => collect($pendingBids)->sum('bids_count'),
            'orders' => $pendingBids,
        ];
    }

    /**
     * Get financial summary for the user (placeholder implementation).
     *
     * @param User $user
     * @return array
     */
    private function getFinancialSummary(User $user): array
    {
        // Get recent completed payments for this user
        $recentPayments = Payment::whereHas('order', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })
            ->where('status', Payment::STATUS_COMPLETED)
            ->with('order')
            ->orderBy('payment_date', 'desc')
            ->take(5)
            ->get();

        // Calculate total spending - use sum of all completed payments, not just recent 5
        $totalSpent = Payment::whereHas('order', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })
            ->where('status', Payment::STATUS_COMPLETED)
            ->sum('amount');

        // Format payment data for the frontend
        $paymentItems = $recentPayments->map(function ($payment) {
            return [
                'id' => $payment->id,
                'order_id' => $payment->order_id,
                'order_number' => $payment->order ? $payment->order->order_number : 'N/A',
                'amount' => $payment->amount,
                'currency' => $payment->currency,
                'method' => $payment->payment_method,
                'date' => $payment->payment_date->format('Y-m-d'),
                'transaction_id' => $payment->transaction_id,
                'status' => $payment->status,
                // Get refund status from the payment
                'is_refunded' => $payment->wasRefunded(),
                'is_partially_refunded' => $payment->isPartiallyRefunded(),
            ];
        });

        // Get saved payment methods if applicable
        $paymentMethods = []; // In a real implementation, you would get this from a payment provider API

        return [
            'wallet' => [
                'balance' => 0, // For future wallet feature
                'currency' => 'USD',
                'status' => 'Coming Soon',
            ],
            'payment_methods' => [
                'count' => count($paymentMethods),
                'methods' => $paymentMethods,
            ],
            'recent_payments' => [
                'total' => $user->orders()->where('payment_status', 'paid')->count(),
                'total_spent' => $totalSpent,
                'last_transaction' => $recentPayments->first() ? [
                    'amount' => $recentPayments->first()->amount,
                    'date' => $recentPayments->first()->payment_date->format('M d, Y'),
                    'method' => ucfirst($recentPayments->first()->payment_method)
                ] : null,
                'items' => $paymentItems,
            ],
        ];
    }
}
