<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Services\ImageOptimizationService;
use App\Helpers\ImageHelper;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ImageController extends Controller
{
    protected $imageService;

    public function __construct(ImageOptimizationService $imageService)
    {
        $this->imageService = $imageService;
    }

    /**
     * Get optimized image data for a specific image
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getOptimizedImage(Request $request): JsonResponse
    {
        $request->validate([
            'path' => 'required|string',
            'type' => 'sometimes|string|in:hero,blog,service,thumbnail,avatar,logo',
            'alt' => 'sometimes|string|max:255',
        ]);

        $imagePath = $request->input('path');
        $conversionType = $request->input('type', 'blog');
        $alt = $request->input('alt', '');

        try {
            // Cache key for this specific image configuration
            $cacheKey = "optimized_image:" . md5($imagePath . $conversionType . $alt);

            $imageData = Cache::remember($cacheKey, 3600, function () use ($imagePath, $conversionType, $alt) {
                return $this->imageService->getResponsiveImageData($imagePath, $conversionType, $alt);
            });

            return response()->json([
                'success' => true,
                'data' => $imageData,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to get optimized image data', [
                'path' => $imagePath,
                'type' => $conversionType,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to optimize image',
                'fallback' => [
                    'sources' => [],
                    'fallback' => [
                        'src' => $imagePath,
                        'alt' => $alt,
                    ],
                ],
            ], 500);
        }
    }

    /**
     * Get multiple optimized images at once
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getBatchOptimizedImages(Request $request): JsonResponse
    {
        $request->validate([
            'images' => 'required|array|max:10',
            'images.*.path' => 'required|string',
            'images.*.type' => 'sometimes|string|in:hero,blog,service,thumbnail,avatar,logo',
            'images.*.alt' => 'sometimes|string|max:255',
        ]);

        $images = $request->input('images');
        $results = [];

        foreach ($images as $index => $imageConfig) {
            $imagePath = $imageConfig['path'];
            $conversionType = $imageConfig['type'] ?? 'blog';
            $alt = $imageConfig['alt'] ?? '';

            try {
                $cacheKey = "optimized_image:" . md5($imagePath . $conversionType . $alt);

                $imageData = Cache::remember($cacheKey, 3600, function () use ($imagePath, $conversionType, $alt) {
                    return $this->imageService->getResponsiveImageData($imagePath, $conversionType, $alt);
                });

                $results[$index] = [
                    'success' => true,
                    'data' => $imageData,
                ];
            } catch (\Exception $e) {
                Log::error('Failed to get optimized image data in batch', [
                    'path' => $imagePath,
                    'type' => $conversionType,
                    'error' => $e->getMessage(),
                ]);

                $results[$index] = [
                    'success' => false,
                    'fallback' => [
                        'sources' => [],
                        'fallback' => [
                            'src' => $imagePath,
                            'alt' => $alt,
                        ],
                    ],
                ];
            }
        }

        return response()->json([
            'success' => true,
            'data' => $results,
        ]);
    }

    /**
     * Convert a static image to optimized format
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function convertStaticImage(Request $request): JsonResponse
    {
        $request->validate([
            'path' => 'required|string',
            'type' => 'sometimes|string|in:hero,blog,service,thumbnail,avatar,logo',
            'collection' => 'sometimes|string|max:50',
        ]);

        $imagePath = $request->input('path');
        $conversionType = $request->input('type', 'blog');
        $collection = $request->input('collection', 'images');

        try {
            $optimizedUrls = $this->imageService->convertStaticImage($imagePath, $conversionType, $collection);

            // Clear related cache
            $cachePattern = "optimized_image:*" . md5($imagePath) . "*";
            Cache::flush(); // In production, use more specific cache clearing

            return response()->json([
                'success' => true,
                'message' => 'Image converted successfully',
                'data' => $optimizedUrls,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to convert static image', [
                'path' => $imagePath,
                'type' => $conversionType,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to convert image: ' . $e->getMessage(),
            ], 500);
        }
    }
}
