<?php

namespace App\Http\Controllers\Messaging;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MessagingAnalyticsController extends Controller
{
    /**
     * Get comprehensive messaging analytics for admins.
     */
    public function index(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        // Only admins can access analytics
        if (!in_array($user->user_type, ['admin', 'super_admin'])) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $timeframe = $request->get('timeframe', '30'); // days
        $startDate = Carbon::now()->subDays($timeframe);

        $analytics = [
            'summary' => $this->getSummaryStats($startDate),
            'conversation_trends' => $this->getConversationTrends($startDate),
            'response_times' => $this->getResponseTimeStats($startDate),
            'user_activity' => $this->getUserActivityStats($startDate),
            'status_distribution' => $this->getStatusDistribution(),
            'resolution_rates' => $this->getResolutionRates($startDate),
            'peak_hours' => $this->getPeakHours($startDate),
            'participant_types' => $this->getParticipantTypeStats(),
        ];

        return response()->json($analytics);
    }

    /**
     * Get summary statistics.
     */
    private function getSummaryStats(Carbon $startDate): array
    {
        $totalConversations = Conversation::count();
        $newConversations = Conversation::where('created_at', '>=', $startDate)->count();
        $totalMessages = Message::count();
        $newMessages = Message::where('created_at', '>=', $startDate)->count();
        
        $activeConversations = Conversation::where('status', 'active')->count();
        $resolvedConversations = Conversation::where('status', 'resolved')->count();
        $escalatedConversations = Conversation::where('status', 'escalated')->count();

        return [
            'total_conversations' => $totalConversations,
            'new_conversations' => $newConversations,
            'total_messages' => $totalMessages,
            'new_messages' => $newMessages,
            'active_conversations' => $activeConversations,
            'resolved_conversations' => $resolvedConversations,
            'escalated_conversations' => $escalatedConversations,
            'resolution_rate' => $totalConversations > 0 ? round(($resolvedConversations / $totalConversations) * 100, 2) : 0,
        ];
    }

    /**
     * Get conversation trends over time.
     */
    private function getConversationTrends(Carbon $startDate): array
    {
        $trends = Conversation::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as count'),
                'status'
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('date', 'status')
            ->orderBy('date')
            ->get()
            ->groupBy('date');

        $formatted = [];
        foreach ($trends as $date => $statusData) {
            $formatted[$date] = [
                'date' => $date,
                'total' => $statusData->sum('count'),
                'by_status' => $statusData->pluck('count', 'status')->toArray(),
            ];
        }

        return array_values($formatted);
    }

    /**
     * Get response time statistics.
     */
    private function getResponseTimeStats(Carbon $startDate): array
    {
        // Calculate average response time between messages in conversations
        $responseStats = DB::select("
            SELECT 
                AVG(TIMESTAMPDIFF(MINUTE, prev_message.created_at, curr_message.created_at)) as avg_response_minutes,
                MIN(TIMESTAMPDIFF(MINUTE, prev_message.created_at, curr_message.created_at)) as min_response_minutes,
                MAX(TIMESTAMPDIFF(MINUTE, prev_message.created_at, curr_message.created_at)) as max_response_minutes
            FROM messages curr_message
            JOIN messages prev_message ON curr_message.conversation_id = prev_message.conversation_id
            WHERE curr_message.id > prev_message.id 
            AND curr_message.sender_id != prev_message.sender_id
            AND curr_message.created_at >= ?
            AND NOT EXISTS (
                SELECT 1 FROM messages middle_message 
                WHERE middle_message.conversation_id = curr_message.conversation_id 
                AND middle_message.id > prev_message.id 
                AND middle_message.id < curr_message.id
            )
        ", [$startDate]);

        $stats = $responseStats[0] ?? null;

        return [
            'avg_response_minutes' => $stats ? round($stats->avg_response_minutes, 2) : 0,
            'min_response_minutes' => $stats ? $stats->min_response_minutes : 0,
            'max_response_minutes' => $stats ? $stats->max_response_minutes : 0,
            'avg_response_hours' => $stats ? round($stats->avg_response_minutes / 60, 2) : 0,
        ];
    }

    /**
     * Get user activity statistics.
     */
    private function getUserActivityStats(Carbon $startDate): array
    {
        $userStats = User::select('user_type')
            ->withCount(['messages as messages_sent' => function ($query) use ($startDate) {
                $query->where('created_at', '>=', $startDate);
            }])
            ->withCount(['participatingConversations as conversations_participated'])
            ->having('messages_sent', '>', 0)
            ->get()
            ->groupBy('user_type');

        $formatted = [];
        foreach ($userStats as $userType => $users) {
            $formatted[$userType] = [
                'total_users' => $users->count(),
                'total_messages' => $users->sum('messages_sent'),
                'avg_messages_per_user' => $users->count() > 0 ? round($users->sum('messages_sent') / $users->count(), 2) : 0,
                'total_conversations' => $users->sum('conversations_participated'),
            ];
        }

        return $formatted;
    }

    /**
     * Get status distribution.
     */
    private function getStatusDistribution(): array
    {
        return Conversation::select('status', DB::raw('COUNT(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
    }

    /**
     * Get resolution rates over time.
     */
    private function getResolutionRates(Carbon $startDate): array
    {
        $resolutionData = Conversation::select(
                DB::raw('DATE(resolved_at) as date'),
                DB::raw('COUNT(*) as resolved_count'),
                DB::raw('AVG(TIMESTAMPDIFF(HOUR, created_at, resolved_at)) as avg_resolution_hours')
            )
            ->whereNotNull('resolved_at')
            ->where('resolved_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return $resolutionData->map(function ($item) {
            return [
                'date' => $item->date,
                'resolved_count' => $item->resolved_count,
                'avg_resolution_hours' => round($item->avg_resolution_hours, 2),
            ];
        })->toArray();
    }

    /**
     * Get peak hours analysis.
     */
    private function getPeakHours(Carbon $startDate): array
    {
        $hourlyData = Message::select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('COUNT(*) as message_count')
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();

        return $hourlyData->map(function ($item) {
            return [
                'hour' => $item->hour,
                'message_count' => $item->message_count,
                'period' => $item->hour < 12 ? 'AM' : 'PM',
                'display_hour' => $item->hour == 0 ? 12 : ($item->hour > 12 ? $item->hour - 12 : $item->hour),
            ];
        })->toArray();
    }

    /**
     * Get participant type statistics.
     */
    private function getParticipantTypeStats(): array
    {
        $participantStats = DB::select("
            SELECT 
                u1.user_type as user1_type,
                u2.user_type as user2_type,
                COUNT(*) as conversation_count
            FROM conversations c
            JOIN conversation_participants cp1 ON c.id = cp1.conversation_id
            JOIN conversation_participants cp2 ON c.id = cp2.conversation_id
            JOIN users u1 ON cp1.user_id = u1.id
            JOIN users u2 ON cp2.user_id = u2.id
            WHERE cp1.user_id < cp2.user_id
            GROUP BY u1.user_type, u2.user_type
            ORDER BY conversation_count DESC
        ");

        return collect($participantStats)->map(function ($stat) {
            return [
                'participant_types' => "{$stat->user1_type} ↔ {$stat->user2_type}",
                'conversation_count' => $stat->conversation_count,
            ];
        })->toArray();
    }

    /**
     * Export analytics data.
     */
    public function export(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        if (!in_array($user->user_type, ['admin', 'super_admin'])) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        if (!config('messaging.analytics.export_enabled')) {
            return response()->json(['error' => 'Export functionality is disabled'], 403);
        }

        $format = $request->get('format', 'json');
        $timeframe = $request->get('timeframe', '30');
        
        if (!in_array($format, config('messaging.analytics.export_formats', ['json']))) {
            return response()->json(['error' => 'Unsupported export format'], 400);
        }

        $startDate = Carbon::now()->subDays($timeframe);
        $analytics = $this->getSummaryStats($startDate);

        // For now, return JSON. CSV and PDF export can be implemented later
        return response()->json([
            'export_data' => $analytics,
            'exported_at' => now(),
            'timeframe_days' => $timeframe,
            'format' => $format,
        ]);
    }
}
