<?php

namespace App\Http\Controllers\Messaging;

use App\Http\Controllers\Controller;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\User;
use App\Events\MessageSent;
use App\Events\MessageRead;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class MessagingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Get conversations for the authenticated user.
     */
    public function getConversations(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        // By default, show active conversations for non-admins, non-archived for admins
        $isAdmin = in_array($user->user_type, ['admin', 'super_admin']);
        $defaultStatus = $isAdmin ? 'not_archived' : 'active';
        $statusFilter = $request->get('status', $defaultStatus);
        
        $query = Conversation::forUser($user)
            ->with(['participants', 'lastMessage.sender']);
            
        // Apply status filtering
        if ($statusFilter === 'not_archived') {
            $query->notArchived();
        } elseif ($statusFilter !== 'all') {
            $query->where('status', $statusFilter);
        }
        
        // Apply search filtering
        if ($search = $request->get('search')) {
            $query->whereHas('participants', function ($q) use ($search, $user) {
                $q->join('users', 'conversation_participants.user_id', '=', 'users.id')
                  ->where('users.id', '!=', $user->id)
                  ->where(function ($subQ) use ($search) {
                      $subQ->where('users.name', 'LIKE', "%{$search}%")
                           ->orWhere('users.email', 'LIKE', "%{$search}%");
                  });
            });
        }
        
        // Apply date range filtering
        if ($request->has('from_date') && $request->get('from_date')) {
            $query->whereDate('created_at', '>=', $request->get('from_date'));
        }
        
        if ($request->has('to_date') && $request->get('to_date')) {
            $query->whereDate('created_at', '<=', $request->get('to_date'));
        }
        
        $conversations = $query
            ->orderBy('status')  // Show active first, then resolved, then escalated
            ->orderBy('last_activity_at', 'desc')
            ->orderBy('updated_at', 'desc')
            ->get()
            ->map(function ($conversation) use ($user) {
                $otherParticipant = $conversation->getOtherParticipant($user);
                $unreadCount = $conversation->getUnreadCountForUser($user);
                
                return [
                    'id' => $conversation->id,
                    'conversation_type' => $conversation->conversation_type,
                    'title' => $conversation->title,
                    'status' => $conversation->status,
                    'resolved_at' => $conversation->resolved_at,
                    'resolved_by' => $conversation->resolved_by,
                    'archived_at' => $conversation->archived_at,
                    'archived_by' => $conversation->archived_by,
                    'last_activity_at' => $conversation->last_activity_at,
                    'priority_level' => $conversation->priority_level,
                    'participants' => $conversation->participants->map(function ($participant) {
                        return [
                            'id' => $participant->id,
                            'name' => $participant->name,
                            'user_type' => $participant->pivot->user_type,
                        ];
                    }),
                    'last_message' => $conversation->lastMessage ? [
                        'id' => $conversation->lastMessage->id,
                        'content' => $conversation->lastMessage->content,
                        'sender_id' => $conversation->lastMessage->sender_id,
                        'sender_name' => $conversation->lastMessage->sender->name,
                        'message_type' => $conversation->lastMessage->message_type,
                        'created_at' => $conversation->lastMessage->created_at,
                    ] : null,
                    'unread_count' => $unreadCount,
                    'updated_at' => $conversation->updated_at,
                ];
            });

        // Calculate conversation statistics for admins
        $stats = [];
        if ($isAdmin) {
            $statsQuery = Conversation::forUser($user);
            
            $stats = [
                'total' => $statsQuery->count(),
                'active' => $statsQuery->clone()->active()->count(),
                'resolved' => $statsQuery->clone()->resolved()->count(),
                'escalated' => $statsQuery->clone()->escalated()->count(),
                'archived' => $statsQuery->clone()->archived()->count(),
            ];
        }

        return response()->json([
            'conversations' => $conversations,
            'stats' => $stats,
        ]);
    }

    /**
     * Get messages for a specific conversation.
     */
    public function getMessages(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Check if user is a participant in this conversation
        if (!$conversation->hasParticipant($user)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $messages = $conversation->messages()
            ->with(['sender', 'reads'])
            ->orderBy('created_at', 'asc')
            ->get()
            ->map(function ($message) use ($user) {
                return [
                    'id' => $message->id,
                    'content' => $message->content,
                    'sender_id' => $message->sender_id,
                    'sender_name' => $message->sender->name,
                    'sender_type' => $message->sender_type,
                    'message_type' => $message->message_type,
                    'file_path' => $message->file_path,
                    'file_name' => $message->file_name,
                    'file_size' => $message->formatted_file_size,
                    'read_at' => $message->isReadBy($user) ? $message->reads->where('user_id', $user->id)->first()?->read_at : null,
                    'created_at' => $message->created_at,
                ];
            });

        // Mark messages as read
        $conversation->messages()
            ->unreadForUser($user)
            ->get()
            ->each(function ($message) use ($user, $conversation) {
                $message->markAsReadBy($user);
                // Broadcast read event
                broadcast(new MessageRead($message, $conversation, $user->id))->toOthers();
            });

        return response()->json([
            'messages' => $messages,
        ]);
    }

    /**
     * Send a message in a conversation.
     */
    public function sendMessage(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Check if user is a participant in this conversation
        if (!$conversation->hasParticipant($user)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'content' => 'required_without:file|string|max:1000',
            'file' => 'nullable|file|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $messageData = [
            'conversation_id' => $conversation->id,
            'sender_id' => $user->id,
            'sender_type' => $user->user_type,
            'content' => $request->input('content', ''),
            'message_type' => 'text',
        ];

        // Handle file upload
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('messages', $fileName, 'public');

            $messageData['message_type'] = 'file';
            $messageData['file_path'] = $filePath;
            $messageData['file_name'] = $file->getClientOriginalName();
            $messageData['file_size'] = $file->getSize();
            $messageData['content'] = 'File: ' . $file->getClientOriginalName();
        }

        $message = Message::create($messageData);
        $message->load('sender');

        // Update conversation's updated_at timestamp and last activity
        $conversation->touch();
        $conversation->updateLastActivity();

        // Broadcast the message to other participants
        broadcast(new MessageSent($message, $conversation))->toOthers();

        return response()->json([
            'message' => [
                'id' => $message->id,
                'content' => $message->content,
                'sender_id' => $message->sender_id,
                'sender_name' => $user->name,
                'sender_type' => $message->sender_type,
                'message_type' => $message->message_type,
                'file_path' => $message->file_path,
                'file_name' => $message->file_name,
                'file_size' => $message->formatted_file_size,
                'created_at' => $message->created_at,
            ],
        ]);
    }

    /**
     * Get unread message count for the authenticated user.
     */
    public function getUnreadCount(): JsonResponse
    {
        try {
            $user = Auth::user();
            
            // Simple approach: count messages where sender is not the current user
            // and the user is a participant in the conversation
            $unreadCount = Message::where('sender_id', '!=', $user->id)
                ->whereHas('conversation.participants', function ($query) use ($user) {
                    $query->where('user_id', $user->id);
                })
                ->count();

            return response()->json([
                'count' => $unreadCount,
            ]);
        } catch (\Exception $e) {
            \Log::error('Error getting unread count: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return 0 unread messages on error to prevent frontend crashes
            return response()->json([
                'count' => 0,
            ]);
        }
    }

    /**
     * Create a new conversation.
     */
    public function createConversation(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();
            
            $validator = Validator::make($request->all(), [
                'participant_id' => 'required|exists:users,id',
                'conversation_type' => 'required|in:client_admin,writer_admin,admin_admin',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $participant = User::findOrFail($request->participant_id);

            // Check if there's an active conversation between these users
            $existingActiveConversation = Conversation::forUser($user)
                ->whereHas('participants', function ($query) use ($participant) {
                    $query->where('user_id', $participant->id);
                })
                ->where('status', 'active')
                ->first();

            if ($existingActiveConversation) {
                return response()->json([
                    'conversation' => $existingActiveConversation->load('participants'),
                    'message' => 'Active conversation already exists',
                ]);
            }

            // Check for recent resolved conversations that could be reopened
            $recentResolvedConversation = Conversation::forUser($user)
                ->whereHas('participants', function ($query) use ($participant) {
                    $query->where('user_id', $participant->id);
                })
                ->where('status', 'resolved')
                ->where('resolved_at', '>', now()->subHours(config('messaging.automation.reopen_grace_period_hours', 48)))
                ->orderBy('resolved_at', 'desc')
                ->first();

            // If there's a recent resolved conversation and user explicitly wants a new one, create new
            // If user wants to continue the resolved conversation, they can reopen it manually
            // For now, we'll always create a new conversation when requested

            // Create new conversation
            $conversation = Conversation::create([
                'conversation_type' => $request->conversation_type,
                'title' => $request->input('title'),
                'status' => 'active',
                'last_activity_at' => now(),
            ]);

            // Add participants
            $conversation->participants()->attach([
                $user->id => ['user_type' => $user->user_type],
                $participant->id => ['user_type' => $participant->user_type],
            ]);

            // Load the participants for the response
            $conversation->load('participants');

            return response()->json([
                'conversation' => $conversation,
                'message' => 'Conversation created successfully',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error creating conversation: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'request_data' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'error' => 'Failed to create conversation',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get available users for new conversations.
     */
    public function getAvailableUsers(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        $query = User::where('id', '!=', $user->id)
                    ->select('id', 'name', 'email', 'user_type', 'status');

        // Apply user type restrictions based on current user
        if ($user->user_type === 'client') {
            // Clients can only message admins and super_admins
            $query->whereIn('user_type', ['admin', 'super_admin']);
        } elseif ($user->user_type === 'writer') {
            // Writers can only message admins and super_admins
            $query->whereIn('user_type', ['admin', 'super_admin']);
        } elseif ($user->user_type === 'admin' || $user->user_type === 'super_admin') {
            // Admins and super_admins can message anyone (clients, writers, other admins, super_admins)
            $query->whereIn('user_type', ['client', 'writer', 'admin', 'super_admin']);
        } else {
            // Default: only admins and super_admins
            $query->whereIn('user_type', ['admin', 'super_admin']);
        }

        // Filter by status - include active users and users without status (for backward compatibility)
        $query->where(function($q) {
            $q->where('status', 'active')
              ->orWhereNull('status')
              ->orWhere('status', '');
        });

        $users = $query->orderBy('name')->get();

        return response()->json([
            'users' => $users,
        ]);
    }

    /**
     * Mark conversation as resolved.
     */
    public function resolveConversation(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Check if user can resolve this conversation
        if (!$conversation->canBeResolvedBy($user)) {
            return response()->json([
                'error' => 'You are not authorized to resolve this conversation'
            ], 403);
        }

        try {
            $conversation->markAsResolved($user);
            
            return response()->json([
                'message' => 'Conversation marked as resolved',
                'conversation' => $conversation->fresh()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error resolving conversation: ' . $e->getMessage(), [
                'conversation_id' => $conversation->id,
                'user_id' => $user->id,
            ]);
            
            return response()->json([
                'error' => 'Failed to resolve conversation'
            ], 500);
        }
    }

    /**
     * Reopen a resolved conversation.
     */
    public function reopenConversation(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Check if user can reopen this conversation
        if (!$conversation->canBeReopenedBy($user)) {
            return response()->json([
                'error' => 'You are not authorized to reopen this conversation'
            ], 403);
        }

        try {
            $conversation->reopen($user);
            
            return response()->json([
                'message' => 'Conversation reopened',
                'conversation' => $conversation->fresh()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error reopening conversation: ' . $e->getMessage(), [
                'conversation_id' => $conversation->id,
                'user_id' => $user->id,
            ]);
            
            return response()->json([
                'error' => 'Failed to reopen conversation'
            ], 500);
        }
    }

    /**
     * Archive a conversation (admin only).
     */
    public function archiveConversation(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Check if user can archive this conversation
        if (!$conversation->canBeArchivedBy($user)) {
            return response()->json([
                'error' => 'You are not authorized to archive this conversation'
            ], 403);
        }

        try {
            $conversation->archive($user);
            
            return response()->json([
                'message' => 'Conversation archived',
                'conversation' => $conversation->fresh()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error archiving conversation: ' . $e->getMessage(), [
                'conversation_id' => $conversation->id,
                'user_id' => $user->id,
            ]);
            
            return response()->json([
                'error' => 'Failed to archive conversation'
            ], 500);
        }
    }

    /**
     * Escalate a conversation (admin only).
     */
    public function escalateConversation(Request $request, Conversation $conversation): JsonResponse
    {
        $user = Auth::user();

        // Only admins can escalate
        if (!in_array($user->user_type, ['admin', 'super_admin'])) {
            return response()->json([
                'error' => 'You are not authorized to escalate this conversation'
            ], 403);
        }

        $request->validate([
            'priority_level' => 'integer|min:1|max:5'
        ]);

        try {
            $priorityLevel = $request->input('priority_level', 5);
            $conversation->escalate($user, $priorityLevel);
            
            return response()->json([
                'message' => 'Conversation escalated',
                'conversation' => $conversation->fresh()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error escalating conversation: ' . $e->getMessage(), [
                'conversation_id' => $conversation->id,
                'user_id' => $user->id,
            ]);
            
            return response()->json([
                'error' => 'Failed to escalate conversation'
            ], 500);
        }
    }

    /**
     * Bulk archive conversations (admin only).
     */
    public function bulkArchiveConversations(Request $request): JsonResponse
    {
        $user = Auth::user();

        // Only admins can bulk archive
        if (!in_array($user->user_type, ['admin', 'super_admin'])) {
            return response()->json([
                'error' => 'You are not authorized to perform bulk operations'
            ], 403);
        }

        $request->validate([
            'conversation_ids' => 'required|array|min:1',
            'conversation_ids.*' => 'integer|exists:conversations,id'
        ]);

        try {
            $conversationIds = $request->input('conversation_ids');
            
            // Only archive conversations where user is participant
            $conversations = Conversation::whereIn('id', $conversationIds)
                ->whereHas('participants', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                })
                ->get();

            $archivedCount = 0;
            foreach ($conversations as $conversation) {
                if ($conversation->archive($user)) {
                    $archivedCount++;
                }
            }
            
            return response()->json([
                'message' => "Successfully archived {$archivedCount} conversation(s)",
                'archived_count' => $archivedCount
            ]);
        } catch (\Exception $e) {
            \Log::error('Error bulk archiving conversations: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'conversation_ids' => $request->input('conversation_ids')
            ]);
            
            return response()->json([
                'error' => 'Failed to archive conversations'
            ], 500);
        }
    }

    /**
     * Bulk resolve conversations.
     */
    public function bulkResolveConversations(Request $request): JsonResponse
    {
        $user = Auth::user();

        $request->validate([
            'conversation_ids' => 'required|array|min:1',
            'conversation_ids.*' => 'integer|exists:conversations,id'
        ]);

        try {
            $conversationIds = $request->input('conversation_ids');
            
            // Only resolve conversations where user is participant and status is active
            $conversations = Conversation::whereIn('id', $conversationIds)
                ->where('status', 'active')
                ->whereHas('participants', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                })
                ->get();

            $resolvedCount = 0;
            foreach ($conversations as $conversation) {
                if ($conversation->markAsResolved($user)) {
                    $resolvedCount++;
                }
            }
            
            return response()->json([
                'message' => "Successfully resolved {$resolvedCount} conversation(s)",
                'resolved_count' => $resolvedCount
            ]);
        } catch (\Exception $e) {
            \Log::error('Error bulk resolving conversations: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'conversation_ids' => $request->input('conversation_ids')
            ]);
            
            return response()->json([
                'error' => 'Failed to resolve conversations'
            ], 500);
        }
    }
} 