<?php

namespace App\Http\Controllers;

use App\Models\OrderFile;
use App\Models\Order;
use App\Notifications\AdminOrderFileUploadedNotification;
use App\Notifications\ClientWorkCompletedNotification;
use App\Notifications\AdminWorkCompletedNotification;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Notification;

class OrderFilesController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Client file upload endpoint
     */
    public function clientUpload(Request $request)
    {
        try {
            // Log the incoming request for debugging
            Log::info('Client file upload attempt', [
                'has_file' => $request->hasFile('file'),
                'file_size' => $request->hasFile('file') ? $request->file('file')->getSize() : null,
                'file_mime' => $request->hasFile('file') ? $request->file('file')->getMimeType() : null,
                'file_name' => $request->hasFile('file') ? $request->file('file')->getClientOriginalName() : null,
                'order_id' => $request->input('order_id'),
                'file_category' => $request->input('file_category'),
                'uploader_user_type' => $request->input('uploader_user_type'),
                'user_id' => Auth::id(),
            ]);

            // Validate the request
            $validator = Validator::make($request->all(), [
                'file' => [
                    'required',
                    'file',
                    'max:40960', // 40MB in kilobytes
                    function ($attribute, $value, $fail) {
                        $allowedMimes = [
                            'application/msword',
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                            'application/vnd.ms-excel',
                            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                            'application/vnd.ms-powerpoint',
                            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
                            'application/pdf',
                            'application/zip',
                            'application/x-zip-compressed',
                            'application/octet-stream', // Fallback for servers with poor MIME detection
                        ];

                        $allowedExtensions = ['doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'pdf', 'zip'];

                        if (!$value) {
                            $fail('No file was uploaded.');
                            return;
                        }

                        $mimeType = $value->getMimeType();
                        $extension = strtolower($value->getClientOriginalExtension());

                        // If MIME type is octet-stream, validate by extension instead
                        if ($mimeType === 'application/octet-stream') {
                            if (!in_array($extension, $allowedExtensions)) {
                                $fail('The file must be one of the following types: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP');
                            }
                        } else {
                            // Normal MIME type validation
                            if (!in_array($mimeType, $allowedMimes)) {
                                $fail('The file must be one of the following types: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP');
                            }
                        }
                    },
                ],
                'order_id' => [
                    'required',
                    'exists:orders,id',
                    function ($attribute, $value, $fail) {
                        $order = Order::find($value);
                        if (!$order || Auth::id() !== $order->user_id) {
                            $fail('You do not have permission to upload files to this order.');
                        }
                    }
                ],
                'file_category' => [
                    'required',
                    'string',
                    Rule::in(['sample', 'instructions'])
                ],
                'description' => 'nullable|string|max:1000',
                'uploader_user_type' => 'required|string|in:client'
            ]);

            if ($validator->fails()) {
                Log::warning('Client file upload validation failed', [
                    'errors' => $validator->errors()->toArray(),
                    'order_id' => $request->input('order_id'),
                    'user_id' => Auth::id(),
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $file = $request->file('file');
            $orderId = $request->input('order_id');
            $fileCategory = $request->input('file_category');
            $description = $request->input('description');
            $uploaderUserType = $request->input('uploader_user_type');

            // Get order details
            $order = Order::findOrFail($orderId);
            
            // Store file in storage/app/public/orders
            $filePath = $file->store('orders', 'public');

            // Create order file record
            $orderFile = OrderFile::create([
                'order_id' => $orderId,
                'file_name' => $file->getClientOriginalName(),
                'file_type' => $file->getMimeType(),
                'uploader_id' => Auth::id(),
                'uploader_name' => Auth::user()->name,
                'uploader_user_type' => $uploaderUserType,
                'file_url' => $filePath,
                'file_size' => $file->getSize(),
                'file_status' => 'pending',
                'file_category' => $fileCategory,
                'description' => $description,
                'access_control' => 'client_admin', // Client files accessible by client and admin
                'date_uploaded' => now()
            ]);

            // Send notification to admin
            $this->notifyAdminOfFileUpload($order, $orderFile);

            Log::info('Client file uploaded successfully', [
                'order_id' => $orderId,
                'file_id' => $orderFile->id,
                'file_name' => $orderFile->file_name,
                'category' => $fileCategory,
                'client_id' => Auth::id()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully',
                'file' => $orderFile
            ]);

        } catch (\Exception $e) {
            Log::error('Client file upload failed', [
                'error' => $e->getMessage(),
                'order_id' => $request->input('order_id'),
                'client_id' => Auth::id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Writer file upload endpoint
     */
    public function writerUpload(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'file' => [
                    'required',
                    'file',
                    'max:40960', // 40MB in kilobytes
                    function ($attribute, $value, $fail) {
                        $allowedMimes = [
                            'application/msword',
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                            'application/vnd.ms-excel',
                            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                            'application/vnd.ms-powerpoint',
                            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
                            'application/pdf',
                            'application/zip',
                            'application/x-zip-compressed'
                        ];

                        if (!$value) {
                            $fail('No file was uploaded.');
                            return;
                        }

                        $mimeType = $value->getMimeType();
                        if (!in_array($mimeType, $allowedMimes)) {
                            $fail('The file must be one of the following types: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP');
                        }
                    },
                ],
                'order_id' => [
                    'required',
                    'exists:orders,id',
                    function ($attribute, $value, $fail) {
                        $order = Order::find($value);
                        if (!$order || Auth::id() !== $order->writer_id) {
                            $fail('You do not have permission to upload files to this order.');
                        }
                    }
                ],
                'file_category' => [
                    'required',
                    'string',
                    Rule::in(['draft', 'complete'])
                ],
                'description' => 'nullable|string|max:1000',
                'uploader_user_type' => 'required|string|in:writer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $file = $request->file('file');
            $orderId = $request->input('order_id');
            $fileCategory = $request->input('file_category');
            $description = $request->input('description');
            $uploaderUserType = $request->input('uploader_user_type');

            // Get order details
            $order = Order::findOrFail($orderId);
            
            // Store file in storage/app/public/orders
            $filePath = $file->store('orders', 'public');

            // Create order file record
            $orderFile = OrderFile::create([
                'order_id' => $orderId,
                'file_name' => $file->getClientOriginalName(),
                'file_type' => $file->getMimeType(),
                'uploader_id' => Auth::id(),
                'uploader_name' => Auth::user()->name,
                'uploader_user_type' => $uploaderUserType,
                'file_url' => $filePath,
                'file_size' => $file->getSize(),
                'file_status' => 'pending',
                'file_category' => $fileCategory,
                'description' => $description,
                'access_control' => 'writer_admin', // Writer files accessible by writer and admin
                'date_uploaded' => now()
            ]);

            // Handle order status update for complete files
            if ($fileCategory === 'complete') {
                $this->handleCompleteWorkSubmission($order);
            }

            // Send notifications
            $this->notifyAdminOfFileUpload($order, $orderFile);
            
            if ($fileCategory === 'complete') {
                $this->notifyClientOfWorkCompletion($order, $orderFile);
            }

            Log::info('Writer file uploaded successfully', [
                'order_id' => $orderId,
                'file_id' => $orderFile->id,
                'file_name' => $orderFile->file_name,
                'category' => $fileCategory,
                'writer_id' => Auth::id(),
                'order_status_updated' => $fileCategory === 'complete'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully',
                'file' => $orderFile,
                'order_status_updated' => $fileCategory === 'complete'
            ]);

        } catch (\Exception $e) {
            Log::error('Writer file upload failed', [
                'error' => $e->getMessage(),
                'order_id' => $request->input('order_id'),
                'writer_id' => Auth::id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Admin file upload endpoint
     */
    public function adminUpload(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'file' => [
                    'required',
                    'file',
                    'max:40960', // 40MB in kilobytes
                    function ($attribute, $value, $fail) {
                        $allowedMimes = [
                            'application/msword',
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                            'application/vnd.ms-excel',
                            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                            'application/vnd.ms-powerpoint',
                            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
                            'application/pdf',
                            'application/zip',
                            'application/x-zip-compressed'
                        ];

                        if (!$value) {
                            $fail('No file was uploaded.');
                            return;
                        }

                        $mimeType = $value->getMimeType();
                        if (!in_array($mimeType, $allowedMimes)) {
                            $fail('The file must be one of the following types: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP');
                        }
                    },
                ],
                'order_id' => [
                    'required',
                    'exists:orders,id',
                    function ($attribute, $value, $fail) {
                        // Admin and super_admin can upload files to any order
                        if (!in_array(Auth::user()->user_type, ['admin', 'super_admin'])) {
                            $fail('Only admins can use this endpoint.');
                        }
                    }
                ],
                'target_type' => [
                    'required',
                    'string',
                    'in:writer,client'
                ],
                'file_category' => [
                    'required',
                    'string',
                    'in:draft,complete,admin'
                ],
                'description' => 'nullable|string|max:1000',
                'uploader_user_type' => 'required|string|in:admin,super_admin'
            ]);

            if ($validator->fails()) {
                // Log validation errors for debugging
                Log::error('Admin file upload validation failed', [
                    'errors' => $validator->errors()->toArray(),
                    'request_data' => $request->all(),
                    'user_id' => Auth::id(),
                    'user_type' => Auth::user()->user_type ?? 'unknown'
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $file = $request->file('file');
            $orderId = $request->input('order_id');
            $targetType = $request->input('target_type');
            $fileCategory = $request->input('file_category');
            $description = $request->input('description');
            $uploaderUserType = $request->input('uploader_user_type');

            // Get order details
            $order = Order::findOrFail($orderId);
            
            // Store file in storage/app/public/orders
            $filePath = $file->store('orders', 'public');

            // Determine access control based on target type
            $accessControl = $targetType === 'writer' ? 'writer_admin' : 'client_admin';

            // Create order file record
            $orderFile = OrderFile::create([
                'order_id' => $orderId,
                'file_name' => $file->getClientOriginalName(),
                'file_type' => $file->getMimeType(),
                'uploader_id' => Auth::id(),
                'uploader_name' => Auth::user()->name,
                'uploader_user_type' => $uploaderUserType,
                'file_url' => $filePath,
                'file_size' => $file->getSize(),
                'file_status' => 'pending',
                'file_category' => $fileCategory,
                'description' => $description,
                'access_control' => $accessControl,
                'date_uploaded' => now()
            ]);

            // Handle order status update for complete files uploaded for client
            if ($targetType === 'client' && $fileCategory === 'complete') {
                $this->handleAdminCompleteWorkSubmission($order, $orderFile);
            }

            // Send notifications
            if ($targetType === 'client' && $fileCategory === 'complete') {
                $this->notifyClientOfWorkCompletion($order, $orderFile);
                $this->notifyClientOfAdminWorkCompletion($order, $orderFile);
            }

            Log::info('Admin file uploaded successfully', [
                'order_id' => $orderId,
                'file_id' => $orderFile->id,
                'file_name' => $orderFile->file_name,
                'target_type' => $targetType,
                'category' => $fileCategory,
                'admin_id' => Auth::id(),
                'order_status_updated' => ($targetType === 'client' && $fileCategory === 'complete')
            ]);

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully',
                'file' => $orderFile,
                'order_status_updated' => ($targetType === 'client' && $fileCategory === 'complete')
            ]);

        } catch (\Exception $e) {
            Log::error('Admin file upload failed', [
                'error' => $e->getMessage(),
                'order_id' => $request->input('order_id'),
                'admin_id' => Auth::id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get files for a specific order with access control
     */
    public function getOrderFiles($orderId)
    {
        try {
            $order = Order::findOrFail($orderId);
            $user = Auth::user();
            
            // Determine user type
            $userType = $this->getUserType($user);
            
            // Get files based on user access
            $files = OrderFile::where('order_id', $orderId)->get();
            
            // Add access control information to each file
            $filesWithAccess = $files->map(function ($file) use ($userType) {
                $file->canBeAccessedBy = $this->getFileAccess($file, $userType);
                return $file;
            });

            return response()->json([
                'success' => true,
                'files' => $filesWithAccess
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to get order files', [
                'error' => $e->getMessage(),
                'order_id' => $orderId
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch files'
            ], 500);
        }
    }

    /**
     * Determine user type for access control
     */
    private function getUserType($user)
    {
        if (in_array($user->user_type, ['admin', 'super_admin'])) {
            return 'admin';
        } elseif ($user->user_type === 'writer') {
            return 'writer';
        } else {
            return 'client';
        }
    }

    /**
     * Get file access permissions
     */
    private function getFileAccess($file, $userType)
    {
        switch ($file->access_control) {
            case OrderFile::ACCESS_ADMIN_ONLY:
                return ['admin'];
            case OrderFile::ACCESS_CLIENT_WRITER:
                return ['client', 'writer'];
            case OrderFile::ACCESS_CLIENT_ADMIN:
                return ['client', 'admin'];
            case OrderFile::ACCESS_WRITER_ADMIN:
                return ['writer', 'admin'];
            default:
                return ['admin'];
        }
    }

    /**
     * Notify admin of file upload
     */
    private function notifyAdminOfFileUpload($order, $orderFile)
    {
        try {
            // Get admin users (including super admins)
            $adminUsers = \App\Models\User::whereIn('user_type', ['admin', 'super_admin'])->get();
            
            if ($adminUsers->count() > 0) {
                Notification::send($adminUsers, new AdminOrderFileUploadedNotification($order, $orderFile));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send admin notification for file upload', [
                'error' => $e->getMessage(),
                'order_id' => $order->id,
                'file_id' => $orderFile->id
            ]);
        }
    }

    /**
     * Handle order status update for complete work submission.
     */
    private function handleCompleteWorkSubmission(Order $order)
    {
        if ($order->order_status !== Order::ORDER_STATUS_IN_PROGRESS) {
            Log::warning('Order status is not in_progress, cannot update to complete.', [
                'order_id' => $order->id,
                'current_status' => $order->order_status
            ]);
            return;
        }

        // Update order status and timeline fields
        $order->order_status = Order::ORDER_STATUS_UNDER_REVIEW;
        $order->work_submitted_at = now();
        $order->status_changed_at = now();
        $order->status_changed_by = auth()->id();
        $order->status_change_reason = 'Writer submitted complete work';
        
        $order->save();
        
        Log::info('Order status updated to under_review for complete work submission.', [
            'order_id' => $order->id,
            'new_status' => $order->order_status,
            'work_submitted_at' => $order->work_submitted_at
        ]);
    }

    /**
     * Handle order status update for admin complete work submission to client.
     */
    private function handleAdminCompleteWorkSubmission(Order $order, OrderFile $orderFile)
    {
        if ($order->order_status !== Order::ORDER_STATUS_IN_PROGRESS) {
            Log::warning('Order status is not in_progress, cannot update to complete.', [
                'order_id' => $order->id,
                'current_status' => $order->order_status
            ]);
            return;
        }

        // Update order status and timeline fields
        $order->order_status = Order::ORDER_STATUS_UNDER_REVIEW;
        $order->work_submitted_at = now();
        $order->status_changed_at = now();
        $order->status_changed_by = auth()->id();
        $order->status_change_reason = 'Admin submitted complete work to client';
        
        $order->save();
        
        Log::info('Order status updated to under_review for admin complete work submission.', [
            'order_id' => $order->id,
            'new_status' => $order->order_status,
            'work_submitted_at' => $order->work_submitted_at
        ]);
    }

    /**
     * Notify client of work completion.
     */
    private function notifyClientOfWorkCompletion(Order $order, OrderFile $orderFile)
    {
        try {
            $client = $order->user; // Assuming user relationship exists
            if ($client) {
                $client->notify(new ClientWorkCompletedNotification($order, $orderFile));
                Log::info('Client notified of work completion.', [
                    'order_id' => $order->id,
                    'client_id' => $client->id
                ]);
            } else {
                Log::warning('Client not found for work completion notification.', [
                    'order_id' => $order->id
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to send client notification for work completion', [
                'error' => $e->getMessage(),
                'order_id' => $order->id,
                'file_id' => $orderFile->id
            ]);
        }
    }

    /**
     * Notify client of admin work completion.
     */
    private function notifyClientOfAdminWorkCompletion(Order $order, OrderFile $orderFile)
    {
        try {
            $client = $order->user; // Assuming user relationship exists
            if ($client) {
                $client->notify(new AdminWorkCompletedNotification($order, $orderFile));
                Log::info('Client notified of admin work completion.', [
                    'order_id' => $order->id,
                    'client_id' => $client->id
                ]);
            } else {
                Log::warning('Client not found for admin work completion notification.', [
                    'order_id' => $order->id
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to send client notification for admin work completion', [
                'error' => $e->getMessage(),
                'order_id' => $order->id,
                'file_id' => $orderFile->id
            ]);
        }
    }


    // public function storeChunk(Request $request)
    // {

    //     Log::info('Received chunk upload request', [
    //         'chunk' => $request->input('chunk'),
    //         'totalChunks' => $request->input('totalChunks'),
    //         'filename' => $request->input('filename'),
    //         'fileType' => $request->input('fileType'),
    //         'chunkSize' => $request->file('file')->getSize(),
    //     ]);
    //     try {
    //         // Define allowed MIME types with friendly names for error messages
    //         $allowedMimes = [
    //             'application/msword' => 'DOC',
    //             'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'DOCX',
    //             'application/vnd.ms-excel' => 'XLS',
    //             'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'XLSX',
    //             'application/vnd.ms-powerpoint' => 'PPT',
    //             'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'PPTX',
    //             'application/pdf' => 'PDF',
    //             'application/zip' => 'ZIP',
    //             'application/x-zip-compressed' => 'ZIP'
    //         ];

    //         // Check authentication first
    //         if (!Auth::check()) {
    //             return response()->json(['error' => 'Unauthorized access'], 401);
    //         }

    //         // Validate the request
    //         $validator = Validator::make($request->all(), [
    //             'file' => [
    //                 'required',
    //                 'file',
    //                 'max:25600', // 25MB in kilobytes
    //                 function ($attribute, $value, $fail) use ($allowedMimes) {
    //                     if (!$value) {
    //                         $fail('No file was uploaded.');
    //                         return;
    //                     }

    //                     $mimeType = $value->getMimeType();
    //                     if (!array_key_exists($mimeType, $allowedMimes)) {
    //                         $allowedTypes = implode(', ', array_values(array_unique($allowedMimes)));
    //                         $fail("The file must be one of the following types: {$allowedTypes}");
    //                     }
    //                 },
    //             ],
    //             'chunk' => 'required|integer|min:0',
    //             'totalChunks' => 'required|integer|min:1',
    //             'filename' => [
    //                 'required',
    //                 'string',
    //                 'regex:/^[\w\-. ]+$/', // Only allow alphanumeric, dash, underscore, dot and space
    //                 'max:255' // Prevent extremely long filenames
    //             ],
    //             'order_id' => [
    //                 'required',
    //                 'exists:orders,id',
    //                 function ($attribute, $value, $fail) {
    //                     // Additional check to ensure user has access to this order
    //                     $order = \App\Models\Order::find($value);
    //                     if (!$order || (Auth::id() !== $order->user_id && Auth::user()->user_type !== 'admin')) {
    //                         $fail('You do not have permission to upload files to this order.');
    //                     }
    //                 }
    //             ],
    //             'fileType' => ['required', 'string', Rule::in(array_keys($allowedMimes))]
    //         ]);

    //         if ($validator->fails()) {
    //             return response()->json([
    //                 'message' => 'Validation failed',
    //                 'errors' => $validator->errors()
    //             ], 422);
    //         }

    //         $chunk = $request->file('file');
    //         $chunkNumber = $request->input('chunk');
    //         $totalChunks = $request->input('totalChunks');
    //         $filename = $request->input('filename');

    //         // Generate a unique filename to prevent overwrites
    //         $sanitizedFilename = time() . '_' . preg_replace('/[^a-zA-Z0-9\-._]/', '', $filename);

    //         // Ensure chunks directory exists and is writable
    //         $chunksPath = Storage::disk('local')->path('chunks');
    //         if (!file_exists($chunksPath)) {
    //             if (!mkdir($chunksPath, 0755, true)) {
    //                 throw new \Exception('Failed to create chunks directory');
    //             }
    //         }

    //         // Store chunk with proper error handling
    //         $chunkPath = "chunks/{$sanitizedFilename}.part{$chunkNumber}";
    //         try {
    //             if (!Storage::disk('local')->put($chunkPath, file_get_contents($chunk))) {
    //                 throw new \Exception('Failed to store chunk');
    //             }
    //         } catch (\Exception $e) {
    //             Log::error("Failed to store chunk: " . $e->getMessage());
    //             return response()->json(['error' => 'Failed to store file chunk'], 500);
    //         }

    //         // Process final merge if this is the last chunk
    //         if ($chunkNumber == $totalChunks - 1) {
    //             try {
    //                 $finalPath = "orders/{$sanitizedFilename}";
    //                 Storage::disk('public')->makeDirectory('orders');

    //                 $finalFile = fopen(Storage::disk('public')->path($finalPath), 'wb');
    //                 if ($finalFile === false) {
    //                     throw new \Exception('Could not create final file');
    //                 }

    //                 // Merge chunks with validation
    //                 $totalSize = 0;
    //                 for ($i = 0; $i < $totalChunks; $i++) {
    //                     $chunkContent = Storage::disk('local')->get("chunks/{$sanitizedFilename}.part{$i}");
    //                     if ($chunkContent === false) {
    //                         throw new \Exception("Failed to read chunk {$i}");
    //                     }
    //                     $totalSize += strlen($chunkContent);

    //                     // Check total size before writing
    //                     if ($totalSize > (25 * 1024 * 1024)) {
    //                         fclose($finalFile);
    //                         Storage::disk('public')->delete($finalPath);
    //                         throw new \Exception('File size exceeds 25MB limit');
    //                     }

    //                     fwrite($finalFile, $chunkContent);
    //                     Storage::disk('local')->delete("chunks/{$sanitizedFilename}.part{$i}");
    //                 }

    //                 fclose($finalFile);

    //                 // Verify the final file
    //                 if (!Storage::disk('public')->exists($finalPath)) {
    //                     throw new \Exception('Final file not created successfully');
    //                 }

    //                 // Create database record
    //                 $orderFile = OrderFile::create([
    //                     'order_id' => $request->order_id,
    //                     'file_name' => $filename, // Original filename for display
    //                     'file_type' => $request->input('fileType'),
    //                     'uploader_id' => Auth::id(),
    //                     'uploader_name' => Auth::user()->name,
    //                     'uploader_user_type' => Auth::user()->user_type,
    //                     'file_url' => $finalPath,
    //                     'file_size' => $totalSize,
    //                     'file_status' => 'uploaded',
    //                     'date_uploaded' => now()
    //                 ]);

    //                 return response()->json([
    //                     'message' => 'File uploaded successfully',
    //                     'file' => $orderFile
    //                 ]);
    //             } catch (\Exception $e) {
    //                 Log::error("Failed to merge chunks: " . $e->getMessage());

    //                 // Clean up any remaining chunks
    //                 for ($i = 0; $i < $totalChunks; $i++) {
    //                     Storage::disk('local')->delete("chunks/{$sanitizedFilename}.part{$i}");
    //                 }

    //                 return response()->json([
    //                     'error' => 'Failed to process file upload',
    //                     'message' => $e->getMessage()
    //                 ], 500);
    //             }
    //         }

    //         return response()->json([
    //             'message' => 'Chunk uploaded successfully',
    //             'chunk' => $chunkNumber,
    //             'totalChunks' => $totalChunks
    //         ]);
    //     } catch (\Exception $e) {
    //         Log::error('File chunk upload error: ' . $e->getMessage());
    //         return response()->json([
    //             'message' => 'Failed to process file chunk',
    //             'error' => $e->getMessage()
    //         ], 500);
    //     }
    // }

    public function storeChunk(Request $request)
    {
        try {
            // Define allowed MIME types with friendly names for error messages
            $allowedMimes = [
                'application/msword' => 'DOC',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'DOCX',
                'application/vnd.ms-excel' => 'XLS',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'XLSX',
                'application/vnd.ms-powerpoint' => 'PPT',
                'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'PPTX',
                'application/pdf' => 'PDF',
                'application/zip' => 'ZIP',
                'application/x-zip-compressed' => 'ZIP'
            ];

            // Add detailed logging for debugging
            Log::info('Chunk upload request', [
                'chunk' => $request->input('chunk'),
                'totalChunks' => $request->input('totalChunks'),
                'filename' => $request->input('filename'),
                'fileType' => $request->input('fileType'),
                'fileSize' => $request->file('file') ? $request->file('file')->getSize() : 'No file',
                'order_id' => $request->input('order_id'),
                'user_id' => Auth::id(),
                'user_type' => Auth::user()->user_type
            ]);

            // Check authentication first
            if (!Auth::check()) {
                return response()->json(['error' => 'Unauthorized access'], 401);
            }

            // Get chunk information
            $chunkNumber = (int)$request->input('chunk');
            $totalChunks = (int)$request->input('totalChunks');
            $fileType = $request->input('fileType');

            // Validate the request - different validation for first chunk vs subsequent chunks
            $validationRules = [
                'file' => ['required', 'file', 'max:25600'], // 25MB max for each chunk
                'chunk' => 'required|integer|min:0',
                'totalChunks' => 'required|integer|min:1',
                'filename' => [
                    'required',
                    'string',
                    'regex:/^[\w\-. ]+$/', // Only allow alphanumeric, dash, underscore, dot and space
                    'max:255' // Prevent extremely long filenames
                ],
                'order_id' => [
                    'required',
                    'exists:orders,id',
                    function ($attribute, $value, $fail) {
                        // Additional check to ensure user has access to this order
                        $order = \App\Models\Order::find($value);
                        if (!$order) {
                            Log::error('Order not found for file upload', [
                                'order_id' => $value,
                                'user_id' => Auth::id()
                            ]);
                            $fail('Order not found.');
                            return;
                        }

                        $user = Auth::user();
                        $userId = $user->id;
                        $userType = $user->user_type;

                        // Log the permission check details for debugging
                        Log::info('Permission check for file upload', [
                            'user_id' => $userId,
                            'user_type' => $userType,
                            'order_id' => $value,
                            'order_user_id' => $order->user_id,
                            'order_writer_id' => $order->writer_id,
                            'order_status' => $order->order_status,
                            'order_payment_status' => $order->payment_status ?? 'unknown'
                        ]);

                        // Check if user has permission based on their role and relationship to the order
                        $hasPermission = false;

                        // Clients can upload files to their own orders
                        if ($userType === 'client' && $userId === $order->user_id) {
                            $hasPermission = true;
                            Log::info('Permission granted: Client owns the order');
                        }
                        // Writers can upload files to orders assigned to them
                        elseif ($userType === 'writer' && $userId === $order->writer_id) {
                            $hasPermission = true;
                            Log::info('Permission granted: Writer is assigned to the order');
                        }
                        // Admins and super_admins can upload files to any order
                        elseif (in_array($userType, ['admin', 'super_admin'])) {
                            $hasPermission = true;
                            Log::info('Permission granted: User is admin/super_admin');
                        }

                        if (!$hasPermission) {
                            Log::warning('Permission denied for file upload', [
                                'user_id' => $userId,
                                'user_type' => $userType,
                                'order_id' => $value,
                                'order_user_id' => $order->user_id,
                                'order_writer_id' => $order->writer_id
                            ]);
                            $fail('You do not have permission to upload files to this order.');
                        }

                        // Additional check: Ensure the order is in a valid state for file uploads
                        $validStatuses = [
                            'pending', 'bidding', 'assigned', 'in_progress', 
                            'submitted', 'under_review', 'revision_requested'
                        ];
                        
                        if (!in_array($order->order_status, $validStatuses)) {
                            Log::warning('Order status not valid for file upload', [
                                'order_id' => $value,
                                'order_status' => $order->order_status,
                                'valid_statuses' => $validStatuses
                            ]);
                            $fail('File uploads are not allowed for orders in the current status.');
                        }

                        // Log successful permission check
                        Log::info('Permission check passed for file upload', [
                            'user_id' => $userId,
                            'user_type' => $userType,
                            'order_id' => $value,
                            'order_status' => $order->order_status
                        ]);
                    }
                ],
                'fileType' => ['required', 'string']
            ];

            // Only validate file type for the first chunk
            if ($chunkNumber === 0) {
                $validationRules['fileType'][] = Rule::in(array_keys($allowedMimes));
            }

            $validator = Validator::make($request->all(), $validationRules);

            if ($validator->fails()) {
                Log::warning('Chunk validation failed', [
                    'errors' => $validator->errors()->toArray(),
                    'chunk' => $chunkNumber,
                    'totalChunks' => $totalChunks
                ]);

                return response()->json([
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $chunk = $request->file('file');
            $filename = $request->input('filename');

            // Generate a unique filename to prevent overwrites
            $sanitizedFilename = time() . '_' . preg_replace('/[^a-zA-Z0-9\-._]/', '', $filename);

            // Ensure chunks directory exists and is writable
            $chunksPath = Storage::disk('local')->path('chunks');
            if (!file_exists($chunksPath)) {
                if (!mkdir($chunksPath, 0755, true)) {
                    throw new \Exception('Failed to create chunks directory');
                }
            }

            // Store chunk with proper error handling
            $chunkPath = "chunks/{$sanitizedFilename}.part{$chunkNumber}";
            try {
                if (!Storage::disk('local')->put($chunkPath, file_get_contents($chunk))) {
                    throw new \Exception('Failed to store chunk');
                }
            } catch (\Exception $e) {
                Log::error("Failed to store chunk: " . $e->getMessage());
                return response()->json(['error' => 'Failed to store file chunk'], 500);
            }

            // Process final merge if this is the last chunk
            if ($chunkNumber == $totalChunks - 1) {
                try {
                    $finalPath = "orders/{$sanitizedFilename}";
                    Storage::disk('public')->makeDirectory('orders');

                    $finalFile = fopen(Storage::disk('public')->path($finalPath), 'wb');
                    if ($finalFile === false) {
                        throw new \Exception('Could not create final file');
                    }

                    // Merge chunks with validation
                    $totalSize = 0;
                    for ($i = 0; $i < $totalChunks; $i++) {
                        $chunkContent = Storage::disk('local')->get("chunks/{$sanitizedFilename}.part{$i}");
                        if ($chunkContent === false) {
                            throw new \Exception("Failed to read chunk {$i}");
                        }
                        $totalSize += strlen($chunkContent);

                        // Check total size before writing
                        if ($totalSize > (40 * 1024 * 1024)) { // Increased to 40MB
                            fclose($finalFile);
                            Storage::disk('public')->delete($finalPath);
                            throw new \Exception('File size exceeds 40MB limit');
                        }

                        fwrite($finalFile, $chunkContent);
                        Storage::disk('local')->delete("chunks/{$sanitizedFilename}.part{$i}");
                    }

                    fclose($finalFile);

                    // Verify the final file
                    if (!Storage::disk('public')->exists($finalPath)) {
                        throw new \Exception('Final file not created successfully');
                    }

                    // Create database record
                    $orderFile = OrderFile::create([
                        'order_id' => $request->order_id,
                        'file_name' => $filename, // Original filename for display
                        'file_type' => $fileType, // Use the original file type from the request
                        'uploader_id' => Auth::id(),
                        'uploader_name' => Auth::user()->name,
                        'uploader_user_type' => Auth::user()->user_type,
                        'file_url' => $finalPath,
                        'file_size' => $totalSize,
                        'file_status' => 'uploaded',
                        'date_uploaded' => now()
                    ]);

                    Log::info('File upload complete', [
                        'filename' => $filename,
                        'size' => $totalSize,
                        'chunks' => $totalChunks
                    ]);

                    return response()->json([
                        'message' => 'File uploaded successfully',
                        'file' => $orderFile
                    ]);
                } catch (\Exception $e) {
                    Log::error("Failed to merge chunks: " . $e->getMessage());

                    // Clean up any remaining chunks
                    for ($i = 0; $i < $totalChunks; $i++) {
                        Storage::disk('local')->delete("chunks/{$sanitizedFilename}.part{$i}");
                    }

                    return response()->json([
                        'error' => 'Failed to process file upload',
                        'message' => $e->getMessage()
                    ], 500);
                }
            }

            return response()->json([
                'message' => 'Chunk uploaded successfully',
                'chunk' => $chunkNumber,
                'totalChunks' => $totalChunks
            ]);
        } catch (\Exception $e) {
            Log::error('File chunk upload error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => Auth::id(),
                'user_type' => Auth::user()->user_type ?? 'unknown',
                'order_id' => $request->input('order_id')
            ]);

            return response()->json([
                'message' => 'Failed to process file chunk',
                'error' => $e->getMessage()
            ], 500);
        }
    }




    public function download(OrderFile $orderFile)
    {
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        try {
            // Clean the file path to prevent directory traversal
            $filePath = storage_path('app/public/' . str_replace('..', '', $orderFile->file_url));

            if (!file_exists($filePath)) {
                Log::error('File not found: ' . $filePath);
                return response()->json(['error' => 'File not found'], 404);
            }

            $fileName = basename($orderFile->file_name);
            $mimeType = mime_content_type($filePath) ?: 'application/octet-stream';

            return response()->file($filePath, [
                'Content-Type' => $mimeType,
                'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
                'Cache-Control' => 'private, no-transform, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0'
            ]);
        } catch (\Exception $e) {
            Log::error('File download error: ' . $e->getMessage());
            return response()->json(['error' => 'Download failed'], 500);
        }
    }

    public function destroy(OrderFile $orderFile)
    {
        // Check if user owns the file or is admin/super_admin
        if (Auth::id() === $orderFile->uploader_id || in_array(Auth::user()->user_type, ['admin', 'super_admin'])) {
            try {
                // Delete the physical file from storage
                Storage::disk('public')->delete($orderFile->file_url);

                // Delete the database record
                $orderFile->delete();

                return response()->json([
                    'success' => true,
                    'message' => 'File deleted successfully'
                ]);
            } catch (\Exception $e) {
                Log::error('File deletion error: ' . $e->getMessage(), [
                    'file_id' => $orderFile->id,
                    'user_id' => Auth::id(),
                    'error' => $e
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete file: ' . $e->getMessage()
                ], 500);
            }
        }

        return response()->json([
            'success' => false,
            'message' => 'Unauthorized action.'
        ], 403);
    }
}
