<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderRating;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Inertia\Response;

class OrderRatingController extends Controller
{
    /**
     * Show the rating form for a completed order.
     */
    public function showRatingForm(Order $order): Response
    {
        // Ensure the authenticated user is the client who placed the order
        if (Auth::id() !== $order->user_id) {
            abort(403, 'You can only rate orders you placed.');
        }

        // Check if order is eligible for rating
        if (!$order->isEligibleForRating()) {
            if ($order->hasRating()) {
                abort(400, 'This order has already been rated.');
            }
            if ($order->isRatingExpired()) {
                abort(400, 'The rating period for this order has expired.');
            }
            abort(400, 'This order is not eligible for rating.');
        }

        return Inertia::render('Order/RatingForm', [
            'order' => [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'type_of_paper' => $order->type_of_paper,
                'subject' => $order->subject,
                'academic_level' => $order->academic_level,
                'pages' => $order->pages,
                'writer' => [
                    'id' => $order->writer->id,
                    'name' => $order->writer->name,
                ],
                'rating_deadline' => $order->getRatingDeadline()?->format('Y-m-d H:i:s'),
                'days_remaining' => $order->getRatingDeadline()?->diffInDays(now()),
            ],
            'ratingCategories' => OrderRating::RATING_CATEGORIES,
            'ratingTags' => OrderRating::RATING_TAGS,
        ]);
    }

    /**
     * Submit a rating for an order.
     */
    public function submitRating(Request $request, Order $order)
    {
        // Ensure the authenticated user is the client who placed the order
        if (Auth::id() !== $order->user_id) {
            return response()->json(['error' => 'You can only rate orders you placed.'], 403);
        }

        // Check if order is eligible for rating
        if (!$order->isEligibleForRating()) {
            if ($order->hasRating()) {
                return response()->json(['error' => 'This order has already been rated.'], 400);
            }
            if ($order->isRatingExpired()) {
                return response()->json(['error' => 'The rating period for this order has expired.'], 400);
            }
            return response()->json(['error' => 'This order is not eligible for rating.'], 400);
        }

        // Validate the request
        $validator = Validator::make($request->all(), [
            'overall_rating' => 'required|integer|min:1|max:5',
            'quality_rating' => 'nullable|integer|min:1|max:5',
            'communication_rating' => 'nullable|integer|min:1|max:5',
            'timeliness_rating' => 'nullable|integer|min:1|max:5',
            'value_rating' => 'nullable|integer|min:1|max:5',
            'review_title' => 'nullable|string|max:255',
            'review_text' => 'nullable|string|max:2000',
            'is_anonymous' => 'boolean',
            'rating_categories' => 'nullable|array',
            'rating_tags' => 'nullable|array',
            'rating_tags.*' => 'string|in:' . implode(',', OrderRating::RATING_TAGS),
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            // Create the rating
            $rating = OrderRating::create([
                'order_id' => $order->id,
                'client_id' => Auth::id(),
                'writer_id' => $order->writer_id,
                'overall_rating' => $request->overall_rating,
                'quality_rating' => $request->quality_rating,
                'communication_rating' => $request->communication_rating,
                'timeliness_rating' => $request->timeliness_rating,
                'value_rating' => $request->value_rating,
                'review_title' => $request->review_title,
                'review_text' => $request->review_text,
                'is_anonymous' => $request->boolean('is_anonymous', false),
                'is_verified' => true, // All ratings from completed orders are verified
                'rating_categories' => $request->rating_categories,
                'rating_tags' => $request->rating_tags,
                'metadata' => [
                    'submitted_at' => now()->toISOString(),
                    'user_agent' => $request->userAgent(),
                    'ip_address' => $request->ip(),
                ],
                'rated_at' => now(),
            ]);

            return response()->json([
                'message' => 'Rating submitted successfully!',
                'rating' => $rating,
            ]);

        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to submit rating. Please try again.'], 500);
        }
    }

    /**
     * View a rating for an order.
     */
    public function viewRating(Order $order)
    {
        if (!$order->hasRating()) {
            abort(404, 'No rating found for this order.');
        }

        $rating = $order->rating;
        
        // Ensure the authenticated user can view this rating
        if (Auth::id() !== $order->user_id && Auth::id() !== $order->writer_id && !Auth::user()->isAdmin()) {
            abort(403, 'You do not have permission to view this rating.');
        }

        return Inertia::render('Order/ViewRating', [
            'order' => [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'type_of_paper' => $order->type_of_paper,
                'subject' => $order->subject,
                'academic_level' => $order->academic_level,
                'pages' => $order->pages,
            ],
            'rating' => [
                'id' => $rating->id,
                'overall_rating' => $rating->overall_rating,
                'quality_rating' => $rating->quality_rating,
                'communication_rating' => $rating->communication_rating,
                'timeliness_rating' => $rating->timeliness_rating,
                'value_rating' => $rating->value_rating,
                'review_title' => $rating->review_title,
                'review_text' => $rating->review_text,
                'is_anonymous' => $rating->is_anonymous,
                'rating_categories' => $rating->rating_categories,
                'rating_tags' => $rating->rating_tags,
                'rated_at' => $rating->rated_at->format('Y-m-d H:i:s'),
                'client' => [
                    'name' => $rating->getClientDisplayName(),
                ],
            ],
        ]);
    }

    /**
     * Get ratings for a specific writer.
     */
    public function getWriterRatings(User $writer, Request $request)
    {
        $query = $writer->ratingsReceived()->verified();

        // Apply filters
        if ($request->has('rating')) {
            $query->ratingRange($request->rating);
        }

        if ($request->has('recent')) {
            $query->recent($request->recent);
        }

        $ratings = $query->with(['order:id,order_number,title,type_of_paper,subject'])
            ->orderBy('rated_at', 'desc')
            ->paginate(10);

        return response()->json([
            'ratings' => $ratings->items(),
            'pagination' => [
                'current_page' => $ratings->currentPage(),
                'last_page' => $ratings->lastPage(),
                'per_page' => $ratings->perPage(),
                'total' => $ratings->total(),
            ],
            'summary' => [
                'average_rating' => $writer->getAverageRating(),
                'total_ratings' => $writer->getTotalRatingsCount(),
                'rating_distribution' => $writer->getRatingDistribution(),
                'positive_percentage' => $writer->getPositiveRatingPercentage(),
            ],
        ]);
    }

    /**
     * Get ratings given by a specific client.
     */
    public function getClientRatings(Request $request)
    {
        $ratings = Auth::user()->ratingsGiven()
            ->with(['order:id,order_number,title,type_of_paper,subject', 'writer:id,name'])
            ->orderBy('rated_at', 'desc')
            ->paginate(10);

        return response()->json([
            'ratings' => $ratings->items(),
            'pagination' => [
                'current_page' => $ratings->currentPage(),
                'last_page' => $ratings->lastPage(),
                'per_page' => $ratings->perPage(),
                'total' => $ratings->total(),
            ],
        ]);
    }

    /**
     * Get orders eligible for rating for the authenticated client.
     */
    public function getEligibleOrders(Request $request)
    {
        $orders = Order::where('user_id', Auth::id())
            ->where('order_status', Order::ORDER_STATUS_APPROVED)
            ->whereDoesntHave('rating')
            ->where('client_approved_at', '>=', now()->subDays(30))
            ->with(['writer:id,name'])
            ->orderBy('client_approved_at', 'desc')
            ->paginate(10);

        return response()->json([
            'orders' => $orders->items(),
            'pagination' => [
                'current_page' => $orders->currentPage(),
                'last_page' => $orders->lastPage(),
                'per_page' => $orders->perPage(),
                'total' => $orders->total(),
            ],
        ]);
    }
}
