<?php

namespace App\Http\Controllers;

use Inertia\Inertia;
use App\Models\Order;
use App\Models\Payment;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Services\AdminNotificationService;
use App\Notifications\AdminPaymentNotification;
use App\Notifications\PaymentFailedNotification;
use App\Services\Payments\PaymentServiceFactory;
use App\Notifications\PaymentSuccessfulNotification;


class PaymentController extends Controller
{
    /**
     * Display payment options for an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $orderId
     * @return \Inertia\Response
     */
    public function showPaymentOptions(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        $this->authorize('pay', $order);

        // Get active payment methods
        $paymentMethods = PaymentMethod::getActive();

        return Inertia::render('Payment/Options', [
            'order' => $order,
            'paymentMethods' => $paymentMethods->map(function ($method) {
                return [
                    'id' => $method->id,
                    'name' => $method->name,
                    'display_name' => $method->display_name,
                ];
            }),
        ]);
    }

    /**
     * Initialize a payment for an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $orderId
     * @return \Illuminate\Http\JsonResponse
     */
    public function initializePayment(Request $request, $orderId)
    {


        $request->validate([
            'payment_method' => 'required|string',
        ]);

        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        $this->authorize('pay', $order);

        // Check if the order is already paid
        if ($order->isPaid()) {
            return response()->json([
                'success' => false,
                'message' => 'This order has already been paid for.',
            ], 400);
        }

        $paymentMethodName = $request->input('payment_method');

        try {
            // Get the payment service for the selected method
            $paymentService = PaymentServiceFactory::create($paymentMethodName);

            // Initialize the payment
            $result = $paymentService->initializePayment($order);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['error'] ?? 'Failed to initialize payment.',
                ], 500);
            }

            return response()->json($result);
        } catch (\Exception $e) {


            Log::error('Payment initialization error', [
                'order_id' => $orderId,
                'payment_method' => $paymentMethodName,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while initializing the payment: ' . $e->getMessage(),
            ], 500);
        }
    }

    // /**
    //  * Process a payment after confirmation.
    //  *
    //  * @param  \Illuminate\Http\Request  $request
    //  * @param  int  $orderId
    //  * @return \Illuminate\Http\JsonResponse
    //  */
    // public function processPayment(Request $request, $orderId)
    // {
    //     $request->validate([
    //         'payment_method' => 'required|string',
    //         'payment_data' => 'required|array',
    //     ]);

    //     $order = Order::findOrFail($orderId);

    //     // Check if the order belongs to the authenticated user
    //     $this->authorize('pay', $order);

    //     // Check if the order is already paid
    //     if ($order->isPaid()) {
    //         return response()->json([
    //             'success' => true,
    //             'message' => 'This order has already been paid for.',
    //             'redirect' => route('orders.show', $order->id),
    //         ]);
    //     }

    //     $paymentMethodName = $request->input('payment_method');
    //     $paymentData = $request->input('payment_data');

    //     try {
    //         // Get the payment service for the selected method
    //         $paymentService = PaymentServiceFactory::create($paymentMethodName);

    //         // Process the payment
    //         $payment = $paymentService->processPayment($order, $paymentData);

    //         if ($payment->isCompleted()) {


    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Payment completed successfully.',
    //                 'redirect' => route('orders.show', $order->id),
    //             ]);
    //         } else {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Payment processing failed: ' . ($payment->metadata['error'] ?? 'Unknown error'),
    //             ], 400);
    //         }
    //     } catch (\Exception $e) {
    //         Log::error('Payment processing error', [
    //             'order_id' => $orderId,
    //             'payment_method' => $paymentMethodName,
    //             'error' => $e->getMessage(),
    //         ]);

    //         return response()->json([
    //             'success' => false,
    //             'message' => 'An error occurred while processing the payment: ' . $e->getMessage(),
    //         ], 500);
    //     }
    // }


    /**
     * Process a payment after confirmation.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $orderId
     * @return \Illuminate\Http\JsonResponse
     */
    public function processPayment(Request $request, $orderId)
    {
        $request->validate([
            'payment_method' => 'required|string',
            'payment_data' => 'required|array',
        ]);

        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        $this->authorize('pay', $order);

        // Check if the order is already paid
        if ($order->isPaid()) {
            return response()->json([
                'success' => true,
                'message' => 'This order has already been paid for.',
                'redirect' => route('orders.show', $order->id),
            ]);
        }

        $paymentMethodName = $request->input('payment_method');
        $paymentData = $request->input('payment_data');

        try {
            // Get the payment service for the selected method
            $paymentService = PaymentServiceFactory::create($paymentMethodName);

            // Process the payment
            $payment = $paymentService->processPayment($order, $paymentData);

            if ($payment->isCompleted()) {
                // Send success notification to user
                $user = Auth::user();
                if ($user) {
                    Log::info('Sending payment success notification from controller', [
                        'user_id' => $user->id,
                        'order_id' => $order->id,
                        'payment_id' => $payment->id
                    ]);

                    // Send notification to user
                    $user->notify(new PaymentSuccessfulNotification($order, $payment));
                    $payment->notified_user = true;
                    $payment->save();

                    // Also notify admins
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification($order, $payment, $user->name)
                    );
                    $payment->notified_admin = true;
                    $payment->save();
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Payment completed successfully.',
                    'redirect' => route('orders.show', $order->id),
                ]);
            } else {
                // Send failure notification to user
                $user = Auth::user();
                if ($user) {
                    $errorMessage = $payment->metadata['error'] ?? 'Unknown error';
                    Log::info('Sending payment failure notification from controller', [
                        'user_id' => $user->id,
                        'order_id' => $order->id,
                        'error' => $errorMessage
                    ]);

                    // Send notification to user
                    $user->notify(new PaymentFailedNotification($order, $payment, $errorMessage));

                    // Also notify admins about failed payment
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification($order, $payment, $user->name, true, $errorMessage)
                    );
                }

                return response()->json([
                    'success' => false,
                    'message' => 'Payment processing failed: ' . ($payment->metadata['error'] ?? 'Unknown error'),
                ], 400);
            }
        } catch (\Exception $e) {
            // Log error and send failure notification for exceptions
            Log::error('Payment processing error', [
                'order_id' => $orderId,
                'payment_method' => $paymentMethodName,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Try to send a notification about the error
            try {
                $user = Auth::user();
                if ($user) {
                    $user->notify(new PaymentFailedNotification($order, null, $e->getMessage()));
                }
            } catch (\Exception $notifyEx) {
                Log::error('Failed to send payment error notification', [
                    'error' => $notifyEx->getMessage()
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing the payment: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Get payment configuration for client-side initialization.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $methodName
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPaymentConfig(Request $request, $methodName)
    {
        try {
            // Get the payment service for the selected method
            $paymentService = PaymentServiceFactory::create($methodName);

            // Get client configuration
            $config = $paymentService->getClientConfig();

            return response()->json([
                'success' => true,
                'config' => $config,
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting payment configuration', [
                'payment_method' => $methodName,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to get payment configuration: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Check the status of a payment.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $paymentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkPaymentStatus(Request $request, $paymentId)
    {
        $payment = Payment::findOrFail($paymentId);

        // Check if the payment's order belongs to the authenticated user
        // $this->authorize('view', $payment->order);

        return response()->json([
            'success' => true,
            'status' => $payment->status,
            'is_completed' => $payment->isCompleted(),
            'updated_at' => $payment->updated_at,
        ]);
    }

    /**
     * Show payment history for an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $orderId
     * @return \Inertia\Response
     */
    public function showPaymentHistory(Request $request, $orderId)
    {
        $order = Order::with('payments')->findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        // $this->authorize('view', $order);

        return Inertia::render('Payment/History', [
            'order' => $order,
            'payments' => $order->payments->map(function ($payment) {
                return [
                    'id' => $payment->id,
                    'method' => $payment->payment_method,
                    'amount' => $payment->amount,
                    'status' => $payment->status,
                    'transaction_id' => $payment->transaction_id,
                    'date' => $payment->payment_date,
                ];
            }),
        ]);
    }
}
