<?php

namespace App\Http\Controllers\Payments;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PayPalDirectController extends Controller
{
    /**
     * Handle successful payment return from PayPal.
     *
     * @param Request $request
     * @param Order $order
     * @return \Illuminate\Http\Response
     */
    public function return(Request $request, Order $order)
    {
        Log::info('PayPal Direct return URL accessed', [
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'query_params' => $request->query()
        ]);

        // Check if payment was actually completed
        if ($order->payment_status === 'paid') {
            return redirect()->route('orders.show', $order->id)
                ->with('success', 'Payment completed successfully! Your order is now being processed.');
        }

        // Payment might still be pending (IPN might not have arrived yet)
        return redirect()->route('orders.show', $order->id)
            ->with('info', 'Payment submitted successfully! We are processing your payment. You will receive a confirmation email shortly.');
    }

    /**
     * Handle cancelled payment return from PayPal.
     *
     * @param Request $request
     * @param Order $order
     * @return \Illuminate\Http\Response
     */
    public function cancel(Request $request, Order $order)
    {
        Log::info('PayPal Direct cancel URL accessed', [
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'query_params' => $request->query()
        ]);

        return redirect()->route('orders.show', $order->id)
            ->with('warning', 'Payment was cancelled. You can try again or choose a different payment method.');
    }
} 