<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Models\Order;

class ProjectsController extends Controller
{
    /**
     * Display completed projects for the authenticated client.
     */
    public function completed(Request $request)
    {
        $search = trim($request->input('search', ''));
        
        $query = Auth::user()->orders()
            ->whereIn('order_status', Order::COMPLETED_STATUSES)
            ->latest('dateposted');

        // Apply search filter if search term exists
        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'LIKE', "%{$search}%")
                    ->orWhere('title', 'LIKE', "%{$search}%")
                    ->orWhere('subject', 'LIKE', "%{$search}%")
                    ->orWhere('type_of_paper', 'LIKE', "%{$search}%");
            });
        }

        $completedProjects = $query->with(['writer', 'payments.refunds' => function ($query) {
                $query->where('status', 'completed');
            }])
            ->paginate(10)
            ->withQueryString()
            ->through(function ($order) {
                return array_merge($order->toArray(), [
                    'is_refunded' => $order->isRefunded(),
                    'is_partially_refunded' => $order->isPartiallyRefunded(),
                    'completion_date' => $order->status_changed_at,
                    'writer_name' => $order->writer ? $order->writer->name : null,
                    'total_paid' => $order->payments->where('status', 'completed')->sum('amount'),
                    'refund_amount' => $order->payments->flatMap->refunds->where('status', 'completed')->sum('amount')
                ]);
            });

        return Inertia::render('Client/Projects/Completed', [
            'completedProjects' => $completedProjects,
            'filters' => [
                'search' => $search
            ]
        ]);
    }

    /**
     * Display active projects for the authenticated client.
     */
    public function active(Request $request)
    {
        $search = trim($request->input('search', ''));
        
        $query = Auth::user()->orders()
            ->whereIn('order_status', Order::ACTIVE_STATUSES)
            ->latest('dateposted');

        // Apply search filter if search term exists
        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'LIKE', "%{$search}%")
                    ->orWhere('title', 'LIKE', "%{$search}%")
                    ->orWhere('subject', 'LIKE', "%{$search}%")
                    ->orWhere('type_of_paper', 'LIKE', "%{$search}%");
            });
        }

        $activeProjects = $query->with(['writer', 'bids'])
            ->paginate(10)
            ->withQueryString()
            ->through(function ($order) {
                return array_merge($order->toArray(), [
                    'bids_count' => $order->bids()->count(),
                    'is_assigned' => !is_null($order->writer_id),
                    'writer_name' => $order->writer ? $order->writer->name : null,
                    'days_remaining' => $order->deadline ? max(0, now()->diffInDays($order->deadline, false)) : null,
                    'is_overdue' => $order->deadline && now()->isAfter($order->deadline)
                ]);
            });

        return Inertia::render('Client/Projects/Active', [
            'activeProjects' => $activeProjects,
            'filters' => [
                'search' => $search
            ]
        ]);
    }
} 