<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Helpers\ImageHelper;
use App\Services\PageService;
use App\Services\SEOService;
use App\Models\Page;

class ServiceController extends Controller
{
    protected PageService $pageService;
    protected SEOService $seoService;

    public function __construct(PageService $pageService, SEOService $seoService)
    {
        $this->pageService = $pageService;
        $this->seoService = $seoService;
    }

    /**
     * Display the main services index page
     */
    public function index()
    {
        // Try to get services page from CMS
        $cmsPage = $this->pageService->getPageWithContent('services');
        
        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicesPage($cmsPage);
        } else {
            // Fallback to static data
            return $this->renderStaticServicesPage();
        }
    }

    /**
     * Render services index page using CMS data
     */
    private function renderCMSServicesPage(Page $page)
    {
        // Get SEO data (may be null) and convert to safe array
        $seoData = $page->seo ?? null;
        $seoArray = $seoData?->toArray() ?? [];

        // Generate dynamic SEO metadata (safe: pass array)
        $openGraphData = $this->seoService->generateOpenGraphData($page->toArray(), $seoArray);
        $twitterData = $this->seoService->generateTwitterCardData($page->toArray(), $seoArray);

        // Get media files
        $mediaFiles = $page->media;

        // Process content blocks
        $contentBlocks = $this->processContentBlocks($page->contentBlocks);
        
        // Get all services for the index page
        $services = $this->getAllServices();
        $services = $this->addOptimizedImagesToServices($services);
        
        return Inertia::render('Services/Index', [
            'page' => $page,
            'contentBlocks' => $contentBlocks,
            'mediaFiles' => $mediaFiles,
            'services' => $services,
            'seoData' => [
                'canonicalUrl' => $seoData?->canonical_url ?? url('/services'),
                'ogUrl' => $seoData?->og_url ?? url('/services'),
                'meta_title' => $seoData?->meta_title ?? $page->title,
                'meta_description' => $seoData?->meta_description ?? '',
            ],
            'openGraph' => $openGraphData,
            'twitterCard' => $twitterData,
            'isCMS' => true,
        ]);
    }

    /**
     * Fallback to static services page
     */
    private function renderStaticServicesPage()
    {
        $services = $this->getAllServices();
        $services = $this->addOptimizedImagesToServices($services);

        return Inertia::render('Services/Index', [
            'services' => $services,
            'seoData' => [
                'canonicalUrl' => url('/services'),
                'ogUrl' => url('/services'),
            ],
            'isCMS' => false,
        ]);
    }

    public function researchPapers()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('research-papers');
        
        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/ResearchPapers');
        } else {
            // Fallback to static data
            return $this->renderStaticServicePage('research-papers', 'Services/ResearchPapers');
        }
    }

    /**
     * Render individual service page using CMS data
     */
    private function renderCMSServicePage(Page $page, string $component)
    {
        // Get SEO data (may be null) and prepare safe array
        $seoData = $page->seo ?? null;
        $seoArray = $seoData?->toArray() ?? [];

        // Generate dynamic SEO metadata (safe: pass array)
        $openGraphData = $this->seoService->generateOpenGraphData($page->toArray(), $seoArray);
        $twitterData = $this->seoService->generateTwitterCardData($page->toArray(), $seoArray);

        // Get media files
        $mediaFiles = $page->media;

        // Process content blocks
        $contentBlocks = $this->processContentBlocks($page->contentBlocks);

        // Prefill structured data if missing
        $structuredData = $this->buildStructuredData($page, $seoArray, $contentBlocks);
        
        // Normalize template path - ensure it uses forward slashes and proper casing
        // CRITICAL: Always prefer the explicitly passed component parameter to override any bad DB values
        $template = $component ?? $page->template ?? 'Services/DynamicService';
        
        // Convert to string and trim - handle null/empty cases
        $template = trim((string) ($template ?? ''));
        if (empty($template)) {
            $template = 'Services/DynamicService';
        }
        
        // Aggressively normalize: convert ALL dots to slashes first
        $template = str_replace('.', '/', $template);
        
        // Fix common template path issues - handle various formats
        $template = preg_replace('/^services[.\/]/i', 'Services/', $template);
        $template = str_replace('services/', 'Services/', $template);
        $template = str_replace('Services/', 'Services/', $template); // Normalize
        
        // Fix casing - ensure Services/ prefix with capital S
        $template = preg_replace('/^services\//i', 'Services/', $template);
        
        // Ensure it starts with Services/ if it's a service page
        if (!str_starts_with($template, 'Services/') && !str_starts_with($template, 'Pages/')) {
            // If it's just a name like "Essays" or "essays", prepend Services/
            $baseName = basename($template);
            // Convert kebab-case or snake_case to PascalCase
            $baseName = ucwords(str_replace(['-', '_'], ' ', $baseName));
            $baseName = str_replace(' ', '', $baseName);
            $template = 'Services/' . $baseName;
        }
        
        // Final cleanup - ensure proper casing for each part
        $parts = explode('/', $template);
        if (count($parts) > 0) {
            $parts[0] = 'Services'; // Force proper casing
            if (count($parts) > 1) {
                // Handle component names - convert to PascalCase
                $componentName = $parts[1];
                // Remove any remaining dots or special chars
                $componentName = preg_replace('/[^a-zA-Z0-9]/', '', $componentName);
                // Convert to PascalCase if it's lowercase
                if (strtolower($componentName) === $componentName && !empty($componentName)) {
                    $componentName = ucfirst($componentName);
                }
                // Handle camelCase (e.g., DynamicService)
                if (!empty($componentName)) {
                    $componentName = ucfirst($componentName);
                }
                $parts[1] = $componentName ?: 'DynamicService';
            } else {
                // If no component name, default to DynamicService
                $parts[] = 'DynamicService';
            }
        } else {
            // Fallback if something went wrong
            $parts = ['Services', 'DynamicService'];
        }
        
        $componentPath = implode('/', $parts);
        
        // Final safety check - if path still looks wrong, force DynamicService
        $lowerPath = strtolower($componentPath);
        if (strpos($componentPath, '.') !== false || 
            !str_starts_with($componentPath, 'Services/') || 
            empty($componentPath) ||
            $lowerPath === 'services/essays' ||
            $lowerPath === 'services.essays' ||
            $lowerPath === 'services/essay-writing' ||
            $lowerPath === 'services.essay-writing' ||
            preg_match('/^services[.\/]essay/i', $lowerPath)) {
            $componentPath = 'Services/DynamicService';
        }
        
        // Log for debugging - include slug to identify which page
        \Log::debug('Component path resolved', [
            'page_slug' => $page->slug ?? 'unknown',
            'original_template' => $page->template ?? 'null',
            'component_param' => $component ?? 'null',
            'normalized_template' => $template,
            'final_path' => $componentPath
        ]);

        return Inertia::render($componentPath, [
            'page' => $page,
            'contentBlocks' => $contentBlocks,
            'mediaFiles' => $mediaFiles,
            'seoData' => [
                'canonicalUrl' => $seoData?->canonical_url ?? url('/services/' . $page->slug),
                'ogUrl' => $seoData?->og_url ?? url('/services/' . $page->slug),
                'meta_title' => $seoData?->meta_title ?? $page->title,
                'meta_description' => $seoData?->meta_description ?? '',
                'meta_keywords' => $seoData?->meta_keywords ?? '',
                'robots' => $seoData?->robots ?? 'index, follow',
                'language' => $seoData?->language ?? 'en-US',
                'structured_data' => $structuredData,
            ],
            'openGraph' => $openGraphData,
            'twitterCard' => $twitterData,
            'isCMS' => true,
        ]);
    }

    /**
     * Fallback to static service page
     */
    private function renderStaticServicePage(string $serviceName, string $component)
    {
        // Get service-specific data from static sources
        $serviceData = $this->getServiceData($serviceName);
        
        return Inertia::render($component, [
            'serviceData' => $serviceData,
            'seoData' => [
                'canonicalUrl' => url('/services/' . str_replace('_', '-', $serviceName)),
                'ogUrl' => url('/services/' . str_replace('_', '-', $serviceName)),
            ],
            'isCMS' => false,
        ]);
    }

    /**
     * Get service-specific data for static fallback
     */
    private function getServiceData(string $serviceName)
    {
        $serviceDataMap = [
            'research-papers' => [
                'serviceName' => 'research-papers',
                'optimizedImages' => [
                    'hero' => ImageHelper::getServiceImage(
                        '/images/research-paper-writing-service.jpg',
                        'Professional Research Paper Writing Service - Custom Research Paper Help with Expert Writers'
                    ),
                    'benefits' => ImageHelper::getServiceImage(
                        '/images/research-paper-benefits.jpg',
                        'Professional Research Paper Writing Service Benefits - Custom Research Paper Help with Expert Writers and Original Research'
                    ),
                    'process' => ImageHelper::getServiceImage(
                        '/images/research-paper-process.jpg',
                        'Research Paper Writing Process - Professional Academic Research'
                    ),
                ],
            ],
        ];

        return $serviceDataMap[$serviceName] ?? [];
    }

    public function dissertations()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('dissertation-writing');

        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/Dissertations');
        }

        // Fallback to static data (existing behavior)
        return $this->renderStaticServicePage('dissertation-writing', 'Services/Dissertations');
    }



    public function casestudies()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('case-studies');

        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/CaseStudies');
        }

        // Fallback to static data (existing behavior)
        return $this->renderStaticServicePage('case-studies', 'Services/CaseStudies');
    }

    public function literatureReviews()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('literature-reviews');

        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/LiteratureReviews');
        }

        // Fallback to static data (existing behavior)
        return $this->renderStaticServicePage('literature-reviews', 'Services/LiteratureReviews');
    }

    public function thesisProposals()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('thesis-proposals');

        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/ThesisStatements');
        }

        // Fallback to static data (existing behavior)
        return $this->renderStaticServicePage('thesis-proposals', 'Services/ThesisStatements');
    }



    /**
     * Get all services for the index page with enhanced SEO-focused content
     */
    private function getAllServices()
    {
        return [
            [
                'title' => 'Professional Essay Writing Service',
                'description' => 'Expert essay writing assistance for all academic levels. Custom argumentative, analytical, and narrative essays with PhD writers.',
                'url' => route('services.essay-writing'),
                'icon' => 'FileText',
                'features' => [
                    'Argumentative & Analytical Essays',
                    'All Citation Styles (APA, MLA, Chicago)',
                    'Unlimited Free Revisions',
                    'Plagiarism-Free Guarantee'
                ]
            ],
            [
                'title' => 'Research Paper Writing Help',
                'description' => 'Comprehensive research papers with thorough analysis, credible sources, and professional formatting by subject experts.',
                'url' => route('services.research-papers'),
                'icon' => 'Search',
                'features' => [
                    'Original Research & Analysis',
                    'Comprehensive Literature Review',
                    'Statistical Data Analysis',
                    'Professional Formatting'
                ]
            ],
            [
                'title' => 'Dissertation Writing Service',
                'description' => 'Expert dissertation assistance from proposal to defense. PhD writers specializing in your field of study.',
                'url' => route('services.dissertation-writing'),
                'icon' => 'BookOpen',
                'features' => [
                    'Dissertation Proposals',
                    'Full Dissertation Writing',
                    'Chapter-by-Chapter Assistance',
                    'Defense Preparation Support'
                ]
            ],
            [
                'title' => 'Assignment Help Service',
                'description' => 'Professional assistance with all types of academic assignments. Expert solutions for complex problems and projects.',
                'url' => route('services.assignments'),
                'icon' => 'Edit',
                'features' => [
                    'Homework & Problem Solving',
                    'Project Assistance',
                    'Lab Reports & Case Studies',
                    'Mathematical & Technical Writing'
                ]
            ],
            [
                'title' => 'Case Study Writing Service',
                'description' => 'Detailed case study analysis and writing for business, medical, legal, and academic purposes.',
                'url' => route('services.case-studies'),
                'icon' => 'Users',
                'features' => [
                    'Business Case Analysis',
                    'Medical Case Studies',
                    'Legal Case Research',
                    'Academic Case Writing'
                ]
            ],
            [
                'title' => 'Literature Review Service',
                'description' => 'Comprehensive literature reviews and systematic analysis of academic sources for research projects.',
                'url' => route('services.literature-reviews'),
                'icon' => 'Library',
                'features' => [
                    'Systematic Literature Reviews',
                    'Meta-Analysis Research',
                    'Critical Source Evaluation',
                    'Thematic Analysis'
                ]
            ],
            [
                'title' => 'Thesis Proposal Writing',
                'description' => 'Professional thesis and research proposals with clear methodology and compelling research questions.',
                'url' => route('services.thesis-proposals'),
                'icon' => 'Target',
                'features' => [
                    'Research Proposal Writing',
                    'Thesis Statement Development',
                    'Methodology Design',
                    'Literature Gap Analysis'
                ]
            ],
            [
                'title' => 'Programming Assignment Help',
                'description' => 'Expert programming assistance with Python, Java, C++, and all programming languages. Professional coders with computer science degrees.',
                'url' => route('services.programming-assignments'),
                'icon' => 'Code',
                'features' => [
                    'Python, Java, C++ Programming',
                    'Algorithm Design & Implementation',
                    'Code Documentation & Testing',
                    'Computer Science Expertise'
                ]
            ]
        ];
    }

    /**
     * Add optimized images to services array
     */
    private function addOptimizedImagesToServices(array $services): array
    {
        $serviceImages = [
            'Professional Essay Writing Service' => '/images/essay-writing-service-card.jpg',
            'Research Paper Writing Help' => '/images/research-paper-service-card.jpg',
            'Dissertation Writing Service' => '/images/dissertation-service-card.jpg',
            'Assignment Help Service' => '/images/assignment-help-card.jpg',
            'Case Study Writing Service' => '/images/case-study-service-card.jpg',
            'Literature Review Service' => '/images/literature-review-card.jpg',
            'Thesis Proposal Writing' => '/images/thesis-proposal-card.jpg',
            'Programming Assignment Help' => '/images/programming-assignment-card.jpg',
        ];

        foreach ($services as $index => $service) {
            $imagePath = $serviceImages[$service['title']] ?? '/images/default-service-card.jpg';
            $services[$index]['optimizedImage'] = ImageHelper::getThumbnailImage(
                $imagePath,
                $service['title'] . ' - ' . $service['description']
            );
        }

        return $services;
    }

    public function programmingAssignments()
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent('programming-assignments');
        
        if ($cmsPage) {
            // Use CMS data
            return $this->renderCMSServicePage($cmsPage, 'Services/ProgrammingAssignments');
        } else {
            // Fallback to static data
            return $this->renderStaticServicePage('programming-assignments', 'Services/ProgrammingAssignments');
        }
    }

    // Add other service methods with the same pattern
    public function essays()
    {
        // Get CMS page using only 'essay-writing' slug
        $cmsPage = $this->pageService->getPageWithContent('essay-writing');
        
        if (!$cmsPage) {
            // If no CMS page exists, abort with 404
            abort(404, 'Essay writing service page not found');
        }
        
        // Verify page is published and is a service page
        if ($cmsPage->status !== 'published') {
            abort(404, 'Essay writing service page not found');
        }
        
        if ($cmsPage->page_type !== 'service' && $cmsPage->page_type !== 'service_page') {
            abort(404, 'Page is not a service page');
        }
        
        return $this->renderCMSServicePage($cmsPage, 'Services/DynamicService');
    }

    public function assignments()
    {
        $cmsPage = $this->pageService->getPageWithContent('assignments');
        
        if ($cmsPage) {
            return $this->renderCMSServicePage($cmsPage, 'Services/Assignments');
        } else {
            return $this->renderStaticServicePage('assignments', 'Services/Assignments');
        }
    }

    public function personalStatements()
    {
        $cmsPage = $this->pageService->getPageWithContent('personal-statements');
        
        if ($cmsPage) {
            return $this->renderCMSServicePage($cmsPage, 'Services/PersonalStatements');
        } else {
            return $this->renderStaticServicePage('personal-statements', 'Services/PersonalStatements');
        }
    }

    /**
     * Buy-intent service pages (CMS-first)
     */
    public function affordableEssayWriting()
    {
        $cmsPage = $this->pageService->getPageWithContent('affordable-essay-writing');

        if (!$cmsPage) {
            abort(404, 'Affordable Essay Writing service page not found');
        }

        return $this->renderCMSServicePage($cmsPage, 'Services/AffordableEssayWriting');
    }

    public function buyEssayWriting()
    {
        $cmsPage = $this->pageService->getPageWithContent('buy-essays-online');

        if (!$cmsPage) {
            abort(404, 'Buy Essay Writing service page not found');
        }

        return $this->renderCMSServicePage($cmsPage, 'Services/BuyEssayWriting');
    }

    public function cheapEssayWriting()
    {
        $cmsPage = $this->pageService->getPageWithContent('cheap-essays-online');

        if (!$cmsPage) {
            abort(404, 'Cheap Essay Writing service page not found');
        }

        return $this->renderCMSServicePage($cmsPage, 'Services/CheapEssayWriting');
    }

    /**
     * Handle dynamically created service pages
     * This is a catch-all for service pages created via CMS that don't have hardcoded routes
     */
    public function show(Request $request, string $slug)
    {
        // Try to get service page from CMS
        $cmsPage = $this->pageService->getPageWithContent($slug);
        
        if (!$cmsPage) {
            abort(404, 'Service page not found');
        }

        // Check if it's a service page
        if ($cmsPage->page_type !== 'service' && $cmsPage->page_type !== 'service_page') {
            abort(404, 'Page is not a service page');
        }

        // Check if page is published
        if ($cmsPage->status !== 'published') {
            abort(404, 'Service page not found');
        }

        // Use CMS data to render the page
        return $this->renderCMSServicePage($cmsPage, 'Services/DynamicService');
    }

    /**
     * Process content blocks for frontend rendering
     */
    private function processContentBlocks($contentBlocks)
    {
        $processed = [];
        
        foreach ($contentBlocks as $block) {
            $processed[] = [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content,
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        }
        
        return $processed;
    }

    /**
     * Build structured data defaults by merging stored SEO structured data with safe fallbacks.
     */
    private function buildStructuredData(Page $page, array $seoData, array $contentBlocks): array
    {
        $existing = isset($seoData['structured_data']) && is_array($seoData['structured_data'])
            ? $seoData['structured_data']
            : [];

        $siteName = config('app.name', 'Academic Scribe');
        $siteUrl = url('/');
        $pageUrl = url('/services/' . $page->slug);
        $language = $seoData['language'] ?? 'en-US';

        // FAQ from content blocks (block key suffix '_faqs')
        $faqItems = [];
        foreach ($contentBlocks as $b) {
            if (str_ends_with($b['key'] ?? '', '_faqs')) {
                $faqs = $b['content']['faqs'] ?? [];
                foreach ($faqs as $f) {
                    $q = trim((string)($f['question'] ?? ''));
                    $a = trim((string)($f['answer'] ?? ''));
                    if ($q !== '' && $a !== '') {
                        $faqItems[] = [
                            '@type' => 'Question',
                            'name' => $q,
                            'acceptedAnswer' => [
                                '@type' => 'Answer',
                                'text' => $a,
                            ],
                        ];
                    }
                }
                break;
            }
        }

        $service = $existing['service'] ?? [
            '@context' => 'https://schema.org',
            '@type' => 'Service',
            'name' => $page->title,
            'areaServed' => 'Global',
            'provider' => [
                '@type' => 'Organization',
                'name' => $siteName,
                'url' => $siteUrl,
            ],
            'url' => $pageUrl,
        ];

        // If FAQ items are empty, prepopulate sensible defaults for service pages
        if (empty($faqItems)) {
            $faqItems = [
                [
                    '@type' => 'Question',
                    'name' => 'What does “affordable” mean for this essay writing service?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'Affordable means fair, transparent pricing with quality controls intact: vetted writers, originality checks, and on‑time delivery. We balance cost with realistic timelines and a proven process.'
                    ],
                ],
                [
                    '@type' => 'Question',
                    'name' => 'Will my essay be original and properly cited?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'Yes. All essays are written from scratch and checked with plagiarism detection. We follow your required style (APA/MLA/Chicago/Harvard), and a plagiarism report is available on request.'
                    ],
                ],
                [
                    '@type' => 'Question',
                    'name' => 'Can you deliver urgent or overnight orders?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'Urgent delivery is available when feasible. For best value, choose a 5–10 day deadline. Rush timelines are supported with dedicated writer and QA availability.'
                    ],
                ],
                [
                    '@type' => 'Question',
                    'name' => 'How can I keep costs down without sacrificing quality?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'Provide a clear brief and sources upfront, choose a realistic deadline, and lock scope early. These steps reduce revision cycles and keep your price fair.'
                    ],
                ],
                [
                    '@type' => 'Question',
                    'name' => 'Who will write my essay?',
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => 'We match your order to a qualified writer with subject‑relevant expertise and strong writing samples. Advanced/specialist tiers are available for complex topics.'
                    ],
                ],
            ];
        }

        // Merge with existing FAQ, but fill empty mainEntity
        if (isset($existing['faq']) && is_array($existing['faq'])) {
            $faq = $existing['faq'];
            $hasMain = isset($faq['mainEntity']) && is_array($faq['mainEntity']) && count($faq['mainEntity']) > 0;
            if (!$hasMain) {
                $faq['@context'] = $faq['@context'] ?? 'https://schema.org';
                $faq['@type'] = $faq['@type'] ?? 'FAQPage';
                $faq['mainEntity'] = $faqItems; // use extracted or default items
            }
        } else {
            $faq = [
                '@context' => 'https://schema.org',
                '@type' => 'FAQPage',
                'mainEntity' => $faqItems,
            ];
        }

        $breadcrumb = $existing['breadcrumb'] ?? [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => [
                [
                    '@type' => 'ListItem',
                    'position' => 1,
                    'name' => 'Home',
                    'item' => $siteUrl,
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 2,
                    'name' => 'Services',
                    'item' => url('/services'),
                ],
                [
                    '@type' => 'ListItem',
                    'position' => 3,
                    'name' => $page->title,
                    'item' => $pageUrl,
                ],
            ],
        ];

        $organization = $existing['organization'] ?? [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => $siteName,
            'url' => $siteUrl,
        ];

        $website = $existing['website'] ?? [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            'name' => $siteName,
            'url' => $siteUrl,
            'potentialAction' => [
                '@type' => 'SearchAction',
                'target' => $siteUrl . '?q={search_term_string}',
                'query-input' => 'required name=search_term_string',
            ],
        ];

        $webpage = $existing['webpage'] ?? [
            '@context' => 'https://schema.org',
            '@type' => 'WebPage',
            'name' => $seoData['meta_title'] ?? $page->title,
            'url' => $pageUrl,
            'inLanguage' => $language,
            'description' => $seoData['meta_description'] ?? '',
        ];

        return [
            'service' => $service,
            'faq' => $faq,
            'breadcrumb' => $breadcrumb,
            'organization' => $organization,
            'website' => $website,
            'webpage' => $webpage,
        ];
    }
}
