<?php

namespace App\Http\Controllers\Webhooks;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use App\Services\Payments\PayPalPaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PayPalWebhookController extends Controller
{
    /**
     * Handle the incoming PayPal webhook.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function handleWebhook(Request $request)
    {
        $payload = $request->all();
        $eventType = $payload['event_type'] ?? 'unknown';

        // Log the webhook event
        Log::info('PayPal webhook received', [
            'event_type' => $eventType,
            'resource_type' => $payload['resource_type'] ?? 'unknown'
        ]);

        try {
            // Get the PayPal payment service
            $paypalMethod = PaymentMethod::where('name', 'paypal')->first();

            if (!$paypalMethod) {
                Log::error('PayPal payment method not found for webhook handling');
                return response('PayPal payment method not configured', 500);
            }

            $paymentService = new PayPalPaymentService($paypalMethod);

            // Handle the webhook
            $result = $paymentService->handleWebhook($payload);

            if ($result) {
                return response('Webhook processed successfully', 200);
            } else {
                Log::warning('Failed to process PayPal webhook', [
                    'event_type' => $eventType
                ]);
                return response('Failed to process webhook', 500);
            }
        } catch (\Exception $e) {
            Log::error('Error processing PayPal webhook: ' . $e->getMessage(), [
                'event_type' => $eventType,
                'trace' => $e->getTraceAsString()
            ]);

            return response('Webhook processing error: ' . $e->getMessage(), 500);
        }
    }
}
