<?php

namespace App\Http\Controllers\Webhooks;

use Stripe\Webhook;
use App\Models\Order;
use App\Models\Payment;
use App\Services\WriterNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Stripe\Exception\SignatureVerificationException;

class StripeWebhookController extends Controller
{
    /**
     * Handle Stripe webhook
     */
    public function handleWebhook(Request $request)
    {
        // Get the webhook payload and signature header
        $payload = $request->getContent();
        $sig_header = $request->header('Stripe-Signature');
        $endpoint_secret = config('services.stripe.webhook_secret');

        try {
            // Verify the event using the signature and secret
            $event = Webhook::constructEvent($payload, $sig_header, $endpoint_secret);

            // Process different event types
            switch ($event->type) {
                case 'payment_intent.succeeded':
                    return $this->handlePaymentIntentSucceeded($event->data->object);

                case 'payment_intent.payment_failed':
                    return $this->handlePaymentIntentFailed($event->data->object);

                case 'charge.succeeded':
                    return $this->handleChargeSucceeded($event->data->object);

                case 'charge.failed':
                    return $this->handleChargeFailed($event->data->object);

                case 'checkout.session.completed':
                    return $this->handleCheckoutSessionCompleted($event->data->object);

                default:
                    // Log unhandled event types
                    Log::info('Received unhandled Stripe event type: ' . $event->type);
            }

            return response()->json(['status' => 'success']);
        } catch (SignatureVerificationException $e) {
            // Invalid signature
            Log::error('Stripe webhook signature verification failed: ' . $e->getMessage());
            return response()->json(['error' => 'Invalid signature'], 400);
        } catch (\Exception $e) {
            // Other errors
            Log::error('Stripe webhook error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Handle payment intent succeeded event
     */
    protected function handlePaymentIntentSucceeded($paymentIntent)
    {
        Log::info('Payment Intent Succeeded:', ['id' => $paymentIntent->id]);

        // Get the metadata to find our order
        $orderId = $paymentIntent->metadata->order_id ?? null;

        if (!$orderId) {
            Log::error('No order ID in payment intent metadata', ['payment_intent' => $paymentIntent->id]);
            return response()->json(['error' => 'No order ID found']);
        }

        // Find the order
        $order = Order::find($orderId);

        if (!$order) {
            Log::error('Order not found for payment intent', [
                'order_id' => $orderId,
                'payment_intent' => $paymentIntent->id
            ]);
            return response()->json(['error' => 'Order not found']);
        }

        // Update the order
        $order->payment_status = 'paid';
        $order->order_status = 'bidding'; // Or your appropriate status
        $order->payment_date = now();
        $order->payment_transaction_id = $paymentIntent->id;
        $order->payment_method = 'stripe';
        $order->save();

        // Create a payment record
        Payment::create([
            'order_id' => $order->id,
            'payment_method' => 'stripe',
            'transaction_id' => $paymentIntent->id,
            'amount' => $paymentIntent->amount / 100, // Convert from cents
            'status' => 'completed',
            'raw_response' => json_encode($paymentIntent),
        ]);

        // Notify qualified writers about the new order available for bidding
        try {
            $writerNotificationService = new \App\Services\WriterNotificationService();
            $writerNotificationService->notifyQualifiedWriters($order);
        } catch (\Exception $e) {
            Log::error('Failed to notify writers about new order', [
                'error' => $e->getMessage(),
                'order_id' => $order->id
            ]);
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Handle payment intent failed event
     */
    protected function handlePaymentIntentFailed($paymentIntent)
    {
        Log::info('Payment Intent Failed:', ['id' => $paymentIntent->id]);

        // Get the order ID from metadata
        $orderId = $paymentIntent->metadata->order_id ?? null;

        if (!$orderId) {
            return response()->json(['error' => 'No order ID found']);
        }

        // Find the order
        $order = Order::find($orderId);

        if (!$order) {
            return response()->json(['error' => 'Order not found']);
        }

        // Create a payment record for the failed payment
        Payment::create([
            'order_id' => $order->id,
            'payment_method' => 'stripe',
            'transaction_id' => $paymentIntent->id,
            'amount' => $paymentIntent->amount / 100, // Convert from cents
            'status' => 'failed',
            'raw_response' => json_encode($paymentIntent),
        ]);

        return response()->json(['status' => 'success']);
    }

    /**
     * Handle charge succeeded event
     */
    protected function handleChargeSucceeded($charge)
    {
        Log::info('Charge Succeeded:', ['id' => $charge->id]);
        // Implement if needed - often redundant with payment_intent.succeeded
        return response()->json(['status' => 'success']);
    }

    /**
     * Handle charge failed event
     */
    protected function handleChargeFailed($charge)
    {
        Log::info('Charge Failed:', ['id' => $charge->id]);
        // Implement if needed - often redundant with payment_intent.payment_failed
        return response()->json(['status' => 'success']);
    }

    /**
     * Handle checkout session completed event
     */
    protected function handleCheckoutSessionCompleted($session)
    {
        Log::info('Checkout Session Completed:', ['id' => $session->id]);

        // Get order ID from the session's metadata
        $orderId = $session->metadata->order_id ?? null;

        if (!$orderId) {
            return response()->json(['error' => 'No order ID found']);
        }

        // Find the order
        $order = Order::find($orderId);

        if (!$order) {
            return response()->json(['error' => 'Order not found']);
        }

        // Update the order
        $order->payment_status = 'paid';
        $order->order_status = 'bidding'; // Or your appropriate status
        $order->payment_date = now();
        $order->payment_transaction_id = $session->payment_intent;
        $order->payment_method = 'stripe';
        $order->save();

        // Create a payment record
        Payment::create([
            'order_id' => $order->id,
            'payment_method' => 'stripe',
            'transaction_id' => $session->payment_intent,
            'amount' => $session->amount_total / 100, // Convert from cents
            'status' => 'completed',
            'raw_response' => json_encode($session),
        ]);

        // Notify qualified writers about the new order available for bidding
        try {
            $writerNotificationService = new WriterNotificationService();
            $writerNotificationService->notifyQualifiedWriters($order);
        } catch (\Exception $e) {
            Log::error('Failed to notify writers about new order', [
                'error' => $e->getMessage(),
                'order_id' => $order->id
            ]);
        }

        return response()->json(['status' => 'success']);
    }
}
