<?php

namespace App\Http\Controllers\Writer;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderFile;
use App\Services\OrderStatusService;
use App\Services\AdminNotificationService;
use App\Notifications\WorkSubmittedNotification;
use App\Notifications\WorkSubmittedAdminNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class WorkSubmissionController extends Controller
{
    protected OrderStatusService $orderStatusService;

    public function __construct(OrderStatusService $orderStatusService)
    {
        $this->orderStatusService = $orderStatusService;
        $this->middleware('auth');
        $this->middleware('writer');
    }

    /**
     * Show the work submission form.
     */
    public function showSubmissionForm(Order $order)
    {
        // Verify the writer owns this order
        if ($order->writer_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        // Allow submission for orders that are in progress, assigned, or even expired
        if (!in_array($order->order_status, ['assigned', 'in_progress', 'submitted', 'under_review', 'revision_requested'])) {
            abort(400, 'Order is not in a state that allows work submission');
        }

        // Check if order is expired
        $isExpired = now()->isAfter($order->writer_deadline);
        
        return inertia('Writer/WorkSubmission', [
            'order' => $order->load(['user', 'writer']),
            'isExpired' => $isExpired,
            'deadline' => $order->writer_deadline,
            'currentTime' => now()
        ]);
    }

    /**
     * Submit completed work.
     */
    public function submitWork(Request $request, Order $order)
    {
        // Verify the writer owns this order
        if ($order->writer_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        // Check if order can be submitted based on submission type
        if ($request->submission_type === Order::SUBMISSION_TYPE_COMPLETE && !$order->canSubmitAsComplete()) {
            abort(400, 'Order cannot be submitted as complete in its current status');
        }
        
        if ($request->submission_type === Order::SUBMISSION_TYPE_DRAFT && !$order->canSubmitAsDraft()) {
            abort(400, 'Order cannot be submitted as draft in its current status');
        }

        // For complete submissions, ensure order is in progress (change from assigned if needed)
        if ($request->submission_type === Order::SUBMISSION_TYPE_COMPLETE && $order->order_status === Order::ORDER_STATUS_ASSIGNED) {
            // Automatically change status to in_progress before submission
            $this->orderStatusService->changeStatus(
                $order,
                Order::ORDER_STATUS_IN_PROGRESS,
                'Writer started working and submitted work immediately'
            );
        }

        $request->validate([
            'submission_notes' => 'nullable|string|max:1000',
            'files' => 'required|array|min:1',
            'files.*' => 'required|file|mimes:pdf,doc,docx,txt|max:10240', // 10MB max
            'submission_type' => 'required|in:' . Order::SUBMISSION_TYPE_DRAFT . ',' . Order::SUBMISSION_TYPE_COMPLETE,
        ]);

        // Additional validation to ensure files are properly uploaded
        if (!$request->hasFile('files') || empty($request->file('files'))) {
            abort(400, 'No files were uploaded');
        }

        DB::beginTransaction();

        try {
            // Log submission attempt
            Log::info('Work submission started', [
                'order_id' => $order->id,
                'writer_id' => auth()->id(),
                'submission_type' => $request->submission_type,
                'current_status' => $order->order_status,
                'files_count' => $request->hasFile('files') ? count($request->file('files')) : 0,
                'request_all' => $request->all(),
                'has_files' => $request->hasFile('files'),
                'files_input' => $request->input('files')
            ]);

            // Check if submission is late
            $isLateSubmission = now()->isAfter($order->writer_deadline);
            $submissionTime = now();

            // Handle file uploads
            $uploadedFiles = [];
            if ($request->hasFile('files')) {
                try {
                    foreach ($request->file('files') as $file) {
                        $path = $file->store('order-submissions', 'public');
                        
                        $orderFile = OrderFile::create([
                            'order_id' => $order->id,
                            'file_name' => $file->getClientOriginalName(),
                            'file_url' => $path,
                            'file_type' => $file->getMimeType(),
                            'file_size' => $file->getSize(),
                            'uploader_id' => auth()->id(),
                            'uploader_name' => auth()->user()->name,
                            'uploader_user_type' => auth()->user()->user_type,
                            'file_status' => 'submitted',
                            'file_category' => $request->submission_type === Order::SUBMISSION_TYPE_DRAFT ? 'draft' : 'complete',
                            'date_uploaded' => now()
                        ]);

                        $uploadedFiles[] = $orderFile;
                    }
                } catch (\Exception $fileError) {
                    Log::error('File upload error: ' . $fileError->getMessage(), [
                        'order_id' => $order->id,
                        'writer_id' => auth()->id(),
                        'file_error' => $fileError
                    ]);
                    throw new \Exception('Failed to upload files: ' . $fileError->getMessage());
                }
            }

            // Update order with submission details
            $order->update([
                'work_submitted_at' => $submissionTime,
                'order_revision' => $request->filled('submission_notes') ? $request->submission_notes : null,
                'submission_was_late' => $isLateSubmission,
                'submission_late_by_hours' => $isLateSubmission ? now()->diffInHours($order->writer_deadline) : 0,
                'submission_type' => $request->submission_type
            ]);

            // Handle status transition based on submission type
            if ($request->submission_type === Order::SUBMISSION_TYPE_COMPLETE) {
                // Use OrderStatusService to handle status transition for complete submissions
                $this->orderStatusService->handleWorkSubmitted($order);
            }
            // For draft submissions, no status change occurs

            DB::commit();

            // Send notifications only for complete submissions
            if ($request->submission_type === Order::SUBMISSION_TYPE_COMPLETE) {
                $this->sendSubmissionNotifications($order, $uploadedFiles);
                $this->sendWriterSubmissionNotification($order, $uploadedFiles, $isLateSubmission);
            }

            // Set appropriate message based on submission type
            if ($request->submission_type === Order::SUBMISSION_TYPE_DRAFT) {
                $message = 'Draft saved successfully! Your work has been saved but the order status remains unchanged.';
            } else {
                $message = $isLateSubmission 
                    ? 'Work submitted successfully! Note: This submission was made after the deadline. Admin will review and assess any penalties.'
                    : 'Work submitted successfully! Admin will review your submission.';
            }

            // Log successful submission
            Log::info('Work submission completed successfully', [
                'order_id' => $order->id,
                'writer_id' => auth()->id(),
                'submission_type' => $request->submission_type,
                'final_status' => $order->fresh()->order_status,
                'is_late' => $isLateSubmission
            ]);

            return redirect()->back()->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to submit work: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'writer_id' => auth()->id(),
                'exception' => $e
            ]);

            return redirect()->back()->with('error', 'Failed to submit work: ' . $e->getMessage());
        }
    }

    /**
     * Start working on an order (change status to in_progress).
     */
    public function startWorking(Order $order)
    {
        // Verify the writer owns this order
        if ($order->writer_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        // Verify the order is assigned
        if ($order->order_status !== 'assigned') {
            abort(400, 'Order is not assigned to you');
        }

        try {
            $this->orderStatusService->changeStatus(
                $order,
                'in_progress',
                'Writer started working on the order'
            );

            // Send notifications
            $this->sendStartWorkingNotifications($order);

            return redirect()->back()->with('success', 'Work started successfully!');

        } catch (\Exception $e) {
            Log::error('Failed to start working: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'writer_id' => auth()->id(),
                'exception' => $e
            ]);

            return redirect()->back()->with('error', 'Failed to start working: ' . $e->getMessage());
        }
    }

    /**
     * Send notifications after work submission.
     */
    private function sendSubmissionNotifications(Order $order, array $uploadedFiles)
    {
        try {
            // Notify the client
            $client = $order->user;
            if ($client) {
                $client->notify(new \App\Notifications\WorkSubmittedNotification($order, $uploadedFiles));
            }

            // Notify all admins
            \App\Services\AdminNotificationService::notifyAllAdmins(
                new \App\Notifications\WorkSubmittedAdminNotification($order, $uploadedFiles)
            );

        } catch (\Exception $e) {
            Log::error('Error sending work submission notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }

    /**
     * Send notification to writer about successful submission.
     */
    private function sendWriterSubmissionNotification(Order $order, array $uploadedFiles, bool $isLateSubmission)
    {
        try {
            // Notify the writer
            $writer = $order->writer;
            if ($writer) {
                $writer->notify(new \App\Notifications\WriterWorkSubmittedNotification($order, $uploadedFiles, $isLateSubmission));
            }

        } catch (\Exception $e) {
            Log::error('Error sending writer submission notification: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }

    /**
     * Send notifications when work starts.
     */
    private function sendStartWorkingNotifications(Order $order)
    {
        try {
            // Notify the writer
            $writer = $order->writer;
            if ($writer) {
                $writer->notify(new \App\Notifications\WriterStartedWorkingNotification($order));
            }

            // Notify all admins
            \App\Services\AdminNotificationService::notifyAllAdmins(
                new \App\Notifications\WriterStartedWorkingAdminNotification($order)
            );

        } catch (\Exception $e) {
            Log::error('Error sending start working notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }
}
