<?php

namespace App\Http\Controllers\Writer;

use Carbon\Carbon;
use Inertia\Inertia;
use App\Models\EssayTopic;
use Illuminate\Support\Str;
use App\Models\AdminSetting;
use Illuminate\Http\Request;
use App\Models\WriterProfile;
use App\Models\WriterEssayTest;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Services\AdminNotificationService;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Notifications\NewEssaySubmissionNotification;
use App\Notifications\EssaySubmissionConfirmationNotification;
use App\Notifications\EssayTestExpiredNotification;

class WriterEssayController extends Controller
{
    // Maximum file size in bytes (10MB)
    const MAX_FILE_SIZE = 10 * 1024 * 1024;

    // Allowed file types
    const ALLOWED_FILE_TYPES = ['application/pdf', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];

    // File extensions mapping
    const FILE_EXTENSIONS = [
        'application/pdf' => 'pdf',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx'
    ];

    // Default test duration in minutes (fallback if not found in settings)
    const DEFAULT_TEST_DURATION_MINUTES = 60; // 1 hour


    /**
     * Get the essay test duration from admin settings
     *
     * @return int Test duration in minutes
     */
    private function getTestDuration()
    {
        // Get the test duration from admin settings
        $adminSetting = AdminSetting::first();

        // Return the setting value or default if not found
        return $adminSetting && $adminSetting->essay_test_duration
            ? $adminSetting->essay_test_duration
            : self::DEFAULT_TEST_DURATION_MINUTES;
    }

    /**
     * Show the essay test instructions page.
     *
     * @return \Inertia\Response
     */
    public function instructions()
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        // Get test duration from admin settings
        $testDuration = $this->getTestDuration();

        if (!$writerProfile) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'Writer profile not found. Please complete your profile first.');
        }

        // Get or create essay test record
        $essayTest = WriterEssayTest::firstOrCreate(
            ['writer_profile_id' => $writerProfile->id],
            ['essay_test_status' => 'not_started']
        );

        // Check if there's a pending test that has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);
            // Refresh the test data after handling expiration
            $essayTest->refresh();
        }

        // Get active essay topic
        $essayTopic = EssayTopic::where('is_active', true)->first();

        // Format previous attempt data if available
        $previousAttempt = null;
        if ($essayTest->essay_test_status === 'essay_failed' || $essayTest->essay_test_status === 'essay_expired') {
            $previousAttempt = [
                'date' => $essayTest->essay_completed_at ? $essayTest->essay_completed_at->format('F j, Y') : null,
                'feedback' => $essayTest->essay_feedback,
                'status' => $essayTest->essay_test_status
            ];
        }

        // Calculate waiting period info if applicable
        $waitingPeriod = false;
        $daysRemaining = 0;
        $retryDate = null;

        if ($essayTest->can_retry_after && now()->lt($essayTest->can_retry_after)) {
            $waitingPeriod = true;
            $daysRemaining = now()->diffInDays($essayTest->can_retry_after);
            $retryDate = $essayTest->can_retry_after->format('F j, Y');
        }

        return Inertia::render('Writer/Essay/Instructions', [
            'canTakeEssay' => $essayTest->canTakeEssay(),
            'waitingPeriod' => $waitingPeriod,
            'daysRemaining' => $daysRemaining,
            'retryDate' => $retryDate,
            'essayTopic' => $essayTopic,
            'previousAttempt' => $previousAttempt,
            'testDuration' => $testDuration,
        ]);
    }

    /**
     * Start the essay test.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function start(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return response()->json([
                'error' => 'Writer profile not found. Please complete your profile first.'
            ], 404);
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest) {
            return response()->json([
                'error' => 'Essay test not found. Please try again.'
            ], 404);
        }

        // Check if there's a pending test that has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);
            // Refresh the test data after handling expiration
            $essayTest->refresh();
        }

        if (!$essayTest->canTakeEssay()) {
            return response()->json([
                'error' => 'You cannot take the essay test at this time.'
            ], 403);
        }

        // Get test duration from admin settings
        $testDuration = $this->getTestDuration();

        // Start a new test session with security tracking
        $sessionToken = $essayTest->startTestSession(
            $testDuration,
            $request->ip(),
            $request->userAgent()
        );

        // Store session token in session for verification
        session(['essay_session_token' => $sessionToken]);

        return response()->json([
            'success' => true,
            'redirect' => route('writer.essay.write')
        ]);
    }

    /**
     * Show the essay writing page.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response|\Illuminate\Http\RedirectResponse
     */
    public function write(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'Writer profile not found. Please complete your profile first.');
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_pending') {
            return redirect()->route('writer.essay.instructions')
                ->with('error', 'You need to start the essay test first.');
        }

        // Verify session token
        $sessionToken = session('essay_session_token');
        if (!$sessionToken) {
            return redirect()->route('writer.essay.instructions')
                ->with('error', 'Invalid session. Please start the test again.');
        }

        // Record activity
        $essayTest->recordActivity($sessionToken, $request->ip());

        // Check if test has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);

            return redirect()->route('writer.essay.instructions')
                ->with('error', 'Your test time has expired. Please check if you are eligible to retry.');
        }

        // Get active essay topic
        $essayTopic = EssayTopic::where('is_active', true)->first();

        return Inertia::render('Writer/Essay/Upload', [
            'writerProfile' => $writerProfile,
            'essayTest' => $essayTest,
            'remainingTime' => $essayTest->getRemainingTime(),
            'maxFileSize' => self::MAX_FILE_SIZE / (1024 * 1024), // Convert to MB for display
            'allowedFileTypes' => array_keys(self::FILE_EXTENSIONS),
            'essayTopic' => $essayTopic,
        ]);
    }

    /**
     * Handle file upload for the essay test.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function uploadFile(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return response()->json(['error' => 'Writer profile not found.'], 404);
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_pending') {
            return response()->json(['error' => 'Invalid test status.'], 400);
        }

        // Verify session token
        $sessionToken = session('essay_session_token');
        if (!$sessionToken || $sessionToken !== $essayTest->session_token) {
            return response()->json(['error' => 'Invalid session.'], 403);
        }

        // Record activity
        $essayTest->recordActivity($sessionToken, $request->ip());

        // Check if test has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);
            return response()->json(['error' => 'Test time has expired.', 'redirect' => route('writer.essay.instructions')], 400);
        }

        // Validate file
        $validator = Validator::make($request->all(), [
            'essay_file' => [
                'required',
                'file',
                'max:' . (self::MAX_FILE_SIZE / 1024), // Convert to KB for validation
                function ($attribute, $value, $fail) {
                    if (!in_array($value->getMimeType(), self::ALLOWED_FILE_TYPES)) {
                        $fail('The file must be a PDF or DOCX document.');
                    }
                },
            ]
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }

        $file = $request->file('essay_file');
        $originalFilename = $file->getClientOriginalName();
        $fileType = $file->getMimeType();
        $fileSize = $file->getSize();
        $extension = self::FILE_EXTENSIONS[$fileType];

        // Generate a unique filename
        $filename = 'essay_' . $writerProfile->id . '_' . Str::random(8) . '.' . $extension;

        // Store the file
        $path = $file->storeAs('essays', $filename, 'private');

        if (!$path) {
            return response()->json(['error' => 'Failed to upload file.'], 500);
        }

        // Store file information
        $essayTest->storeFileInformation($path, $originalFilename, $fileType, $fileSize);

        return response()->json([
            'success' => true,
            'message' => 'File uploaded successfully.',
            'filename' => $originalFilename,
            'fileType' => $fileType,
            'fileSize' => $fileSize
        ]);
    }

    /**
     * Submit the essay test.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function submit(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return response()->json(['error' => 'Writer profile not found.'], 404);
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_pending') {
            return response()->json(['error' => 'Invalid test status.'], 400);
        }

        // Verify session token
        $sessionToken = session('essay_session_token');
        if (!$sessionToken) {
            return response()->json(['error' => 'Invalid session.'], 403);
        }

        // Record activity
        $essayTest->recordActivity($sessionToken, $request->ip());

        // Check if test has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);
            return response()->json([
                'success' => false,
                'error' => 'Your test time has expired.',
                'redirect' => route('writer.essay.instructions')
            ], 400);
        }

        // Check if a file has been uploaded
        if (!$essayTest->hasUploadedFile()) {
            return response()->json(['error' => 'Please upload your essay file before submitting.'], 400);
        }

        // Complete the test
        if (!$essayTest->completeTest($sessionToken)) {
            return response()->json([
                'error' => 'Failed to submit test. ' . ($essayTest->validation_message ?? 'Unknown error.')
            ], 400);
        }

        // Get the essay topic for notification
        $essayTopic = EssayTopic::where('is_active', true)->first();
        $essayTopicTitle = $essayTopic ? $essayTopic->title : 'Essay Topic';

        // Notify admins of the new essay submission
        AdminNotificationService::notifyAllAdmins(
            new NewEssaySubmissionNotification(
                $user->name,
                $user->email,
                $essayTopic ? $essayTopic->id : 0,
                $essayTopicTitle,
                now(),
                false, // not auto-submitted
                $essayTest->id,
                $user->id
            )
        );

        // Notify the writer of successful submission
        $user->notify(new EssaySubmissionConfirmationNotification(
            $essayTest,
            $essayTopicTitle,
            false // not auto-submitted
        ));

        // Clear session token
        session()->forget('essay_session_token');

        return response()->json([
            'success' => true,
            'message' => 'Essay submitted successfully.',
            'redirect' => route('writer.essay.confirmation')
        ]);
    }

    /**
     * Auto-submit the essay test when time expires (client-side trigger).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autoSubmit(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return response()->json(['error' => 'Writer profile not found.'], 404);
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_pending') {
            return response()->json(['error' => 'No active test found.'], 400);
        }

        // Verify session token
        $sessionToken = session('essay_session_token');
        if (!$sessionToken || $sessionToken !== $essayTest->session_token) {
            return response()->json(['error' => 'Invalid session.'], 403);
        }

        // Get the essay topic for notification
        $essayTopic = EssayTopic::where('is_active', true)->first();
        $essayTopicTitle = $essayTopic ? $essayTopic->title : 'Essay Topic';

        // Process auto-submission
        if ($essayTest->autoSubmit($sessionToken)) {
            // Clear session token
            session()->forget('essay_session_token');

            if ($essayTest->hasUploadedFile()) {
                // Notify admins of the auto-submitted essay
                AdminNotificationService::notifyAllAdmins(
                    new NewEssaySubmissionNotification(
                        $user->name,
                        $user->email,
                        $essayTopic ? $essayTopic->id : 0,
                        $essayTopicTitle,
                        now(),
                        true, // auto-submitted
                        $essayTest->id,
                        $user->id
                    )
                );

                // Notify the writer that their essay was auto-submitted
                $user->notify(new EssaySubmissionConfirmationNotification(
                    $essayTest,
                    $essayTopicTitle,
                    true // auto-submitted
                ));

                return response()->json([
                    'success' => true,
                    'message' => 'Your essay was automatically submitted as time expired.',
                    'redirect' => route('writer.essay.confirmation')
                ]);
            } else {
                // Notify the writer that their test expired without submission
                $user->notify(new EssayTestExpiredNotification(
                    $essayTest,
                    $essayTopicTitle
                ));

                return response()->json([
                    'success' => true,
                    'message' => 'Your test time expired without a submission.',
                    'redirect' => route('writer.essay.instructions')
                ]);
            }
        }

        return response()->json(['error' => 'Failed to process auto-submission.'], 500);
    }

    /**
     * Show the essay submission confirmation page.
     *
     * @return \Inertia\Response|\Illuminate\Http\RedirectResponse
     */
    public function confirmation()
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'Writer profile not found. Please complete your profile first.');
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_submitted') {
            return redirect()->route('writer.essay.instructions')
                ->with('error', 'You have not submitted an essay test.');
        }

        return Inertia::render('Writer/Essay/Confirmation', [
            'writerProfile' => $writerProfile,
            'essayTest' => [
                'submittedAt' => $essayTest->essay_completed_at->format('F j, Y \a\t g:i a'),
                'fileName' => $essayTest->original_filename,
                'fileType' => $essayTest->file_type,
                'fileSize' => $this->formatFileSize($essayTest->file_size),
                'autoProcessed' => !is_null($essayTest->auto_processed_at),
            ]
        ]);
    }

    /**
     * Check remaining time for the essay test.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkTime(Request $request)
    {
        $user = Auth::user();
        $writerProfile = WriterProfile::where('user_id', $user->id)->first();

        if (!$writerProfile) {
            return response()->json(['error' => 'Writer profile not found.'], 404);
        }

        $essayTest = WriterEssayTest::where('writer_profile_id', $writerProfile->id)->first();

        if (!$essayTest || $essayTest->essay_test_status !== 'essay_pending') {
            return response()->json(['error' => 'No active test found.'], 400);
        }

        // Verify session token
        $sessionToken = session('essay_session_token');
        if (!$sessionToken || $sessionToken !== $essayTest->session_token) {
            return response()->json(['error' => 'Invalid session.'], 403);
        }

        // Record activity
        $essayTest->recordActivity($sessionToken, $request->ip());

        // Check if test has expired
        if ($essayTest->hasExpiredWithoutSubmission()) {
            $this->handleExpiredTest($essayTest);
            return response()->json([
                'isExpired' => true,
                'message' => 'Your test time has expired.',
                'redirect' => route('writer.essay.instructions')
            ]);
        }

        $remainingTime = $essayTest->getRemainingTime();
        $isExpired = $remainingTime <= 0;

        return response()->json([
            'remainingTime' => $remainingTime,
            'isExpired' => $isExpired,
            'formattedTime' => $this->formatTime($remainingTime)
        ]);
    }

    /**
     * Handle an expired test
     *
     * @param WriterEssayTest $essayTest
     * @return void
     */
    private function handleExpiredTest(WriterEssayTest $essayTest)
    {
        // Get the writer user for notifications
        $user = User::find($essayTest->writerProfile->user_id);

        // Get essay topic for notifications
        $essayTopic = EssayTopic::where('is_active', true)->first();
        $essayTopicTitle = $essayTopic ? $essayTopic->title : 'Essay Topic';

        if ($essayTest->hasUploadedFile()) {
            $essayTest->processPartialSubmission();

            // If we have a user and an essay was uploaded, notify about partial submission
            if ($user) {
                // Notify admins about the partially submitted essay
                AdminNotificationService::notifyAllAdmins(
                    new NewEssaySubmissionNotification(
                        $user->name,
                        $user->email,
                        $essayTopic ? $essayTopic->id : 0,
                        $essayTopicTitle,
                        now(),
                        true, // auto-submitted due to expiration
                        $essayTest->id,
                        $user->id
                    )
                );

                // Notify the writer about partial submission
                $user->notify(new EssaySubmissionConfirmationNotification(
                    $essayTest,
                    $essayTopicTitle,
                    true // auto-submitted
                ));
            }
        } else {
            $essayTest->markAsExpired('Detected during controller action - no file uploaded');

            // If we have a user, notify about the expired test with no submission
            if ($user) {
                $user->notify(new EssayTestExpiredNotification(
                    $essayTest,
                    $essayTopicTitle
                ));
            }
        }
    }

    /**
     * Format time in seconds to a readable format.
     *
     * @param  int  $seconds
     * @return string
     */
    private function formatTime($seconds)
    {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;

        return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
    }

    /**
     * Format file size to human-readable format.
     *
     * @param  int  $bytes
     * @return string
     */
    private function formatFileSize($bytes)
    {
        if ($bytes < 1024) {
            return $bytes . ' bytes';
        } elseif ($bytes < 1048576) {
            return round($bytes / 1024, 1) . ' KB';
        } else {
            return round($bytes / 1048576, 1) . ' MB';
        }
    }
}
